/**
 * receive_message_queue.c
 *
 * The BMC receive message queue. Stores messages to the
 * system interface and creates the necessary status flag.
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */


#include "receive_message_queue.h"
#include <pp/dlist.h>
#include <pp/bmc/bmc_imsg.h>
#include "nc_interrupt.h"


#define MAX_RECV_MSG_QUEUE 20
static dlist_t* recv_msg_queue = NULL;
int recv_msg_queue_size = 0;


int bmc_receive_queue_add(imsg_t* imsg) {
    assert(dlist_count(recv_msg_queue) == recv_msg_queue_size);
    
    if (recv_msg_queue_size >= MAX_RECV_MSG_QUEUE) {
        /* queue is full */
        return PP_ERR;
    }

    if (recv_msg_queue_size == 0) {
        /* we now have messages in the queue, set the interrupt */
        bmc_interrupt_flag_set(BMC_INTERRUPT_RECEIVE_MSG_QUEUE, 1);
    }
        
    dlist_push_back(recv_msg_queue, imsg);
    recv_msg_queue_size++;
    
    return PP_SUC;
}

imsg_t* bmc_receive_queue_get() {
    assert(dlist_count(recv_msg_queue) == recv_msg_queue_size);

    if (recv_msg_queue_size == 0) {
        return NULL;
    }

    if (recv_msg_queue_size == 1) {
        /* we remove the last msg from the queue, clear interrupt */
        bmc_interrupt_flag_set(BMC_INTERRUPT_RECEIVE_MSG_QUEUE, 0);
    }
    
    recv_msg_queue_size--;
    return dlist_pop_front(recv_msg_queue);
}

void bmc_receive_queue_clear() {
    dlist_clear(recv_msg_queue);
    recv_msg_queue_size = 0;
    bmc_interrupt_flag_set(BMC_INTERRUPT_RECEIVE_MSG_QUEUE, 0);
}



int bmc_receive_message_queue_init() {
    recv_msg_queue = dlist_new((void*)pp_bmc_imsg_delete);
    recv_msg_queue_size = 0;
    
    return PP_SUC;
}

void bmc_receive_message_queue_cleanup() {
    dlist_delete(recv_msg_queue);
    recv_msg_queue = NULL;
    recv_msg_queue_size = 0;
}

