/**
 * host_graceful_ops.h
 *
 * Implements Supermicro's graceful power OEM commands.
 * (done by help of OS agent)
 *
 * (c) 2006 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <pp/bmc/host_power.h>
#include <pp/bmc/host_graceful_ops.h>
#include <pp/bmc/bmc_event.h>
#include <pp/bmc/debug.h>

#include <pp/base.h>
#include <pp/selector.h>

/*
 * Global values
 */

// initial values set during ...init() to allow re-initialization
static int pending_op;
static int op_chan; // the channel, where the cmd came in
static int timer_id;
static int execute_id;

/*
 * Internal functions
 */

static int timer(int id UNUSED, void* ctx UNUSED)
{
    timer_id = -1;
    pp_bmc_log_warn("[HW] graceful power operation %d timed out", pending_op);
    pending_op = -1;
    return PP_SUC;
}

static int execute(void* ctx UNUSED)
{
    execute_id = -1;
    if (pending_op != -1) {
	pp_bmc_log_info("[HW] execute graceful power action %d", pending_op);
	if (PP_FAILED(pp_bmc_host_power_control(pending_op, HOST_POWER_CONTROL_SOURCE_OEM, op_chan))) {
	    pp_bmc_log_error("[HW] unable to execute power control (%d)", pending_op);
	}
	pending_op = -1;
    }
    return PP_SUC;
}

static void power_control_hndl(int control_action)
{
    if (pending_op != -1 && control_action == HOST_POWER_CONTROL_RESET) {
	pp_select_remove_to(timer_id);
	timer_id = -1;

	// don't call power control during power control callback
	execute_id = pp_select_add_sf(execute, NULL);
    }
}

/*
 * Public functions
 */

int pp_bmc_host_graceful_ops_init()
{
    pending_op = -1; // no op pending
    timer_id = -1;
    execute_id = -1;

    // register hander to get reset events
    pp_bmc_host_power_control_subscribe(power_control_hndl);

    return PP_SUC;
}

void pp_bmc_host_graceful_ops_cleanup()
{
    if (execute_id != -1) pp_select_remove_sf(execute_id); 
    if (timer_id != -1) pp_select_remove_to(timer_id);
    pp_bmc_host_power_control_unsubscribe(power_control_hndl);
}

int pp_bmc_host_graceful_ops_execute(int op, int chan)
{
    if (pending_op != -1) {
	pp_select_remove_to(timer_id);
    }

    pending_op = op;
    op_chan = chan;

    pp_bmc_receive_event(0x0, 0x20, 0x0, 0x1,0xc9,0xff,0x0, 0x6f,0x68, 0x8f,0xff);

    timer_id = pp_select_add_to(5 * 60 * 1000, 0, timer, NULL);
    if (timer_id == -1) {
	pp_bmc_log_error("[HW] unable to create timer for GRACEFUL POWER OPERATION");
	goto fail;
    }

    return PP_SUC;

fail:
    pending_op = -1;
    return PP_ERR;
}
