/**
 * host_hardware.h
 * 
 * Defines abstract host hardware that can be mapped to different hosts.
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __HOST_HARDWARE_H__
#define __HOST_HARDWARE_H__

#include <pp/base.h>

#if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
#include <pp/bmc/smi_adapter.h>
#endif
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/host_sensors.h>

/**
 * Host frontpanel button control.
 */

/* sensors that can be disabled */
#define HOST_BUTTON_POWER      PP_SENSOR_POWER_BTN
#define HOST_BUTTON_RESET      PP_SENSOR_RESET_BTN
#define HOST_BUTTON_STANDBY    PP_SENSOR_SLEEP_BTN
#define HOST_BUTTON_DIAG_INT   PP_SENSOR_DIAG_BTN
#define HOST_BUTTON_ID         PP_SENSOR_ID_BTN
/* sensors that can not be disabled */
#define HOST_BUTTON_NMI        PP_SENSOR_NMI_BTN

#define HOST_BUTTON_ENABLE     PP_SUC
#define HOST_BUTTON_DISABLE    PP_ERR


/**
 * Initialize the host hardware library and its modules.
 */
int pp_bmc_hardware_init(void);

/**
 * Cleanup the host hardware library and its modules.
 */
void pp_bmc_hardware_cleanup(void);


/**
 * Enable/disable the selected button, software based.
 */
void pp_bmc_hardware_button_enable_set(int button, int enable);

/**
 * Check if the selected button is enabled/disabled.
 * Software based.
 */
int pp_bmc_hardware_button_enable_get(int button);

/**
 * Returns PP_SUC if frontpanel lockout is active.
 * (power or reset button locked, software based)
 */
int pp_bmc_hardware_frontpanel_lockout_get(void);

/**
 * Returns whether the specified button can be disabled or not.
 * (Software based)
 * @returns PP_SUC if the specified button can be disabled
 * @returns PP_ERR if the specified button cannot be disabled
 */
int pp_bmc_hardware_button_enable_check(int button);


/**
 * Set a physical frontpanel lockout signal that locks all
 * frontpanel buttons, local keyboard and local mouse.
 * (currently used on the OPMA board)
 * 
 * @param   lockout 1 to lock, 0 to unlock
 * @returns PP_SUC  lock/unlock executed successfully
 *          PP_ERR  not able to lock/unlock, possibly
 *                  not supported on this plattform
 */
int pp_bmc_hardware_frontpanel_hard_lockout_set(int lockout);

/**
 * Get the status of the opma frontpanel lockout.
 * @returns 0  if frontpanel is unlocked
 *          1  if frontpanel is locked
 *         -1  if frontpanel lock is not supported
 */
int pp_bmc_hardware_frontpanel_hard_lockout_get(void);


/**
 * Returns PP_SUC if a diagnostic interrupt exists on the host.
 * @returns PP_SUC/PP_ERR
 */
int pp_bmc_hardware_diag_int_exists(void);

/**
 * Returns PP_SUC if the chassis has an intrusion detection sensor.
 * @returns PP_SUC/PP_ERR
 */
int pp_bmc_hardware_chassis_intrusion_exists(void);

/**
 * Return the chassis intrusion status.
 * @returns PP_SUC if the chassis is closed
 * @returns PP_ERR else
 */
int pp_bmc_hardware_chassis_closed(void);

/**
 * Returns PP_SUC if chassis intrusion flag has to be cleared manually.
 * @returns PP_SUC/PP_ERR
 */
int pp_bmc_hardware_clr_intrusion_exists(void);

/**
 * Clear the chassis intrusion flag.
 * @returns PP_SUC if the chassis intrusion flag has been cleared
 * @returns PP_ERR else
 */
int pp_bmc_hardware_clr_intrusion_flag(void);

/**
 * Query the drive health status.
 * @returns PP_SUC if the drives are healthy
 * @returns PP_ERR if there are faults in the drives
 */
int pp_bmc_hardware_drive_health(void);

/**
 * Query the fan health status
 * @returns PP_SUC if the fans work correct
 * @returns PP_ERR if the fan system has faults
 */
int pp_bmc_hardware_fan_health(void);

/**
 * @returns PP_SUC if a chassis identify device exists
 * @returns PP_ERR else
 */
int pp_bmc_hardware_chassis_identify_device_exists(void);

/**
 * Turn on the chassis identify device. The device will stop after
 * the provided time interval (in secs) has passed.
 * @param interval The work interval time in seconds. Set to 0 to turn the
 *     device off. Set to -1 to turn on idefinitely.
 */
void pp_bmc_hardware_chassis_identify_set(int interval);

/**
 * Clear the system bios cmos
 * @returns PP_SUC if operation completed successful
 * @returns PP_ERR if its not possible to clear the cmos
 */
int pp_bmc_hardware_clear_cmos(void);

/**
 * This function is called whenever a new sensor reading is received.
 * ctx contains the index of the sensor
 */
void pp_bmc_hardware_receive_reading(int reading, void* ctx);

/**
 * If necessary for a board, communicate the system interface we
 * use.
 *
 * @param mode system interface mode
 */
#if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
int pp_bmc_hardware_setup_interface_id(pp_bmc_smi_mode_t mode);
#endif


#endif
