/**
 * serial_lowlevel.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * A serial port lowlevel abstraction that can easily be mapped
 * to different motherboard/serial port sharing solutions.
 */

#include <stdio.h>
#include <sys/poll.h>
#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/selector.h>
#include <pp/termios.h>

#include <pp_gpio.h>
#include <lara.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/tp_smx_act.h>
#include <pp/bmc/host_sensors.h>

#include <pp/bmc/serial_lowlevel.h>

#define MAX_IPMI_MSG_LEN 172     // same value as in openIPMI

typedef struct {
    pp_bmc_serial_mux_state_t smux_state;
    int smux_switch_in_progress;
    pp_bmc_serial_mux_state_t smux_newstate; // only valid if smux_switch_in_progress == 1
    
    pp_bmc_serial_mux_activate_hndl   activate_state[3];
    pp_bmc_serial_mux_deactivate_hndl deactivate_state[3];
} sll_globals_t;

static sll_globals_t my_sllg;
static sll_globals_t* sllg = NULL;


static inline pp_tp_smx_act_t* ser_get_smux(void) {
    return (pp_tp_smx_act_t*)pp_bmc_host_sensor_get_actor(PP_ACTOR_SMUX);
}

pp_bmc_serial_mux_state_t pp_bmc_serial_mux_get_state(void) {
    return sllg->smux_state;
}

static int pp_bmc_serial_mux_switch_force(pp_bmc_serial_mux_state_t newstate) {
    pp_tp_smx_act_t* smx;
    int serial_debug = 0;
    int i;

    /* dont use mux if serial debug is active */
    ioctl(eric_fd, ERICIOCCHECKSERIALDEBUG, &serial_debug);
    if ( serial_debug == 1 ) {
	pp_bmc_log_error("[SER] serial debug is active. S-Mux ignored");
        return PP_ERR;
    }
    
    /* do we have a mux actor */
    if (NULL == (smx = ser_get_smux())) {
        pp_bmc_log_error("[SER] no S-Mux actor registered");
        return PP_ERR;
    }    
    
    if (sllg->smux_switch_in_progress == 1) {
        // switch in progress - deny mux switch request
        pp_bmc_log_debug("[SER] mux switch in progress - denied mux switch request to ", newstate);
        return PP_ERR;
    } else {
        // initiate switch - set newstate, switch_in_progress and start deactivation
        pp_bmc_log_debug("[SER] initiating mux switch from %d to %d", sllg->smux_state, newstate);
        i = (int)sllg->smux_state;
        
        if (sllg->deactivate_state[i] != NULL) {
            // deactivate this handler
            sllg->smux_switch_in_progress = 1;
            sllg->smux_newstate = newstate;
            sllg->deactivate_state[i](newstate);
        } else {
            // cannot deactivate old handler, no handler registered
            // skip deactivation and switch immediately
            sllg->smux_switch_in_progress = 1;
            sllg->smux_newstate = newstate;
            pp_bmc_serial_mux_deactivate_finished();
        }
    }
    
    return PP_SUC;
}

int pp_bmc_serial_mux_switch(pp_bmc_serial_mux_state_t newstate) {
    if (sllg->smux_state != newstate) {
        pp_bmc_log_debug("[SER] initiated mux switch to %d", newstate);
	return pp_bmc_serial_mux_switch_force(newstate);
    } else {
        pp_bmc_log_debug("[SER] ignored serial mux switch request - already switched to state %d", newstate);
    }
    
    return PP_SUC;
}

void pp_bmc_serial_mux_deactivate_finished() {
    pp_tp_smx_act_t* smx;
    pp_bmc_serial_mux_state_t old_state;
    int i;
    smx = ser_get_smux();  // cannot fail, already checked
    
    // the old handler has been deactivated, switch and activate new handler
    assert(sllg->smux_switch_in_progress == 1);
    sllg->smux_switch_in_progress = 0;
    
    switch (sllg->smux_newstate) {
    case PP_SMUX_HOST_CHASSIS:
        pp_tp_smx_switch_host_chassis(smx);
        break;
    case PP_SMUX_BMC_CHASSIS:
        pp_tp_smx_switch_bmc_chassis(smx);
        break;
    case PP_SMUX_HOST_BMC:
        pp_tp_smx_switch_host_bmc(smx);
        break;
   default:
        assert(0);
   }
   old_state = sllg->smux_state;
   sllg->smux_state = sllg->smux_newstate;
   
   i = (int)(sllg->smux_state);
   if (sllg->activate_state[i] != NULL) {
       sllg->activate_state[i](old_state);
   }
   
   pp_bmc_log_debug("[SER] switched to new smux state %d", sllg->smux_state);
}

void pp_bmc_serial_mux_register_handler(pp_bmc_serial_mux_state_t state,
                                        pp_bmc_serial_mux_activate_hndl activate_hndl,
                                        pp_bmc_serial_mux_deactivate_hndl deactivate_hndl)
{
    int i;
    
    i = (int)state;
    
    assert( ( ((activate_hndl != NULL) && (deactivate_hndl != NULL)) ||
              ((activate_hndl == NULL) && (deactivate_hndl == NULL)) ) );
    
    if (sllg->smux_state == state) {
        /* the handler for the active state is changed, make sure that the old
         * handler gets deactivated and that the new handler gets activated */

        /* set new activation handler  before (possible) reactivation */
        sllg->activate_state[i] = activate_hndl;
        
        /* reactivate this state with the new handler */
        pp_bmc_serial_mux_switch_force(sllg->smux_state);
        
        /* set deactivation handler after (possible) reactivation */
        sllg->deactivate_state[i] = deactivate_hndl;
    } else {
        /* state not activ - just set */
        sllg->activate_state[i] = activate_hndl;
        sllg->deactivate_state[i] = deactivate_hndl;
    }
}

inline void pp_bmc_serial_mux_clear_handler(pp_bmc_serial_mux_state_t state) {
    pp_bmc_serial_mux_register_handler(state, NULL, NULL);
}

int pp_bmc_serial_lowlevel_init() {
    int i;
    
    assert(sllg == NULL);
    sllg = &my_sllg;

    for (i=0; i<3; i++) {
        sllg->activate_state[i] = NULL;
        sllg->deactivate_state[i] = NULL;
    }
    
    /* Note: currently we always start without switch mux because libpp_hardware is
     * initialized before libpp_sensor. This does not seems to make a difference yet. */
    if (ser_get_smux() != NULL) {
        sllg->smux_newstate = PP_SMUX_HOST_CHASSIS;
        // set mux correctly, initialize sllg->mux_state with this command:
        sllg->smux_switch_in_progress = 1;
        pp_bmc_serial_mux_deactivate_finished();
    } else {
        //pp_bmc_log_error("[SER] Serial lowlevel handler started without mux switch");
        // no mux installed, we just assume that we are in this state:
        sllg->smux_state = PP_SMUX_HOST_CHASSIS;
        sllg->smux_switch_in_progress = 0;
    }
    
    return PP_SUC;
}

void pp_bmc_serial_lowlevel_cleanup() {
    sllg = NULL;
}
