/**
 * ipmb_adapter.c
 *
 * (c) 2006 Peppercon AG, 2006/04/05, thomas@peppecon.de
 */

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/i2c.h>
#include <pp_i2c.h>

#include <pp/base.h>
#include <pp/i2c.h>
#include <pp/selector.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/session_manager.h>

#include <pp/bmc/host_sensors.h>
#include <pp/bmc/tp_i2c_comdev.h>

#include <pp/bmc/ipmb_adapter.h>

#define noIPMB_MSG_DBG
#define MAX_IPMI_MSG_LEN 172     // same value as in openIPMI
#define IPMB_RSP_TIMEOUT 500     // ipmb response timeout in ms

typedef struct {
    pp_tp_i2c_comdev_t* i2cdev;
    int sf_id;
    imsg_t *imsg;
} ipmb_state_t;

typedef struct {
    unsigned char channel_no;
    int comdev_id;
    int (*msg_hndlr)(imsg_t*);
    ipmb_state_t* state;
} ipmb_chan_t;

/*
 * private function prototypes
 * ----------------------------
 */
static ipmb_chan_t* ipmb_find_chan(unsigned char channel_no);
static int ipmb0_receive_bmc_msg(imsg_t *imsg);
static int ipmb1_receive_bmc_msg(imsg_t *imsg);
static int ipmb_receive_bmc_msg_muxed(ipmb_chan_t* chan, imsg_t *imsg);
static int ipmb_receive_ipmb_msg_muxed(void* ctx);
static int ipmb_msg_build(ipmb_chan_t* ipmb, imsg_t *imsg,
				   unsigned char msgbuf[]);
static int ipmb_msg_send(ipmb_chan_t* ipmb, ipmi_msg_hdr_t* msg, int msize);
static imsg_t* ipmb_msg_recv(ipmb_chan_t* ipmb, unsigned char priv_level);

/*
static int ipmb_send_to(const int item_id, void* ctx);
static int ipmb_receive_ipmb_msg(const int item_id, const int fd,
				 const short event, void *ctx);
*/


ipmb_chan_t chan[] = {
    {	.channel_no = IPMI_CHAN_PRIMARY_IPMB,
	.comdev_id = PP_BMC_I2C_COMDEV_IPMB0,
	.msg_hndlr = ipmb0_receive_bmc_msg,
	.state = NULL    },
    {	.channel_no = IPMI_CHAN_SECDARY_IPMB,
	.comdev_id = PP_BMC_I2C_COMDEV_IPMB1,
	.msg_hndlr = ipmb1_receive_bmc_msg,
	.state = NULL    },
};

static const pp_bmc_router_chan_adapter_info_t ipmb_chan_info = {
    .medium =       IPMI_CHAN_MED_IPMB,
    .proto =        IPMI_CHAN_PROT_IPMB,
    .sess_supp =    IPMI_CHAN_SESS_LESS,
};

/*
 * public implementation
 * ----------------------
 */

/*
 * Note: BE ALERTED!!
 *
 * IPMB channel implementation is not complete!
 *
 * Currently the IPMB channel works strictly request-response based.
 * Asynchronous messages from IPMB are _NOT_ supported.
 * 
 * This limitation is introduces by the i2c-slave-mode-driver-implementation,
 * which is capable receiving from a single slave address only, which works
 * if you had a single satelite controller only, but only then!
 *
 * I will need to change this, however for ICP project, current
 * impl should do!
 */
int pp_bmc_ipmb_init(unsigned char channel_no) {
    ipmb_chan_t* ipmb;

    if (NULL == (ipmb = ipmb_find_chan(channel_no))) {
	assert(0);
	return PP_ERR;
    }

    assert(ipmb->state == NULL);
    ipmb->state = malloc(sizeof(ipmb_state_t));
    ipmb->state->imsg = NULL;
    
    /* get i2c com device from topology */
    ipmb->state->i2cdev = pp_bmc_host_get_i2c_comdev(ipmb->comdev_id);
    if (ipmb->state->i2cdev == NULL) {
	pp_bmc_log_error("[IPMB] channel (%d) failed: no i2c device!",
			 ipmb->channel_no);
	errno = ENODEV;
	return PP_ERR;
    }
    if (pp_tp_i2c_comdev_root_handle(ipmb->state->i2cdev) < 0) {
	pp_bmc_log_error("[IPMB] channel (%d) failed, wrong i2cdev (%s)!",
			 ipmb->channel_no, ipmb->state->i2cdev->base.id);
	errno = ENODEV;
	return PP_ERR;
    }	

    pp_tp_i2c_comdev_duplicate(ipmb->state->i2cdev);

    /* register with bmc, for messages from BMC */
    if (pp_bmc_router_reg_chan(ipmb->channel_no, ipmb->msg_hndlr,
			       NULL, NULL, &ipmb_chan_info) == NULL) {
        pp_bmc_log_fatal("[IPMB] cannot register channel (%d)",
			 ipmb->channel_no);
        return PP_ERR;
    }

    pp_bmc_log_info("[IPMB] channel (%d) intialized", ipmb->channel_no);
    return PP_SUC;

    
}        

void pp_bmc_ipmb_cleanup(unsigned char channel_no)  {
    ipmb_chan_t* ipmb;

    if (NULL == (ipmb = ipmb_find_chan(channel_no))) {
	assert(0);
	return;
    }
    assert(ipmb->state != NULL);
    pp_bmc_router_unreg_chan(ipmb->channel_no);
    pp_tp_i2c_comdev_release(ipmb->state->i2cdev);
    /* no need to kill timeout and stuff,    *
     * selector shouldn't be running anymore */
    free(ipmb->state);
    ipmb->state = NULL;
    pp_bmc_log_info("[IPMB] channel (%d) shut down", ipmb->channel_no);
}

/*
 * private implementation
 * ----------------------
 */

static int ipmb0_receive_bmc_msg(imsg_t *imsg) {
    assert(chan[0].state != NULL);
    return ipmb_receive_bmc_msg_muxed(&chan[0], imsg);
}

static int ipmb1_receive_bmc_msg(imsg_t *imsg) {
    assert(chan[1].state != NULL);
    return ipmb_receive_bmc_msg_muxed(&chan[1], imsg);
}

static int ipmb_receive_bmc_msg_muxed(ipmb_chan_t* ipmb, imsg_t *simsg) {
    unsigned char msgbuf[MAX_IPMI_MSG_LEN];
    ipmi_msg_hdr_t* msghdr = (ipmi_msg_hdr_t*)&msgbuf[0];
    pp_tp_i2c_comdev_t* i2cdev;
    unsigned char i2c_slv_addr;
    int i2cfd, msize, r, ret = PP_ERR;
    
    struct pollfd fds[1];
    imsg_t* rimsg;

    i2cdev = ipmb->state->i2cdev;
    i2cfd = pp_i2c_get_fd(pp_tp_i2c_comdev_root_handle(i2cdev));
    assert(i2cfd >= 0);

    /* check whether we are already sending a messages, chance is small */
    if(ipmb->state->imsg != NULL) {
	pp_bmc_log_perror("[IPMB] %s: bussy with previous request",
			  i2cdev->base.id);
	errno = EAGAIN;
	goto bailout;
    }
    
    if (PP_ERR == (msize = ipmb_msg_build(ipmb, simsg, msgbuf))) {
	goto bailout;
    }
    
    /* lock the device and set mux */
    i2c_slv_addr = msghdr->rs_addr >> 1;
    if (PP_SUC != pp_tp_i2c_comdev_pre_com(i2cdev, i2c_slv_addr)) {
	pp_bmc_log_perror("[IPMB] %s: pre_com(%#x) failed",
			  i2cdev->base.id, i2c_slv_addr);
	goto bailout;
    }

    if (PP_ERR == ipmb_msg_send(ipmb, msghdr, msize)) {
	goto bailout_unlock;
    }
    
    /* schedule receive */
    while(1) {
	fds[0].fd = i2cfd;
	fds[0].events = POLLIN;
	fds[0].revents = 0;
	r = poll(fds, sizeof(fds)/sizeof(struct pollfd), 100 /* ms */);
	if (r < 0 && errno == EINTR) continue;
	break;
    }
    if (r < 0) {         /* obscure error */
	pp_bmc_log_perror("[IPMB] %s: poll failed", i2cdev->base.id);
	goto bailout_unlock;
    } else if (r == 0) { /* timeout */
	pp_bmc_log_error("[IPMB] %s: poll timed out", i2cdev->base.id);
	goto bailout_unlock;
    } else if ((fds[0].revents & (POLLERR | POLLHUP | POLLNVAL)) != 0) {
	pp_bmc_log_error("[IPMB] %s: poll fd (%d) had error (%#x)",
			 i2cdev->base.id, fds[0].fd, fds[0].revents);
	goto bailout_unlock;
    }

    /* read message */
    if (NULL == (rimsg = ipmb_msg_recv(ipmb, simsg->priv_level))) {
	goto bailout_unlock;
    }

    /* schedule a function that injects received message into BMC, so that *
     * it feels like a message that really got asynchronously received.    *
     * otherwise msg tracker will not work transparently.                  */
    ipmb->state->imsg = rimsg;
    ipmb->state->sf_id = pp_select_add_sf(ipmb_receive_ipmb_msg_muxed, ipmb);
    ret = PP_SUC;

 bailout_unlock:
    /* unlock i2c device  */
    pp_tp_i2c_comdev_post_com(i2cdev, i2c_slv_addr);
 bailout:
    /* delete, as we don't pass it along */
    pp_bmc_imsg_delete(simsg);
    return ret;
}

/* scheduled function */
static int ipmb_receive_ipmb_msg_muxed(void* ctx) {
    ipmb_chan_t* ipmb = (ipmb_chan_t*)ctx;
    pp_bmc_core_handle_msg(ipmb->state->imsg);
    ipmb->state->imsg = NULL;
    return PP_SUC;
}

static int ipmb_msg_build(ipmb_chan_t* ipmb UNUSED, imsg_t *imsg,
			  unsigned char msgbuf[]) {
    ipmi_msg_hdr_t* msghdr = (ipmi_msg_hdr_t*)&msgbuf[0];
    int msize, dsize;

    /* Note: this might be a hack, as we set the rqSA address to what we
     * know we are on that bus. the client may have put it differently
     * but it shouldn't bother him either
     */
    imsg->rq_addr = I2C_OWN_ADDRESS << 1;
    imsg->rq_lun  = 0x0;
    
    /* build msg */
    dsize = imsg->data_size;
    msize = dsize + sizeof(ipmi_msg_hdr_t) + 1/*chk2*/;
    if (msize > MAX_IPMI_MSG_LEN) {
	errno = EINVAL;
	pp_bmc_log_error("[IPMB] msg (rs=%#.2hhx, rq=%#.2hhx) too long!",
			 imsg->rs_addr >> 1, I2C_OWN_ADDRESS);
	return PP_ERR;
    }
    msghdr->rs_addr = imsg->rs_addr;
    msghdr->rs_lun  = imsg->rs_lun;
    msghdr->netfn   = imsg->netfn;
    msghdr->chk1    = pp_bmc_calc_ipmi_checksum(msgbuf, 0, 1);
    msghdr->rq_addr = imsg->rq_addr;
    msghdr->rq_lun  = imsg->rq_lun;
    msghdr->rq_seq  = imsg->rq_seq;
    msghdr->cmd     = imsg->cmd;
    memcpy(msgbuf + 6, imsg->data, imsg->data_size);
    msgbuf[msize - 1] =	pp_bmc_calc_ipmi_checksum(msgbuf, 3, msize - 2);
    return msize;
}

static int ipmb_msg_send(ipmb_chan_t* ipmb, ipmi_msg_hdr_t* msg, int msize) {
    pp_tp_i2c_comdev_t* i2cdev;
    int i2cfd;
    int i2chandle;
    unsigned char i2c_slv_addr = msg->rs_addr >> 1;

    /* get i2c base-lib handle and file descriptor */
    i2cdev = ipmb->state->i2cdev;
    if (0 > (i2chandle = pp_tp_i2c_comdev_root_handle(i2cdev)) ||
	0 > (i2cfd = pp_i2c_get_fd(i2chandle))) {
	pp_bmc_log_perror("[IPMB] %s: wrong kind of i2cdev ", i2cdev->base.id);
	return PP_ERR;
    }
    /* clear buffer */
    if (ioctl(i2cfd, I2C_SLAVE_FLUSH_BUFFER, 0) < 0) {
	pp_bmc_log_perror("[IPMB] %s: I2C_SLAVE_FLUSH_BUFFER failed",
			  i2cdev->base.id);
	return PP_ERR;
    }
    /* and set slave */
    if (pp_i2c_set_slave_address(pp_tp_i2c_comdev_root_handle(i2cdev),
				 i2c_slv_addr) == PP_ERR) {
	pp_bmc_log_perror("[IPMB] %s: can't set I2C slave address (%#x)",
			  i2cdev->base.id, i2c_slv_addr);
	return PP_ERR;
    }
    /* send */
    pp_bmc_log_debug("[IPMB] %s: write %d bytes to i2c slave (%#x)",
		     i2cdev->base.id, msize - 1, i2c_slv_addr);
#ifdef IPMB_MSG_DBG
    {   int i;
	printf("===> ");
	for (i = 1; i < msize; ++i) printf("%x ", ((char*)msg)[i]);
	printf("\n"); }
#endif
    if (write(i2cfd, &((char*)msg)[1], msize - 1) != (int)(msize - 1)) {
	pp_bmc_log_perror("[IPMB] %s: can't write to i2c slave (%#x)",
			  i2cdev->base.id, i2c_slv_addr);
	return PP_ERR;
    }
    return PP_SUC;
}

static imsg_t* ipmb_msg_recv(ipmb_chan_t* ipmb, unsigned char priv_level) {
    unsigned char msgbuf[MAX_IPMI_MSG_LEN];
    imsg_t* imsg;
    int i2cfd, cnt;
    
    /* read data */
    i2cfd = pp_i2c_get_fd(pp_tp_i2c_comdev_root_handle(ipmb->state->i2cdev));
    
    if (0 > (cnt = read(i2cfd, msgbuf, sizeof(msgbuf)))) {
	pp_bmc_log_perror("[IPMB] %s: receive read failed",
			  ipmb->state->i2cdev->base.id);
	return NULL;
    } else if (cnt < 7) {
	pp_bmc_log_perror("[IPMB] %s: receive msg incomplete",
			  ipmb->state->i2cdev->base.id);
	return NULL;
    }
    pp_bmc_log_debug("[IPMB] %s: successfully read %d bytes",
		     ipmb->state->i2cdev->base.id, cnt);
#ifdef IPMB_MSG_DBG
    {   int i;
	printf("===> ");
	for (i = 1; i < cnt; ++i) printf("%x ", msgbuf[i]);
	printf("\n"); }
#endif

    /* construct an imsg from the received data */
    imsg = pp_bmc_imsg_new(cnt - 6);
    imsg->rs_addr = PP_BMC_IPMB_ADDR;
    imsg->netfn = msgbuf[0] >> 2;
    imsg->rs_lun = msgbuf[0] & 0x3;
    // chk1 is ignored
    imsg->rq_addr = msgbuf[2];
    imsg->rq_seq = msgbuf[3] >> 2;
    imsg->rq_lun = msgbuf[3] & 0x03;
    imsg->cmd = msgbuf[4];
    memcpy(imsg->data, (&msgbuf[5]), imsg->data_size);
    // chk2 is ignored
    imsg->chan = ipmb->channel_no;
    imsg->priv_level = priv_level;
    imsg->session = NULL;
    imsg->buf = NULL;
    imsg->buf_size = 0;
    
    return imsg;
}

static ipmb_chan_t* ipmb_find_chan(unsigned char channel_no) {
    unsigned char i;
    for (i = 0; i < (sizeof(chan) / sizeof(ipmb_chan_t)); ++i) {
	if (chan[i].channel_no == channel_no)
	    return &chan[i];
    }
    return NULL;
}
