/**
 * lan_alert.c
 *
 * The lan channel specific alert functionality.
 * Supports one alert at a time.
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <time.h>
#include <sys/select.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <getopt.h>

#include <ucd-snmp/asn1.h>
#include <ucd-snmp/snmp_api.h>
#include <ucd-snmp/snmp_client.h>
#include <ucd-snmp/mib.h>
#include <ucd-snmp/snmp.h>
#include <ucd-snmp/snmp_parse_args.h>

#include <pp/bmc/bmc_channel_alert.h>
#include <pp/bmc/ipmi_chan.h>

#include <pp/cfg.h>
#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/lan_serv.h>

#include <pp/smtp.h>

#include "lan_alert.h"
#include "lan_serv_intern.h"

/* all information concerning the active alert */
typedef struct {
    // alert type, only valid types
    unsigned char alert_type;
    // ipmi_send_trap params
    in_addr_t ip_addr;
    unsigned char community_str[19];   // 18 byte community string with trailing \0
    unsigned short trap_seq;
    unsigned char guid[16];
    sel_entry_t event;
    unsigned char event_severity;
    // retry params
    unsigned char need_ack;
    unsigned char timeout;
    unsigned char retries;             // remaining (re)tries
    
    int timeout_hndl;                  // the handle of the timeout once the alert is running
    void* alert_ctx;                     // alert context for bmc_pef alert module
} alert_progress_t;

/* functions prototypes used in this file from ucd-snmp/system.h
 * including the whole file is a pain.
 * we should probably replace those functions with something more local */
in_addr_t get_myaddr (void);
long get_uptime (void);


static oid ipmi_trap_oid[] = {1, 3, 6, 1, 4, 1, 3183, 1, 1};
static oid ipmi_trap_oid_varbind[] = {1, 3, 6, 1, 4, 1, 3183, 1, 1, 1};

static alert_progress_t* active_alert;
static int lan_alert_channel_hndl;



/** some internal functions blueprints */
static void* initiate_alert(unsigned char destination,
                          unsigned char* alert_string,
                          sel_entry_t* event,
                          unsigned char event_severity,
                          unsigned char* guid,
                          unsigned char force,
                          void* context);
static int ipmi_send_trap(in_addr_t ip_addr, unsigned char* community_str,
                   unsigned short trap_seq, unsigned char* guid,
                   sel_entry_t* event, unsigned char event_sev);
static int snmp_input(int operation UNUSED, struct snmp_session *session UNUSED,
                      int reqid UNUSED, struct snmp_pdu *pdu UNUSED,
                      void *magic UNUSED);
static int snmp_send_trap(in_addr_t ip_addr, unsigned char* community_str,
                   int event_specific, unsigned char* varbind,
                   int varbind_len);
static int send_next_lan_alert(int item_id UNUSED, void* ctx);
static void acknowledge_alert(void* alert_hndl UNUSED);
static int ipmi_mail_alert(int destination, const sel_entry_t* event, unsigned char* guid);

/**
 * Register the lan alert functionality at the pef handler.
 */
int pp_bmc_lan_alert_init() {
    active_alert = NULL;
    lan_alert_channel_hndl = pp_bmc_pef_alert_register_channel(IPMI_CHAN_LAN, initiate_alert, acknowledge_alert);
    return PP_SUC;
}

/**
 * Unregister the lan alert functionality at the pef handler.
 */
void pp_bmc_lan_alert_cleanup() {
    pp_bmc_pef_unregister_channel(lan_alert_channel_hndl);
    if (active_alert != NULL) {
        free(active_alert);
        active_alert = NULL;
    }
}
     
static void* initiate_alert(unsigned char destination,
                          unsigned char* alert_string UNUSED,
                          sel_entry_t* event,
                          unsigned char event_severity,
                          unsigned char* guid,
                          unsigned char force, 
                          void* context)
{
    alert_progress_t* apro = NULL;
    int suc;
    char* community;
    char* destination_type;
    char* ip;
    int i;
    
    /* check if alerting is disabled for this channel */
    if ((force == 0) & ((get_lan_channel_config())->alerting_disable != 0)) {
        /* alerting is disabled, immediately end alert with an error and return*/
        pp_bmc_log_debug("[PET] alerting on lan channel is disabled");
        pp_bmc_pef_report_alert_status(context, 0);
        return NULL;
    }
    
    // get alert parameters from lan channel config
    if (destination >= 16) {
        pp_bmc_log_error("[PET] lan alerting supports only 15 destinations");
        pp_bmc_pef_report_alert_status(context, 0);
        return NULL;
    }
    
    if (pp_cfg_get_nodflt(&destination_type, "bmc.lan.alert.dest[%u].type", destination) == PP_ERR) {
        pp_bmc_log_warn("[PEF] alert type %d not configured", destination);
        pp_bmc_pef_report_alert_status(context, 0);
        return NULL;
    }
    
    // ok, start processing
    
    if (strcmp(destination_type, "PET") == 0) {
        free(destination_type);
        // pet - snmp trap, use alert progress mechanism
        apro = malloc (sizeof(alert_progress_t));

        // copy params
        pp_cfg_get(&community, "bmc.lan.alert.community");
        strncpy(apro->community_str, community, 18);
        free(community);

        apro->alert_type = 0;  // PET
        apro->trap_seq = event->id_le16;
        memcpy(apro->guid, guid, 16);
        pp_cfg_is_enabled(&i, "bmc.lan.alert.dest[%u].acknowledge",
			  destination);
        apro->need_ack = i;
        pp_cfg_get_int(&i, "bmc.lan.alert.dest[%u].retry_no", destination);
        apro->retries = i;
        pp_cfg_get_int(&i, "bmc.lan.alert.dest[%u].timeout", destination);
        apro->timeout = i;
        pp_cfg_get(&ip, "bmc.lan.alert.dest[%u].trap_addr", destination);
        apro->ip_addr = inet_addr(ip);
        free(ip);

        // gather other info
        apro->event = *event;
        apro->event_severity = event_severity;
        apro->retries++;  // we count first try also as retry
        apro->community_str[18] = 0x00;  // add trailing \0 to be sure the string is terminated
        apro->timeout_hndl = -1;
        apro->alert_ctx = context;

        active_alert = apro;
        
        pp_bmc_log_debug("[PEF] initiating trap");
        send_next_lan_alert(0, (void*)apro);
        return (void*)apro;
    }
    
    if (strcmp(destination_type, "MAIL") == 0) {
        free(destination_type);
        suc = ipmi_mail_alert(destination, event, guid);
        
        if (suc == PP_SUC)
            suc = 1;
        else
            suc = 0;
        
        pp_bmc_pef_report_alert_status(context, suc);
        return NULL;
    }
    
    // we should never come here
    pp_bmc_log_error("[PET] internal error - unknown destination type");
    free(destination_type);
    pp_bmc_pef_report_alert_status(context, 0);
    return NULL;
}

/**
 * Acknowledge the specified alert that was initiated at the channel.
 * Upon reception, the channel may stop further alerting and report a
 * successfull alert to the bmc_dev_pef_alert module.
 */
static void acknowledge_alert(void* alert_hndl UNUSED) {
    void* ctx;
    if ((active_alert->need_ack == 1) &&
        (active_alert->timeout_hndl != -1))
    {
        // stop the timer, end the alert
        pp_bmc_log_debug("[PET] alert %d successfully acknowledged", active_alert->event.id_le16);
        pp_select_remove_to(active_alert->timeout_hndl);
        ctx = active_alert->alert_ctx;
        free(active_alert);
        active_alert = NULL;
        pp_bmc_pef_report_alert_status(ctx, 1);
    }
}

/**
 * process one iteration/retry of an alert
 */
static int send_next_lan_alert(int item_id UNUSED, void* context)  {
    alert_progress_t* apro;
    void* ctx;
    
    apro = (alert_progress_t*)context;
    
    apro->timeout_hndl = -1;
    
    // either a timeout occured or a new alert is started
    if (apro->retries > 0) {
        // start another "iteration", first send alert 
        pp_bmc_log_debug("[PET] sending alert");
        
        if (apro->alert_type == 0) {
            ipmi_send_trap(apro->ip_addr,
                           apro->community_str,
                           apro->trap_seq,
                           apro->guid,
                           &(apro->event),
                           apro->event_severity);
        }
        // insert other alert types here
        
        apro->retries--;
        
        // requeue timeout (always wait for acknowledged, unacknowledged only if retries left)
        if ( (apro->need_ack == 1) |
            ((apro->need_ack == 0) & (apro->retries > 0)) )
        {
            apro->timeout_hndl = 
                pp_select_add_to(apro->timeout*1000, 0, send_next_lan_alert, context);
        } else {
            // unack, no more retries, finish right now
            send_next_lan_alert(0, context);
        }
    } else {
        // no more retries left, end alert
        ctx = active_alert->alert_ctx;
        free(active_alert);
        active_alert = NULL;
        if (apro->need_ack == 1) {
            // no retries and no ack - alert unsuccessful
            pp_bmc_log_debug("[PET] alert unsuccessful");
            pp_bmc_pef_report_alert_status(ctx, 0);
        } else {
            // no acks required, all retries done, alert successful
            pp_bmc_log_debug("[PET] alert successful");
            pp_bmc_pef_report_alert_status(ctx, 1);
        }
    }
    return PP_SUC;
}

/**
 * Construct and send an ipmi pet trap from raw data.
 * @params ip_addr   the destination ip
 * @params guid      the bmc's guid
 * @params event     the event causing the trap
 * @params event_sev the event severity (from the matching pef filter)
 */

static int ipmi_send_trap(in_addr_t ip_addr,
                   unsigned char* community_str,
                   unsigned short trap_seq,
                   unsigned char* guid,
                   sel_entry_t* event,
                   unsigned char event_sev)
{
    int event_specific;
    unsigned char varbind[47];
    unsigned short* usp;
    unsigned int* uip;
    time_t *time_p;
    unsigned char c;
    
    // format the event specific int
    event_specific = 0;
    event_specific = event_specific | (event->sys.sensor_type << 16);
    event_specific = event_specific | ((event->sys.event_type & 0x7F) << 8);
    event_specific = event_specific | ((event->sys.event_dir & 0x01) << 7);
    event_specific = event_specific | (event->sys.event_data[0] & 0x0F);

    // format the varbind string
    //memset(varbind, 'x', 47);
    memcpy(varbind, guid, 16);                //  1..16   guid
    usp = (unsigned short*)(varbind + 16);    // 17..18   sequence#  (unspecified)
    *usp = trap_seq;
    time_p = (time_t*)(varbind + 18);         // 19..22   localtime, 1/1/98 based (time offset 883612800, dont forget timezone offset)
    *time_p = time(NULL);
    if ((*time_p) > ((time_t)883612800)) {
        *time_p = *time_p - ((time_t)883612800);
    } // else: avoid negative timediffs, return undefined time result
    *time_p = cpu_to_be32(*time_p);           // data must be in network order (most significant byte first)
    usp = (unsigned short*)(varbind + 22);    // 23..24   UTC offset  (unspecified)
    *usp = 0xffff;
    varbind[24] = 0x20;                       // 25       Trap Source Type (IPMI)
    varbind[25] = 0x20;                       // 26       Event Source Type (IPMI)
    varbind[26] = event_sev;                  // 27       event severity, from PEF filter config
    c = event->sys.is_swid << 7;              // 28       sensor device
    varbind[27] = c | (event->sys.addr_swid & 0x7F);
    varbind[28] = event->sys.sensor_no;       // 29       sensor number
    varbind[29] = 0x00;                       // 30       entity id (unspecified)
    varbind[30] = 0x00;                       // 31       entity instance (unspecified)
                                              // FIXME: do a SDR lookup for entity id and instance instead of unspec.
    varbind[31] = event->sys.event_data[0];   // 32..39   event data fields
    varbind[32] = event->sys.event_data[1];
    varbind[33] = event->sys.event_data[2];
    varbind[34] = 0;
    varbind[35] = 0;
    varbind[36] = 0;
    varbind[37] = 0;
    varbind[38] = 0;
    varbind[39] = 0x00;                       // 40       language code (unspecified)
    uip = (unsigned int*)(varbind + 40);      // 41 - 44  manufacturer id
    *uip = MANUFACTURER_ID;
    usp = (unsigned short*)(varbind + 44);    // 45 - 46  product id
    *usp = PRODUCT_ID;
    varbind[46] = 0xC1;                       // 47       OEM custom fields (no more records)
    
    return snmp_send_trap(ip_addr, community_str, event_specific, varbind, 47);
}

/* Fake callback function for snmplib */
static int snmp_input(int operation UNUSED,
                      struct snmp_session *session UNUSED,
                      int reqid UNUSED,
                      struct snmp_pdu *pdu UNUSED,
                      void *magic UNUSED)
{
    return 1;
}

/**
 * Send an SNMP trap to the specified destination.
 * (not generic - limited to sending ipmi trap parameters)
 * @returns PP_SUC or PP_ERR
 */
static int snmp_send_trap(in_addr_t ip_addr,
                   unsigned char* community_str,
                   int event_specific,
                   unsigned char* varbind,
                   int varbind_len)
{
    struct snmp_session session, *ss;
    struct snmp_pdu *pdu;
    struct sockaddr_in *pduIp;

    int status;
    
    putenv(strdup("POSIXLY_CORRECT=1"));

    memset(&session, 0, sizeof(struct snmp_session));
    session.version = SNMP_VERSION_1;
    session.community = strdup(community_str);
    session.community_len = strlen(session.community);

    session.timeout = 100000;  // in uSec.
    session.retries = 1;
    
    session.callback = snmp_input;
    session.callback_magic = NULL;
    session.remote_port = SNMP_TRAP_PORT;

    ss = snmp_open(&session);
    if (ss == NULL){
        /* diagnose snmp_open errors with the input struct snmp_session pointer */
        pp_bmc_log_debug("[PET] cannot create session for SNMP trap sending");
        snmp_sess_perror("", &session);
        return PP_ERR;
    }

    pdu = snmp_pdu_create(SNMP_MSG_TRAP);
    
    // determine oid (fixed for ipmi trap)
    pdu->enterprise = (oid *)malloc(sizeof (ipmi_trap_oid));
    memcpy(pdu->enterprise, ipmi_trap_oid, sizeof(ipmi_trap_oid));
    pdu->enterprise_length = sizeof(ipmi_trap_oid)/sizeof (oid);

    // agent ip (my ip)
    pduIp = (struct sockaddr_in *)&pdu->agent_addr;
    pduIp->sin_family = AF_INET;
    pduIp->sin_addr.s_addr = get_myaddr();

    pduIp = (struct sockaddr_in *)&pdu->address;
    pduIp->sin_family = AF_INET;
    pduIp->sin_addr.s_addr = ip_addr;
    pduIp->sin_port = htons(SNMP_TRAP_PORT);

    // trap type
    pdu->trap_type = 6;
    pdu->specific_type = event_specific;

    // uptime
    pdu->time = get_uptime();

    // add varbind field
    snmp_pdu_add_variable(pdu, 
                          ipmi_trap_oid_varbind,
                          sizeof(ipmi_trap_oid_varbind)/sizeof(oid),
                          ASN_OCTET_STR,
                          varbind,
                          varbind_len);
        
    status = snmp_send(ss, pdu) == 0;
    if (status) {
        pp_bmc_log_debug("[PET] could not send SNMP ipmi trap");
        //snmp_sess_perror("snmptrap", ss);
        snmp_free_pdu(pdu);
        snmp_close(ss);
        return PP_ERR;
    }

    snmp_close(ss);
    return PP_SUC;
}

/**
 * Send an alert mail.
 * @returns PP_SUC/PP_ERR
 */
static int ipmi_mail_alert(int destination, const sel_entry_t* event,
			   unsigned char* guid)
{    
    int ret = PP_ERR, i;
    mail_data_t data;
    char *mail_txt, *subject, *to, *host, *from, *ipaddr;
    pp_strstream_t sbuf = PP_STRSTREAM_INITIALIZER;
    mail_txt = subject = to = host = from = ipaddr = NULL;
    
    
    if (destination > 16) {
        // no email receiver to read
        return PP_ERR;
    }
    
    pp_cfg_get(&to, "bmc.lan.alert.dest[%u].mail_dest", destination);
    if (strnlen(to, 64) == 0) goto bailout;

    if (pp_cfg_get_nodflt(&host, "bmc.lan.alert.smtp_server") == PP_ERR) {
        // no smtp server, very bad
        pp_bmc_log_debug("[PET] no smtp server configured");
        goto bailout;
    }

    if (pp_cfg_get_nodflt(&from, "bmc.lan.alert.mail_alert_from") == PP_ERR) {
        pp_bmc_log_debug("[PET] no mail alert sender specified");
        // set backup value
        from = strdup("mailalert@bmc");
    }
    
    if (pp_cfg_get(&ipaddr, "network.ipaddr") == PP_SUC) {
	asprintf(&subject, "PEF EMail alert from %s", ipaddr);
    } else {
        subject = strdup("PEF EMail alert");
    }
    
    /* FIXME: we really should create a proper text here */
    pp_strappend (&sbuf, "PEF EMail alert\n");
    pp_strappendf(&sbuf, "System IP: %s\n", ipaddr);
    pp_strappend (&sbuf, "System GUID:");
    for (i=0; i<16; i++) {
        pp_strappendf(&sbuf, " %.2x", guid[i]);
    }
    pp_strappend (&sbuf, "\n\n");
    pp_strappendf(&sbuf, "Sensor Type: %.2x\n", event->sys.sensor_type);
    pp_strappendf(&sbuf, "Sensor No.:  %.2x\n", event->sys.sensor_no);
    pp_strappendf(&sbuf, "Event Type:  %.2x\n", event->sys.event_type);
    pp_strappendf(&sbuf, "Data:        %.2x %.2x %.2x\n\n",
		 event->sys.event_data[0], event->sys.event_data[1],
		 event->sys.event_data[2]);
    mail_txt = pp_strstream_buf_and_free(&sbuf);

    /* fill mail structure and send off */
    data.time_stamp = time(NULL);
    data.mail_txt = mail_txt;
    data.subject = subject;
    data.to = to;
    data.host = host;
    data.from = from;
    data.extra_hdrs = NULL;
    if (PP_ERR == pp_smtp_send_mail(&data)) {
        pp_bmc_log_debug("[PET] SMTP lan alert failed");
	goto bailout;
    }
    ret = PP_SUC;

 bailout:
    free(ipaddr);
    free(subject);
    free(from);
    free(to);
    free(host);
    free(mail_txt);
    return ret;
}

