/**
 * lan_ipmi20.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Handles the IPMIv20 messages for the ipmi class of RMCP+, including RAKP authentication.
 */

#include "rmcp.h"
#include "lan_cipher.h"
#include "lan_ipmi20.h"
#include "lan_serv_intern.h"

#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_sess.h>

#include <pp/bmc/lan_serv.h>
#include <pp/bmc/session_manager.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>

#include <pp/base.h>
#include <pp/cfg.h>

#include <openssl/hmac.h>

const char *PP_BMC_CHANNEL_ACCESS_KEYGEN_KEY = "bmc.channel_access[%u].keygen_key";

/* Internal prototypes */
static int send_rakp_error_response(unsigned char payload_type, unsigned char message_tag, unsigned char status_code,
                                    unsigned int session_id, lan_addr_t* addr);
static int handle_rakp_open_session_request(unsigned char* data, int len,
                                imsg_session_t* e_session, lan_addr_t* addr);
static int handle_rakp_message1(unsigned char* data, int len,
                                imsg_session_t* e_session, lan_addr_t* addr);
static int handle_rakp_message3(unsigned char* data, int len,
                                imsg_session_t* e_session, lan_addr_t* addr);
static int handle_sol_message(unsigned char* data, int len,
			      imsg_session_t* e_session, lan_addr_t* addr);

static int calculate_SIK(imsg_session_t* session);
static int calculate_RAKP_message2_authcode(imsg_session_t* session, unsigned char* dest);
static int check_RAKP_message3_authcode    (imsg_session_t* session, unsigned char* authcode);
static int calculate_RAKP_message4_authcode(imsg_session_t* session, unsigned char* dest);



int pp_ipmi20_receive(unsigned char* data, int data_len, lan_addr_t* addr) {
    unsigned char authtype;
    unsigned char payload_type;
    unsigned char payload_encrypted;
    unsigned char payload_authenticated;
    
    unsigned int session_id;
    unsigned int sequence_no;
        
    unsigned short payload_raw_len;
    unsigned char* payload_raw;   // payload with confidentiality_header/encryption
    unsigned short payload_data_len;
    unsigned char* payload_data;  // payload without encryption (may be the same as payload_raw)
    
    imsg_session_t *session;
    
    unsigned int *uip;
    unsigned short *usp;
    int i;
    int rv;
    
    // we have at least 12 bytes session header
    if (data_len < 12) {
        pp_bmc_log_notice("[LAN] IPMIv20 message too short - ignored");
        return PP_ERR;
    }
    
    // parse session header
    authtype = data[0];
    if (authtype != IPMI_AUTH_RMCPP) {
        // should not happen, authtype is already checked
        pp_bmc_log_notice("[LAN] IPMIv20 unknown authtype - message ignored");
        return PP_ERR;
    }
    payload_type = data[1] & 0x3F;
    payload_encrypted =  ((data[1] & 0x80) == 0x80);
    payload_authenticated = ((data[1] & 0x40) == 0x40);
    
    // check if we can handle this type of payload
    if ((payload_type != IPMI_PAYLOAD_IPMI) &&
	(payload_type != IPMI_PAYLOAD_SOL) &&
        (payload_type != IPMI_PAYLOAD_RAKP_OPEN_SESSION_REQUEST) &&
        (payload_type != IPMI_PAYLOAD_RAKP_MESSAGE1) &&
        (payload_type != IPMI_PAYLOAD_RAKP_MESSAGE3))
    {
        pp_bmc_log_notice("[LAN] IPMIv20 unknown payload type - message ignored");
        return PP_ERR;
    }
    
    uip = (unsigned int*)(data+2);
    session_id = le32_to_cpu(*uip);
    uip = (unsigned int*)(data+6);
    sequence_no = le32_to_cpu(*uip);
    usp = (unsigned short*)(data+10);
    payload_raw_len = le16_to_cpu(*usp);
    payload_raw = data+12;

    // Note: ipmitool illegaly passes the session_id in rakp open messages
    if ((payload_type == IPMI_PAYLOAD_RAKP_OPEN_SESSION_REQUEST) ||
        (payload_type == IPMI_PAYLOAD_RAKP_MESSAGE1) ||
        (payload_type == IPMI_PAYLOAD_RAKP_MESSAGE3))
    {
        session_id = 0;
    }

    // try to get session
    if (session_id != 0) {
        session = pp_bmc_get_session(session_id);
        if (session == NULL) {
            pp_bmc_log_notice("[LAN] IPMIv20 ignored message with unknown session id");
            return PP_ERR;
        }
        
        if (payload_authenticated == 0) {
            if (session->integrity_cipher != IPMI20_INTEGRITY_NONE) {
                pp_bmc_log_notice("[LAN] IPMIv20 session requires authentication");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
            // try to authenticate with session->integrity_cipher later on
        }
        
        if (payload_encrypted == 0) {
            if (session->confidentiality_cipher != IPMI20_CONF_NONE) {
                pp_bmc_log_notice("[LAN] IPMIv20 session requires encryption");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
            // try to decrypt with session->confidentiality_cipher later on
        }
    } else {
        // no session, no need to delete session
        if (payload_authenticated == 1) {
            pp_bmc_log_notice("[LAN] IPMIv20 cannot authenticate sessionless message");
            return PP_ERR;
        }
        if (payload_encrypted == 1) {
            pp_bmc_log_notice("[LAN] IPMIv20 cannot decrypt sessionless message");
            return PP_ERR;
        }
        session = NULL;
    }

    // selected integrity/encryption types allowed, session checked
    
    // authenticate msg if necessary, check msglen
    if (payload_authenticated == 1) {
        // authenticated payload, session exists
        int int_len;
        int int_pad_len;
        unsigned char* trailer;
        
        // calculate integrity pad (without pad_len and next_header field)
        int_pad_len = 4 - ((12+payload_raw_len+2) % 4);
        if (int_pad_len == 4) int_pad_len = 0;
        assert(int_pad_len >= 0);
        assert(int_pad_len < 4);
        
        // the size of the authcode
        int_len = pp_integrity_length(session->integrity_cipher);

        // this is a workaround because ICTS sends unpadded data
        if (int_pad_len != 0 && data_len == (12 + payload_raw_len + 2 + int_len)) {
            pp_bmc_log_warn("[LAN] IPMIv20 messages unpadded (ICTS?) - continue anyway");
            int_pad_len = 0;  // assume that this is an unpadded icts message and check again
        }
        // check data len 
        if (data_len != (12 + payload_raw_len + int_pad_len + 2 + int_len)) {
            pp_bmc_log_notice("[LAN] IPMIv20 message to short (%d instead of %d)", data_len, (12 + payload_raw_len + int_pad_len + 2 + int_len));
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        // check integrity pad
        trailer = data+12+payload_raw_len;
        if (trailer[int_pad_len] != int_pad_len) {
            pp_bmc_log_notice("[LAN] IPMIv20 invalid integrity pad length");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        for (i=0; i<int_pad_len; i++)
            if (trailer[i] != 0xFF) {
                pp_bmc_log_notice("[LAN] IPMIv20 invalid integrity pad");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
        
        // integrity pad correct, check authcode
        if (pp_integrity_check(session, data, (12 + payload_raw_len + int_pad_len + 2 ),
                              (trailer + int_pad_len + 2)) != PP_SUC)
        {
            pp_bmc_log_notice("[LAN] IPMIv20 message integrity check failed - ignored");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
    } else {
        // unauthenticated payload, no trailer
        if (data_len < 12 + payload_raw_len) {
            pp_bmc_log_notice("[LAN] IPMIv20 message with incorrect payload size - ignored");
            if (session) pp_bmc_session_delete(session);
            return PP_ERR;
        }
    }
    
    // decrypt message if necessary, msglen is okay
    if (payload_encrypted == 1) {
        // payload encrypted
        int decrypted_payload_len;
        unsigned char* encrypt_inbuf;

        // copy input data as we want to use the same buffer for output
        // (decrypted message overwrites original payload)
        encrypt_inbuf = malloc(payload_raw_len);
        memcpy(encrypt_inbuf, payload_raw, payload_raw_len);
        decrypted_payload_len = pp_confidentiality_decrypt(session,
                                    encrypt_inbuf, payload_raw_len, payload_raw);
        free(encrypt_inbuf);

        if (decrypted_payload_len < 0) {
            pp_bmc_log_notice("[LAN] IPMIv20 could not decrypt message - message ignored");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        payload_data_len = decrypted_payload_len;
        payload_data = payload_raw;
    } else {
        // payload is unencrypted (same as raw)
        payload_data = payload_raw;
        payload_data_len = payload_raw_len;
    }
    
    // decrypted and authenticated
    rv = PP_SUC;
    
    if (session != NULL) {
        /* Check the sequence number. All our messages are either authenticated *
         * or unauthenticated, so we do not have two pairs of sequence number    */
        
        int wrap_add = 0;
        
        if (sequence_no < 0x0000000f) {
            wrap_add = 30;
        }

        /* we check the sequence as in ipmi15, not as ipmi20 proposes, see spec 6.12.14 */
        if ((sequence_no + wrap_add) <= (session->inbound_sequence + wrap_add)
            || (sequence_no + wrap_add) > (session->inbound_sequence + 8 + wrap_add))
        {
            /* sequence number is either old or more than 8 ahead */
            if ((session->last_message_seq != 0) &&
                (session->last_message_seq == sequence_no) &&
                (session->last_message_data) )
            {
                /* client resent previous message, resend last response */
                pp_bmc_log_warn("[LAN] IPMIv20 resent previous message in binary form");
                return pp_send_rmcp_msg(session->last_message_data, session->last_message_len, &(session->last_message_addr), RMCP_CLASS_IPMI);
            }

            /* sequence number is completely wrong */
            pp_bmc_log_warn("[LAN] IPMIv20 received msg with invalid sequence number "
                "(is=%d, expect=%d..%d)",
                sequence_no + wrap_add,
                session->inbound_sequence + 1 + wrap_add,
                session->inbound_sequence + 8 + wrap_add);
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        session->inbound_sequence = sequence_no;
        session->last_message_seq = sequence_no;
        if (session->inbound_sequence == 0) {
            /* client sent invalid sequence number (wrong wraparound at client) */
            pp_bmc_log_warn("[LAN] wrong sequence number wraparound at client detected");
            session->inbound_sequence = 1;
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
    }
    
    if ((session != NULL) && (session->ipmi20_session_state == IPMI20_STATE_ACTIVATED)) {
        session->ipmi20_session_state = IPMI20_STATE_ACTIVE;
    }
    
    switch (payload_type) {
    case IPMI_PAYLOAD_IPMI:
	rv = pp_bmc_lanserv_ipmi_handle_receive(payload_data, payload_data_len, 
					        session, 0, addr);
	break;
    case IPMI_PAYLOAD_SOL:
        pp_bmc_session_refresh(session);
	rv = handle_sol_message(payload_data, payload_data_len, 
				session, addr);
	break;
    case IPMI_PAYLOAD_RAKP_OPEN_SESSION_REQUEST:
	rv = handle_rakp_open_session_request(payload_data, payload_data_len, 
					      session, addr);
	break;
    case IPMI_PAYLOAD_RAKP_MESSAGE1:
	rv = handle_rakp_message1(payload_data, payload_data_len, session, 
				  addr);
	break;
    case IPMI_PAYLOAD_RAKP_MESSAGE3:
	rv = handle_rakp_message3(payload_data, payload_data_len, session, 
				  addr);
	break;
    default:
	// should not happen as payload_type was checked above but who knows
	pp_bmc_log_warn("[LAN] ignored unknown payload");
	rv = PP_ERR;
	break;
    }
    
    if (session != NULL) {
        pp_bmc_session_delete(session);
    }
    
    return rv;
}

int pp_ipmi20_send_payload(unsigned char* data, int data_len, unsigned char payload_type,
                           imsg_session_t* session, lan_addr_t* addr)
{
    unsigned char* buf;
    int buflen;            // header + confidentialtiy(payload) + integrity_pad + int_pad_len/nexthdr + integrity
    
    int payload_len;       // length of the encrypted payload
    int integrity_pad_len = 0; // number of pad bytes (without  len/nexthdr bytes)
    int integrity_len = 0;     // authcode len
    int trailer_len = 0;     // IPMI trailer
    
    unsigned int session_id;
    unsigned int sequence_no;
    
    unsigned char payload_authenticated;
    unsigned char payload_encrypted;

    unsigned int *uip;
    unsigned short *usp;
    int rv;
    
    if (session == NULL) {
        session_id = 0;
        sequence_no = 0;
        payload_authenticated = 0;
        payload_encrypted = 0;
        // calculate the buffer length
        payload_len = data_len;
    } else {
        session_id = session->manager_session_id;
        
        /* Handle the sequence number. All our messages are either authenticated *
         * or unauthenticated, so we do not have two pairs of sequence number    */
        session->outbound_sequence++;
        if (session->outbound_sequence == 0) session->outbound_sequence++;
        sequence_no = session->outbound_sequence;

        // check if to authenticate / encrypt
        if (session->integrity_cipher == IPMI20_INTEGRITY_NONE)
            payload_authenticated = 0;
        else
            payload_authenticated = 1;
        
        if (session->confidentiality_cipher == IPMI20_CONF_NONE)
            payload_encrypted = 0;
        else
            payload_encrypted = 1;
        
        // calculate the buffer length, including integrity pad ...
        if (payload_encrypted)
            payload_len = pp_confidentiality_length(session->confidentiality_cipher, data_len);
        else
            payload_len = data_len;

        if (payload_authenticated) {
            integrity_len = pp_integrity_length(session->integrity_cipher);
            
            integrity_pad_len = 4 - ( (12 + payload_len + 1) % 4 ) + 1;
            integrity_pad_len = integrity_pad_len - 2; // FIXME: rgue: really? can be negative!

            // integrity_pad + pad_len/next_hdr + authcode
            trailer_len = integrity_pad_len + 2 + integrity_len; 
        }
    }
    // buffer = header + crypt(payload) + trailer
    buflen = 12 + payload_len + trailer_len;
    buf = malloc(buflen);
    
    buf[0] = IPMI_AUTH_RMCPP; 
    buf[1] = payload_type | (payload_authenticated << 6) | (payload_encrypted << 7);
    
    uip = (unsigned int*)(buf + 2);
    *uip = cpu_to_le32(session_id);
    uip = (unsigned int*)(buf + 6);
    *uip = cpu_to_le32(sequence_no);
    
    if (payload_encrypted) {
        payload_len = pp_confidentiality_encrypt(session, data, data_len, buf+12);

        if (payload_len < 0) {
            pp_bmc_log_notice("[LAN] IPMIv20 failed to encrypt payload");
            free(buf);
            return PP_ERR;
        }
    } else {
        // unencrypted payloads, just copy
        payload_len = data_len;
        memcpy(buf+12, data, data_len);
    }

    usp = (unsigned short*)(buf + 10);
    *usp = cpu_to_le16(payload_len);
    
    if (payload_authenticated) {
        // add integrity pad bytes
        memset(buf+12+payload_len, 0xFF, integrity_pad_len);
        buf[12+payload_len+integrity_pad_len] = integrity_pad_len;
        // next header byte, fixed
        buf[12+payload_len+integrity_pad_len+1] = 0x07;

        if (pp_integrity_calculate(session,
                               buf, (12+payload_len+integrity_pad_len+2),
                               (buf + 12+payload_len+integrity_pad_len+2)) < 0)
        {
            pp_bmc_log_notice("[LAN] IPMIv20 failed to calculate integrity code");
            free(buf);
            return PP_ERR;
        }
    } else {
        // unauthenticated payload
        // trailer not present
    }
    
    if (session && sequence_no != 0) {
        /* this message belongs to a session. Store this message and
         * resend it later on if the client repeats the request. We 
         * assume that there is only one active message and thus we
         * have only one buffer. */
        pp_bmc_session_clear_last_message(session);
        session->last_message_data = malloc(buflen);
	memcpy(session->last_message_data, buf, buflen);
	session->last_message_len = buflen;
	session->last_message_addr = *addr;
    }
    
    rv = pp_send_rmcp_msg(buf, buflen, addr, RMCP_CLASS_IPMI);
    
    free(buf);
    return rv;
}


static int send_rakp_error_response(unsigned char payload_type, unsigned char message_tag, unsigned char status_code,
                                    unsigned int session_id, lan_addr_t* addr)
{
    unsigned char buf[20];
    unsigned int* uip;
    
    buf[0] = IPMI_AUTH_RMCPP;
    buf[1] = payload_type;
    memset (buf+2, 0, 8);  // sess_id, seq = 0
    buf[10] = 0x08;
    buf[11] = 0x00;
    buf[12] = message_tag;
    buf[13] = status_code;
    buf[14] = 0;
    buf[15] = 0;
    uip = (unsigned int*)(buf+16);
    *uip = cpu_to_le32(session_id);
    
    return pp_send_rmcp_msg(buf, 20, addr, RMCP_CLASS_IPMI);
}


/**
 *  handle a RAKP opensession request
 */
static int handle_rakp_open_session_request(unsigned char* data, int data_len,
                                            imsg_session_t* e_session, lan_addr_t* addr)
{
    unsigned char message_tag;
    unsigned char req_priv_lvl;
    unsigned int  remote_session_id;

    unsigned char auth_cipher;
    unsigned char integrity_cipher;
    unsigned char conf_cipher;
    
    imsg_session_t* session;

    unsigned int *uip;

    unsigned char  buffer[36];  // fixed response length
    
    pp_bmc_log_debug("[LAN] RAKP received open session request");
    
    if (e_session != NULL) {
        // spec says that session_id should be 0 but we could easily ignore this
        pp_bmc_log_debug("[LAN] RAKP attempt to reopen 2.0 session ignored");
        return PP_ERR;
    }
    
    if (data_len != 32) {
        // fixed length
        pp_bmc_log_warn("[LAN] RAKP open_session_request with wrong msg length "
                        "(is=%d expected=%d)", data_len, 32);
        return PP_ERR;
    }
    
    message_tag = data[0];
    req_priv_lvl = data[1] & 0xF;
    // data[2,3] reserved
    uip = (unsigned int*)(data+4);
    remote_session_id = le32_to_cpu(*uip);
    
    // data[8-15]  Authentication cipher request (for RAKP session creation)
    if (data[8] != 0x0) {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect authentication payload");
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
    }
    if (data[11] == 0x08) {
        auth_cipher = data[12];
    } else
    if (data[11] == 0x00) {
        auth_cipher = IPMI20_AUTH_ANY;
    } else {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect authentication payload");
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
    }
    
    // data[16-23] Integrity cipher request  (for RMCP+ message authentication)
    if (data[16] != 0x1) {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect integrity payload");
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
    }
    if (data[19] == 0x08) {
        integrity_cipher = data[20];
    } else
    if (data[19] == 0x00) {
        integrity_cipher = IPMI20_INTEGRITY_ANY;
    } else {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect integrity payload");
        send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                 IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
        return PP_ERR;
    }

    // data[24-31] Confidentiality cipher request (for RMCP+ message encryption)
    if (data[24] != 0x2) {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect confidentiality payload");
        send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                 IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
        return PP_ERR;
    }
    if (data[27] == 0x08) {
        conf_cipher = data[28];
    } else
    if (data[27] == 0x00) {
        conf_cipher = IPMI20_CONF_ANY;
    } else {
        pp_bmc_log_notice("[LAN] RAKP open_session_request with incorrect confidentiality payload");
        send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                 IPMI_RAKP_STAT_INVALID_PARAM, remote_session_id, addr);
        return PP_ERR;
    }
    
    session = pp_bmc_create_session20(req_priv_lvl, auth_cipher, integrity_cipher, conf_cipher);

    if (session == NULL) {
        /* this can have several reasons, usually the requested cipher suites do not match */
        pp_bmc_log_debug("[LAN] RAKP open_sess_req: no such cipher suite (%d, %d, %d) "
                         "or insufficient priv lvl (%d)",
                         auth_cipher, integrity_cipher, conf_cipher, req_priv_lvl);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, message_tag,
                                        IPMI_RAKP_STAT_NO_MATCHING_CIPHER, remote_session_id, addr);
    }
    session->manager_session_id = remote_session_id;
    session->ipmi20_session_state = IPMI20_STATE_WAITING_RAKP1;
    
    // ready to compute response
    buffer[0] = message_tag;
    buffer[1] = 0x00;          // no errors
    buffer[2] = session->req_priv_level;
    buffer[3] = 0x00;
    uip = (unsigned int*)(buffer + 4);
    *uip = cpu_to_le32(session->manager_session_id);
    uip = (unsigned int*)(buffer + 8);
    *uip = cpu_to_le32(session->session_id);
    
    memset(buffer+12, 0, 24);

    buffer[12] = 0x00;
    buffer[15] = 0x08;
    buffer[16] = session->authentication_cipher;
    
    buffer[20] = 0x01;
    buffer[23] = 0x08;
    buffer[24] = session->integrity_cipher;
    
    buffer[28] = 0x02;
    buffer[31] = 0x08;
    buffer[32] = session->confidentiality_cipher;

    /* free our handle to the session (the session will still be refcounted in the sessionlist */
    pp_bmc_session_delete(session);
    
    // send the message, use NULL for session (as the answer must be unauthenticated)
    return pp_ipmi20_send_payload(buffer, 36, IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE, NULL, addr);
}


/**
 * handle a RAKP message 1
 */
static int handle_rakp_message1(unsigned char* data, int data_len,
                                imsg_session_t* e_session, lan_addr_t* addr)
{
    unsigned char  message_tag;
    unsigned int   session_id;
    unsigned char  username_len;
    unsigned char* username;
    unsigned char  name_only_lookup;
    unsigned char  req_priv;
    
    unsigned int* uip;
    imsg_session_t * session;

    unsigned char* buffer;  // length = 41 + authcode_len
    int buffer_len;
    
    int rv;

    pp_bmc_log_debug("[LAN] RAKP message 1 received");

    if (e_session != NULL) {
        // spec says that session_id should be 0 but we could easily ignore this
        pp_bmc_log_notice("[LAN] RAKP ignored message 1 within session");
        return PP_ERR;
    }

    if (data_len < 28) {
        pp_bmc_log_notice("[LAN] RAKP message 1 with wrong msg length");
        return PP_ERR;
    }
    
    message_tag = data[0];
    // byte 2-4 reserved
    uip = (unsigned int*)(data+4);
    session_id = le32_to_cpu(*uip);
    
    session = pp_bmc_get_session(session_id);
    if (session == NULL) {
        pp_bmc_log_notice("[LAN] RAKP could not retrieve session for message 1");
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_INVAL_SESS_ID, session_id, addr);
    }
    
    switch (session->ipmi20_session_state) {
    case IPMI20_STATE_WAITING_RAKP1:
        /* thats what we expect */
        break;
    case IPMI20_STATE_WAITING_RAKP3:
        /* we're already one message ahead, maybe the client did not receive *
         * message 2 and thus resends message 1 ?                            */
        // accept anyway, realloc all values, redo all changes
        break;
    case IPMI20_STATE_ACTIVATED:
        pp_bmc_log_notice("[LAN] RAKP received RAKP message 1 for active session");
        pp_bmc_session_delete(session);
        return PP_ERR;
        break;
    case IPMI20_STATE_ACTIVE:
    default:
        pp_bmc_log_debug("[LAN] RAKP message 1 ignored");
        pp_bmc_session_delete(session);
        return PP_ERR;
        break;
    }
    
    if (session->manager_random != NULL) free (session->manager_random);
    session->manager_random = malloc(16);
    memcpy(session->manager_random, data+8, 16);

    username_len = data[27];
    username = &(data[28]);
    if (username_len > 16) {
        pp_bmc_log_notice("[LAN] RAKP message 1 username too long");
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_INVAL_NAME_LEN, session_id, addr);
    }

    // not sure if data_len has to be 28+16 or 28+username_len
    if (data_len < (28 + username_len)) {
        pp_bmc_log_notice("[LAN] RAKP message 1 is too short for given username");
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_INVAL_NAME_LEN, session_id, addr);
    }
    
    name_only_lookup = ((data[24] & 0x10) == 0x10) ? 1:0;
    req_priv = data[24] & 0x0f;

    if (!name_only_lookup && req_priv > session->req_priv_level) {
        pp_bmc_log_debug("[LAN] RAKP invalid privilege specified in message 1 (userpriv=%d sesspriv=%d)",
                         req_priv, session->req_priv_level);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_UNAUTH_ROLE_OR_PRIV, session_id, addr);
    }

    /* open the session                                                *
     * sets the username, password, max privilege level and challenge  */
    if (pp_bmc_open_session20(session, username_len, username, name_only_lookup, req_priv) == PP_ERR) {
        pp_bmc_log_debug("[LAN] RAKP invalid user specified in message 1 (user=%s)", username);
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_UNAUTH_NAME, session_id, addr);
    }
    
    // compute rakp message 2
    
    buffer_len = 40 + pp_authentication_length(session->authentication_cipher, IPMI_PAYLOAD_RAKP_MESSAGE2);
    
    buffer = malloc(buffer_len);
    memset(buffer, 0, buffer_len);
    
    buffer[0] = message_tag;
    buffer[1] = 0x00;
    buffer[2] = 0x00;  // reserved
    buffer[3] = 0x00;  // reserved
    uip = (unsigned int*)(buffer + 4);
    *uip = cpu_to_le32(session->manager_session_id);
    memcpy(buffer+8, session->challenge, 16);  // my random

    pp_bmc_get_guid(buffer+24);
    
    // 40 - x: authcode
    if (calculate_RAKP_message2_authcode(session, buffer+40) < 0) {
        pp_bmc_log_notice("[LAN] IPMIv20 could not calculate authcode for RAKP message2");
        free(buffer);
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE2, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, session_id, addr);
    }
    
    pp_bmc_log_debug("[LAN] sening RAKP message 2");
    rv = pp_ipmi20_send_payload(buffer, buffer_len, IPMI_PAYLOAD_RAKP_MESSAGE2, NULL, addr);
    session->ipmi20_session_state = IPMI20_STATE_WAITING_RAKP3;
    
    pp_bmc_session_delete(session);
    free(buffer);
    return rv;
}


/**
 * handle a RAKP message 3
 */
static int handle_rakp_message3(unsigned char* data, int data_len,
                                imsg_session_t* e_session, lan_addr_t* addr)
{
    unsigned char message_tag;
    unsigned int session_id;
    unsigned int* uip;
    imsg_session_t * session;

    unsigned char* buffer;
    int buffer_len;
    int rv;

    
    pp_bmc_log_debug("[LAN] RAKP message 3 received");

    if (e_session != NULL) {
        // spec says that session_id should be 0 but we could easily ignore this
        pp_bmc_log_notice("[LAN] RAKP ignored message 3 within session");
        return PP_ERR;
    }

    if (data_len < 8) {
        pp_bmc_log_notice("[LAN] RAKP message3 with wrong msg length");
        return PP_ERR;
    }

    message_tag = data[0];
    
    uip = (unsigned int*)(data + 4);
    session_id = le32_to_cpu(*uip);
    
    session = pp_bmc_get_session(session_id);
    if (session == NULL) {
        pp_bmc_log_notice("[LAN] RAKP could not retrieve session for message 3");
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE4, message_tag,
                                        IPMI_RAKP_STAT_INACTIVE_SESS_ID, session_id, addr);
    }

    switch (session->ipmi20_session_state) {
    case IPMI20_STATE_WAITING_RAKP1:
        pp_bmc_log_debug("[LAN] RAKP received RAKP message 3 without message 1");
        pp_bmc_session_delete(session);
        return PP_ERR;
        break;
    case IPMI20_STATE_WAITING_RAKP3:
        /* this is the message we're waiting for  */
        break;
    case IPMI20_STATE_ACTIVATED:
        /* maybe message 4 got lost and the client resends message 3 */
        break;
    case IPMI20_STATE_ACTIVE:
    default:
        pp_bmc_log_debug("[LAN] RAKP message 3 ignored");
        pp_bmc_session_delete(session);
        return PP_ERR;
        break;
    }

    if (data_len != (8 + pp_authentication_length(session->authentication_cipher, IPMI_PAYLOAD_RAKP_MESSAGE3))) {
        // wrong size, better ignore
        pp_bmc_log_notice("[LAN] RAKP message 3 has wrong size (is=%d exp=%d cypher-suite=%d) - ignored",
            data_len,
            8 + pp_authentication_length(session->authentication_cipher, IPMI_PAYLOAD_RAKP_MESSAGE3),
            session->authentication_cipher);
        pp_bmc_session_delete(session);
        return PP_ERR;
    }
    
    // check data[9..n] for authcode
    if (check_RAKP_message3_authcode(session, data+8) == PP_ERR) {
        pp_bmc_log_notice("[LAN] RAKP message 3 has wrong authcode - ignored");
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE4, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, session_id, addr);
    }
    
    if (data[1] != 0x00) {
        pp_bmc_log_notice("[LAN] RAKP message 3 contained error code");
        pp_bmc_session_delete(session);
        return PP_ERR;
    }
    
    // construct SIK
    if (calculate_SIK(session) != PP_SUC) {
        // this is very bad and we cannot recover, so better close that session ...
        pp_bmc_log_debug("[LAN] RAKP message3 could not calculate SIK");
        pp_bmc_close_session(session->session_id);
        pp_bmc_session_delete(session);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE4, message_tag,
                                        IPMI_RAKP_STAT_INSUF_RES, session_id, addr);
    }
    
    // activate session
    pp_bmc_log_debug("[LAN] activated ipmi20 session %d", session->session_id);
    session->max_priv_level = session->req_priv_level;
    session->cur_priv_level = session->req_priv_level;
    session->inbound_sequence = 0;
    
    // compute rakp message 4
    
    buffer_len = 8 + pp_authentication_length(session->authentication_cipher, IPMI_PAYLOAD_RAKP_MESSAGE4);
    buffer = malloc(buffer_len);
    
    buffer[0] = message_tag;
    buffer[1] = 0x00;
    buffer[2] = 0x00;   // reserved
    buffer[3] = 0x00;   // reserved
    uip = (unsigned int*)(buffer+4);
    *uip = cpu_to_le32(session->manager_session_id);
    
    // construct hmac in buffer+8
    if (calculate_RAKP_message4_authcode(session, buffer+8) != PP_SUC) {
        pp_bmc_log_debug("[LAN] RAKP message 4 could not calculate authcode");
        pp_bmc_session_delete(session);
        free(buffer);
        return send_rakp_error_response(IPMI_PAYLOAD_RAKP_MESSAGE4, message_tag,
                                        IPMI_RAKP_STAT_INVALID_PARAM, session_id, addr);
    }
    
    pp_bmc_log_debug("[LAN] sening RAKP message 4");
    rv = pp_ipmi20_send_payload(buffer, buffer_len, IPMI_PAYLOAD_RAKP_MESSAGE4, NULL, addr);
    session->ipmi20_session_state = IPMI20_STATE_ACTIVATED;

    pp_bmc_session_delete(session);
    free(buffer);
    return rv;
}

/**
 * Calculate the session integrity key based on the other values stored in
 * session.
 * @return PP_ERR/PP_SUC
 */
static int calculate_SIK(imsg_session_t* session) {
    unsigned char sik_src[50];
    unsigned int sik_src_len;
    int len;
    int i;

    char *keygen_key;
    size_t keygen_length;
    unsigned char const1[20];
    unsigned char const2[20];
    
    if (session->manager_random == NULL)
        return PP_ERR;
    if (session->challenge == NULL)
        return PP_ERR;

    if (session->sik != NULL)
        free(session->sik);
    if (session->k1 != NULL)
        free(session->k1);
    if (session->k2 != NULL)
        free(session->k2);
    
    // any message other than RAKP message 4 is okay here ...
    len = pp_authentication_length(session->authentication_cipher, 0);
    if (len == 0)
        return PP_SUC;
    if (len < 0)
        return PP_ERR;

    session->sik_len = len;
    session->sik = malloc(len);
    session->k1 = malloc(len);
    session->k2 = malloc(len);
    
    memcpy(sik_src, session->manager_random, 16);
    memcpy(sik_src+16, session->challenge, 16);

    sik_src[32] = session->req_priv_level | (session->name_only_lookup << 4);

    sik_src[33] = session->username_len;
    memcpy(sik_src+34, session->username, session->username_len);
    sik_src_len = 34 + session->username_len;

    if (PP_SUCCED( pp_cfg_get_binary(&keygen_key, &keygen_length,
                    PP_BMC_CHANNEL_ACCESS_KEYGEN_KEY, session->chan) )) {
        // K_G is available, use it
        int ret = pp_key_calc(session->authentication_cipher,
                                  keygen_key, keygen_length,
                                  sik_src, sik_src_len,
                                  session->sik);
        free(keygen_key);
        if (PP_FAILED(ret)) return PP_ERR;
    } else {
        // no K_G, use password
        if (PP_FAILED(pp_key_calc(session->authentication_cipher,
                                  session->password, session->password_len,
                                  sik_src, sik_src_len,
                                  session->sik))) {
            return PP_ERR;
        }
    }

    // calculate k1, k2, not stored in session yet
    for (i=0; i<20; i++)
        const1[i] = 0x01;
    for (i=0; i<20; i++)
        const2[i] = 0x02;

    // calc more key material
    if (PP_FAILED(pp_key_calc(session->authentication_cipher,
                              session->sik, session->sik_len,
                              const1, 20,
                              session->k1))) {
        return PP_ERR;
    }
    if (PP_FAILED(pp_key_calc(session->authentication_cipher,
                              session->sik, session->sik_len,
                              const2, 20,
                              session->k2))) {
        return PP_ERR;
    }

    //pp_bmc_printbuf("SIK", session->sik, session->sik_len);
    //pp_bmc_printbuf("K1", session->k1, session->sik_len);
    //pp_bmc_printbuf("K2", session->k2, session->sik_len);
    
    return PP_SUC;
}

/**
 * Calculate the authcode for RAKP message2. The authcode size is pp_authcode_length(ciphersuite);
 * @return PP_ERR/PP_SUC
 */
static int calculate_RAKP_message2_authcode(imsg_session_t* session, unsigned char* dest) {
    // construct data to be hmac'd
    unsigned char mac_src[74];  // max length with 16 bytes username
    unsigned char mac_src_len;
    unsigned int *uip;
    
    if (session->manager_random == NULL) {
        pp_bmc_log_warn("[LAN] calculating RAKP message 2 without manager_random failed");
        return PP_ERR;
    }
    if (session->challenge == NULL) {
        pp_bmc_log_warn("[LAN] calculating RAKP message 2 without bmc_random(challenge) failed");
        return PP_ERR;
    }
    
    if (session->authentication_cipher == IPMI20_AUTH_NONE) {
        // nothing to authenticate - authcode not present
        // should not happen (already checked)
        return PP_SUC;
    }
    
    uip = (unsigned int*)(mac_src);
    *uip = cpu_to_le32(session->manager_session_id);
    uip = (unsigned int*)(mac_src + 4);
    *uip = cpu_to_le32(session->session_id);
        
    memcpy(mac_src+8, session->manager_random, 16);
    memcpy(mac_src+24, session->challenge, 16);

    pp_bmc_get_guid(mac_src+40);

    mac_src[56] = session->req_priv_level | (session->name_only_lookup << 4);

    mac_src[57] = session->username_len;
    memcpy(mac_src+58, session->username, session->username_len);
    mac_src_len = 58 + session->username_len;

    if (pp_authentication_calculate(session, IPMI_PAYLOAD_RAKP_MESSAGE2, mac_src, mac_src_len, dest) < 0)
        return PP_ERR;

    return PP_SUC;
}

/**
 * Check if the authcode of RAKP message3 is correct. The authcode size must be
 * at least pp_authcode_length() bytes.
 * @return PP_ERR/PP_SUC
 */
static int check_RAKP_message3_authcode(imsg_session_t* session, unsigned char* authcode) {
    unsigned int* uip;
    unsigned char mac_src[38];
    unsigned char mac_src_len;
    
    int rv;
    
    if (session->authentication_cipher == IPMI20_AUTH_NONE) {
        // nothing to authenticate ...
        return PP_SUC;
    }
    
    // construct authcode src
        
    memcpy(mac_src, session->challenge, 16);
    uip = (unsigned int*)(mac_src+16);
    *uip = cpu_to_le32(session->manager_session_id);
    mac_src[20] = session->req_priv_level | (session->name_only_lookup << 4);

    mac_src[21] = session->username_len;
    memcpy(mac_src + 22, session->username, session->username_len);
    mac_src_len = 22 + session->username_len;
    
    rv = pp_authentication_check(session, IPMI_PAYLOAD_RAKP_MESSAGE3, mac_src, mac_src_len, authcode);

    if (rv < 0)
        return PP_ERR;
    
    return PP_SUC;
}

/**
 * Calculate the auth code for RAKP message4. Dest size must be
 * at least pp_authcode_length() bytes.
 * @return PP_ERR/PP_SUC
 */
static int calculate_RAKP_message4_authcode(imsg_session_t* session, unsigned char* dest) {
    unsigned char mac_src[38];
    unsigned int* uip;

    if (session->authentication_cipher == IPMI20_AUTH_NONE) {
        // nothing to authenticate ...
        return PP_SUC;
    }
    
    memcpy(mac_src, session->manager_random, 16);
    uip = (unsigned int*)(mac_src + 16);
    *uip = cpu_to_le32(session->session_id);

    pp_bmc_get_guid(mac_src+20);

    return pp_authentication_calculate(session, IPMI_PAYLOAD_RAKP_MESSAGE4, mac_src, 36, dest);
}


/*
 * SOL handling: forward data to a handler, in case there is one defined
 */
static int handle_sol_message(unsigned char* data, int len,
			      imsg_session_t* session, lan_addr_t* addr) {
    if (session->sol_handler != NULL) {
	return session->sol_handler(data, len, session, addr);
    } else {
	return PP_ERR;
    }
}
