/**
 * rmcp.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * This file handles
 * <ul>
 * <li> the RMCP layer (Switch by class, send ACKs), 
 * <li> ASF class of RMCP (PINGPONG)
 * <li> IPMI class (decision whether to forward this msg to IPMIv15 or IPMIv20)
 * </ul>
 */

#include <pp/base.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/lan_lowlevel.h>

#include "rmcp.h"
#include "lan_ipmi15.h"
#include "lan_ipmi20.h"

/* Internal prototypes */
static int asf_handle_receive(unsigned char* data, int len, lan_addr_t* addr);
static int ipmi_handle_receive(unsigned char* data, int len, lan_addr_t *addr);


/**
 * Send message to rmcp layer. The data will be wrapped with an rmcp frame with
 * the specified class and sent to the specified lan address.
 */
int pp_send_rmcp_msg(unsigned char* data, int len, lan_addr_t* addr, unsigned char msg_class) {
    unsigned char *buf;
    int rv;
    
    buf = malloc(len+4);
    
    // add rmcp header
    buf[0] = 0x06; // rmcp version
    buf[1] = 0x00;
    buf[2] = 0xFF; // we don't need acks
    buf[3] = msg_class;
    memcpy(buf+4, data, len);
    
    // send msg
    rv = pp_bmc_lan_lowlevel_send(buf, len+4, &(addr->addr), &(addr->addr_len));
    free(buf);
    return rv;
}

/**
 * Internal rmcp function, sends an ack if necessary.
 * *data must  point to begin of the rmcp frame
 */
static void check_send_ack(unsigned char* data, lan_addr_t* addr) {
    static unsigned char buf[4];
    if (data[2] != 0xFF) {
        memcpy(buf, data, 4);
        buf[3] = buf[3] & 0x80;
        //pp_bmc_log_debug("[RMCP] send ack");
        pp_bmc_lan_lowlevel_send(buf, 4, &(addr->addr), &(addr->addr_len));
    } else {
        //pp_bmc_log_debug("[RMCP] no ack required");
    }
}

/**
 * Handle data from the lan.
 * both *data and *addr are pointers to structures on the stack which
 * are not persistent and will stop existing after this function terminates.
 */
int pp_rmcp_handle_receive(unsigned char* data, int len, struct sockaddr* addr, socklen_t* addr_len) {
    unsigned char msg_class;
    int rv;
    lan_addr_t l_addr;
    
    l_addr.addr = *addr;
    l_addr.addr_len = *addr_len;
    
    if (len < 4) {
        // message too short for RMCP header, ignore
        return PP_ERR;
    }
    // pp_bmc_log_debug("[LAN] RMCP received message version=%.2x", data[0]);
    // seq = data[2];
    msg_class = data[3] & 0x1F;
    
    rv = PP_SUC;
    if ((data[3] & 0x80) == 0x80) {
        // ACK, ignore
        if (msg_class == RMCP_CLASS_ASF) {
            // Someone sent us an ACK message. We can savely ignore this
            //pp_bmc_log_debug("[LAN] RMCP received ACK msg");
        }
        // else invalid combination, ignore
        pp_bmc_log_notice("[LAN] RMCP received invalid class ACK");
    } else {
        // real RMCP message
        if (msg_class == RMCP_CLASS_ASF) {
            //pp_bmc_log_debug("[LAN] RMCP received ASF msg");
            check_send_ack(data, &l_addr);
            rv = asf_handle_receive(data+4, len-4, &l_addr);
        } else
        if (msg_class == RMCP_CLASS_IPMI) {
            //pp_bmc_log_debug("[LAN] RMCP received IPMI msg");
            check_send_ack(data, &l_addr);
            rv = ipmi_handle_receive(data+4, len-4, &l_addr);
        } else {
            // else unknown class, ignore
            pp_bmc_log_debug("[LAN] RMCP received msg with unknown class");
        }
    }
    
    return rv;
}


static int asf_handle_receive(unsigned char* data, int len, lan_addr_t* addr) {
    if (len < 8) {
        pp_bmc_log_notice("[LAN] ASF: msg too short");
        return PP_ERR;
    }
    if (!((data[3] == 0xbe) && (data[2] == 0x11) && (data[1] == 0x00) && (data[0] == 0x00))) {
        pp_bmc_log_notice("[LAN] ASF: no IANA msg");
        return PP_ERR;
    }
    if (data[4] != 0x80) {
        pp_bmc_log_notice("[LAN] ASF: unknown message type");
        return PP_ERR;
    }
    
    // build pong response in the same buffer
    // data[0] to data[3] are copied
    data[4] = 0x40;
    // data[5] copied
    data[6] = 0x00;
    data[7] = 0x10;
    data[8] = 0xbe;
    data[9] = 0x11;
    data[10] = 0x00;
    data[11] = 0x00;
    data[12] = 0x00;
    data[13] = 0x00;
    data[14] = 0x00;
    data[15] = 0x00;
    data[16] = 0x81;
    data[17] = 0x00;
    data[18] = 0x00;
    data[19] = 0x00;
    data[20] = 0x00;
    data[21] = 0x00;
    data[22] = 0x00;
    data[23] = 0x00;
    
    //pp_bmc_log_debug("[LAN] ASF: received ping, sending pong");
    return pp_send_rmcp_msg(data, 24, addr, RMCP_CLASS_ASF);
}

/**
 * Handle rmcp messages. Decide if message is v1.5 or v2.0 and call appropriate handler.
 * Both *data and *addr are pointers to structures on the stack which
 * are not persistent and will stop existing after this function returnes.
 */
static int ipmi_handle_receive(unsigned char* data, int len, lan_addr_t *addr) {
    unsigned char auth_type;
    
    if (len < 1) {
        pp_bmc_log_notice("[LAN] IPMIclass message to short - ignored");
        return PP_ERR;
    }
    auth_type = data[0] & 0x0F;
    
    //pp_bmc_log_debug("[LAN] IPMIclass auth_type = %.2x", auth_type);
    
    if ((auth_type == IPMI_AUTH_NONE) ||
        (auth_type == IPMI_AUTH_MD2) ||
        (auth_type == IPMI_AUTH_MD5) ||
        (auth_type == IPMI_AUTH_STRAIGHT))
    {
        addr->ipmi_version = IPMI_LAN_VERSION_15;  // for returning anonymous messages
        return pp_ipmi15_receive(data, len, addr);
    } else
    if (auth_type == IPMI_AUTH_RMCPP) {
        addr->ipmi_version = IPMI_LAN_VERSION_20;  // for returning anonymous messages
        return pp_ipmi20_receive(data, len, addr);
    } else {
        pp_bmc_log_notice("[LAN] IPMIclass unknown ipmi auth_type, msg ignored");
        return PP_ERR;
    }
    
    return PP_SUC; // cannot happen
}
