/**
 * loopi_adapter.c
 *
 * (c) 2005 Peppercon AG, 2005/2/24, tbr@peppecon.de
 *
 * This is a local loop interface that can be used to access the BMC
 * from within the erla. It is based on named pipes and currently
 * offers to pairs of pipes, one for communication within the eric
 * process, one for communication with the snmp daemon.
 * In a sense the loopi adapter is similar to the smi adapter, as
 * plain IPMI-Msgs are exchanged.
 * 
 * Note:
 * Currently we _assume_ that a read from a pipe is atomic. If this
 * assumption is wrong, a read could fail and result in seriously broken
 * messages. 
 * TODO: either confirm that reads are not fragmented or implement some
 * kind of framing (start/stop bytes or length byte)
 */

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/debug.h>

#include <pp/bmc/loopi_adapter.h>
#include <pp/bmc/loopi_adapter_chan.h>

#define MAX_IPMI_MSG_LEN 172     // same value as in openIPMI

/*
 * the global variables
 * --------------------
 */
typedef struct {
    imsg_session_t *sess;     // empty session (to allow session mgmt cmds and user auth)
    unsigned char channel_no; // the channel number assigned to this channel

    /* server (bmc) side files */
    int cmdp_rfd;  // command pipe read fd
    int rspp_wfd;  // response pipe write fd
    int cmdp_rsid; // command pipe selector read id
} loopi_state_t;

static loopi_state_t* loopi_state[PP_BMC_LOOPI_MAX] = { NULL, NULL, NULL };

/*
 * private function prototypes
 * ----------------------------
 */
static int loopi_receive_bmc_msg(imsg_t *imsg);
static int loopi_receive_bmc_msg_internal(imsg_t *imsg, loopi_state_t* state);
static int loopi_receive_loop_msg(const int item_id, const int fd,
				  const short event, void *context);

static int pp_bmc_loopi_channel_init(loopi_state_t* state,
				     const char* cmd_fifo,
				     const char* rsp_fifo, int channel_no);
static void pp_bmc_loopi_channel_cleanup(loopi_state_t* state);


/*
 * public implementation
 * ----------------------
 */
int pp_bmc_loopi_init(int chan) {
    int ret;
    loopi_state_t* lstate;
    
    assert(chan < PP_BMC_LOOPI_MAX);
    assert(loopi_state[chan] == NULL);

    lstate = malloc(sizeof(loopi_state_t));
    ret = pp_bmc_loopi_channel_init(lstate,
				    pp_bmc_loopi_chan[chan].cmd_pipe,
				    pp_bmc_loopi_chan[chan].rsp_pipe,
				    pp_bmc_loopi_chan[chan].channel);
    if (ret == PP_SUC) loopi_state[chan] = lstate;
    else free(lstate);
    
    return ret;
}

static int pp_bmc_loopi_channel_init(loopi_state_t* state,
				     const char* cmd_fifo,
				     const char* rsp_fifo,
				     int channel_no) {
    pp_bmc_log_debug("[loopi_adapter] channel %d init", channel_no);

    if (0 > mkfifo(cmd_fifo, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH
		   | S_IWOTH) && errno != EEXIST) {
        pp_bmc_log_pfatal("[loopi_adapter] cannot create named pipe '%s'",
			  cmd_fifo);
        return PP_ERR;
    }

    if (0 > mkfifo(rsp_fifo, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH
		   | S_IWOTH) && errno != EEXIST) {
        pp_bmc_log_pfatal("[loopi_adapter] cannot create named pipe '%s'",
			  rsp_fifo);
        return PP_ERR;
    }
        
    if (0 > (state->cmdp_rfd = open(cmd_fifo, O_RDWR | O_NONBLOCK)) ||
        0 > (state->rspp_wfd = open(rsp_fifo, O_RDWR | O_NONBLOCK)) ) {
        pp_bmc_log_pfatal("[loopi_adapter] cannot connect named pipes");
        return PP_ERR;
    }

    /* register with bmc, for messages from BMC                           *
     * channel info can be supplied as NULL, as this channel should never *
     * show up anywhere                                                   */
    if (pp_bmc_router_reg_chan(channel_no, loopi_receive_bmc_msg,
			       NULL, NULL, NULL) == NULL) {
        pp_bmc_log_fatal("[loopi_adapter] cannot register channel");
        return PP_ERR;
    }
    pp_bmc_log_debug("[loopi_adapter] channel registered");

    state->channel_no = channel_no;

    /* create empty session (not in the session manager, just plain struct) */
    state->sess = pp_bmc_session_new();
    if (!state->sess) {
        pp_bmc_log_pfatal("[loopi_adapter] pp_bmc_create_session failed");
        return PP_ERR;
    }
    state->sess->max_priv_level = IPMI_PRIV_ADMIN;
    /* start with full access - may be reduced by loopi client */
    state->sess->cur_priv_level = IPMI_PRIV_ADMIN;

    /* register with selector, for messages from the loop channel */
    if (0 > (state->cmdp_rsid = pp_select_add_fd(state->cmdp_rfd,
						      POLLIN,
						      loopi_receive_loop_msg,
						      (void*)state))) {
        pp_bmc_log_pfatal("[loopi_adapter] pp_select_add_fd failed");
        return PP_ERR;
    }

    pp_bmc_log_info("[loopi_adapter] channel %d intialized", channel_no);
    return PP_SUC;
 }

void pp_bmc_loopi_cleanup(int chan)  {
    assert(chan < PP_BMC_LOOPI_MAX);
    assert(loopi_state[chan] != NULL);

    pp_bmc_loopi_channel_cleanup(loopi_state[chan]);
    free(loopi_state[chan]);
    loopi_state[chan] = NULL;
}

static void pp_bmc_loopi_channel_cleanup(loopi_state_t* state) {
    pp_bmc_session_delete(state->sess);
    pp_bmc_router_unreg_chan(state->channel_no);
    pp_select_remove_fd(state->cmdp_rsid);
    close(state->cmdp_rfd);
    close(state->rspp_wfd);
    pp_bmc_log_debug("[loopi_adapter] channel %d cleanup", state->channel_no);
}

/*
 * private implementation
 * ----------------------
 */

/**
 * Called by the selector to indicate that new messages are available on the
 * system interface. The message is read and forwarded to the core.
 * @param context contains the loopi_state_t 'status' of the incoming
 * channel.
 */
static int loopi_receive_loop_msg(UNUSED const int item_id, const int fd,
				  UNUSED const short event,
				  void *context)
{
    char buf[MAX_IPMI_MSG_LEN];
    size_t cnt;
    imsg_t *imsg;
    loopi_state_t* state;

    state = (loopi_state_t*)context;

    /* read msg from command pipe, we do not expect to find   *
     * multiple msg in the queue at the same time             */
    while (0 >= (cnt = read(fd, buf, MAX_IPMI_MSG_LEN))) {
	if (errno == EINTR) continue;
	pp_bmc_log_perror("[loopi_adapter] loopi_receive_loop_msg: "
			  "read failed");
	return PP_ERR;
    }
    if (cnt < 2) {
	pp_bmc_log_warn("[loopi_adapter] loopi_receive_loop_msg: "
			"message too short (cnt < 2)");
	return PP_ERR;
    }

    /* construct an imsg from the received data */
    imsg = pp_bmc_imsg_new(cnt - 2);
    imsg->netfn = buf[0] >> 2;
    imsg->rq_lun = buf[0] & 3;
    imsg->cmd = buf[1];
    memcpy (imsg->data, (&buf[2]), imsg->data_size);
    imsg->chan = state->channel_no;
    imsg->priv_level = state->sess->cur_priv_level;
    imsg->session = state->sess;
    imsg->buf = NULL;
    imsg->buf_size = 0;
    imsg->rs_lun = 0;

    /* only a reference is given away to imsg! */
    state->sess->ref_count++;

    pp_bmc_core_handle_msg(imsg);

    return PP_SUC;
}

/**
 * Called by the bmccore/msgrouter to send messages to the smi
 */
static int  loopi_receive_bmc_msg(imsg_t *imsg) {
    int lchan;
    
    for (lchan = 0; lchan < PP_BMC_LOOPI_MAX; ++lchan) {
	loopi_state_t* state = loopi_state[lchan];
	if (state != NULL && state->channel_no == imsg->chan) {
	    return loopi_receive_bmc_msg_internal(imsg, state);
	}
    }
    
    /* this should not happen, we should have returned yet */
    assert(0);
    pp_bmc_log_warn("[loopi_adapter] receive_bmc_msg() - no such channel(%d)",
		    imsg->chan);
    /* delete, as we don't pass it along */
    pp_bmc_imsg_delete(imsg);

    return PP_ERR;
}

/**
 * Called by the bmccore/msgrouter to send messages to the smi.
 * Internal version with loopi_state_t* resolved.
 */
static int  loopi_receive_bmc_msg_internal(imsg_t *imsg, loopi_state_t*state) {
    char buf[MAX_IPMI_MSG_LEN];

    /* prepare msg */
    buf[0] = (imsg->netfn<<2) | (imsg->rs_lun & 0x03);
    buf[1] = imsg->cmd;
    memcpy(&buf[2], imsg->data, imsg->data_size);

    /* and stuff it into response pipe */
    if (0 > write(state->rspp_wfd, buf, imsg->data_size+2)) {
        pp_bmc_log_perror("[loopi_adapter] loopi_receive_bmc_msg: "
                          "write failed");
    }

    /* delete, as we don't pass it along */
    pp_bmc_imsg_delete(imsg);

    return PP_SUC;
}
