/**
 * ltc4305_i2c_mux.h
 *
 * A concrete implementation for the LTC4305 2-Channel 2-Wire Bus Multiplexer
 * (used as I2C multiplexer)
 * (c) 2006 Peppercon AG, 08/15/2006, mkl@peppercon.de
 */

#include <pp/base.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/tp_i2c_mux.h>
#include <pp/bmc/topo_factory.h>

#include "ltc4305_i2c_mux.h"

#define PP_I2C_MUX_4305_CHANNELS	2

#define REG_CHANNEL_SELECT		0x03

/**
 * pp i2c device constructor
 */
pp_tp_obj_t* pp_i2c_mux_ltc4305_ctor (const char* id, vector_t* args) {
    const char* fn = __FUNCTION__;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_i2c_mux_ltc4305_t* o = NULL;

    pp_tp_i2c_comdev_t* i2cdev;
    u_char i2caddr;
    u_char channel;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d<c>d<c>", &i2cdev, &i2caddr, &channel)
	!= 3) {
        pp_bmc_log_error("%s: '%s' failed: %s (%s)",
                fn, id, strerror(errno), pp_strstream_buf(&err));
	goto bail;
    } else if (channel > PP_I2C_MUX_4305_CHANNELS - 1) {
        pp_bmc_log_error("%s: '%s' failed: invalid channel number: %d (max is %d)",
                fn, id, channel, PP_I2C_MUX_4305_CHANNELS);
        goto bail;
    }

    o = malloc(sizeof(pp_i2c_mux_ltc4305_t));  

    pp_i2c_mux_ltc4305_init(o, id, i2cdev, i2caddr, channel);

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

void pp_i2c_mux_ltc4305_dtor (pp_tp_obj_t* d)
{
    pp_i2c_mux_ltc4305_cleanup(d);
    free(d);
}

static int pp_i2c_mux_ltc4305_switch_channel(pp_tp_i2c_mux_t* d)
{
    u_char r;
    if (PP_FAILED(d->parent->rx_byte_data(d->parent, d->i2caddr, REG_CHANNEL_SELECT, &r))) {
	return PP_ERR;
    }

    return d->parent->tx_byte_data(d->parent, d->i2caddr, REG_CHANNEL_SELECT, r | (1 << (7 - d->channel)) );
}

static int pp_i2c_mux_ltc4305_disable_channels(pp_tp_i2c_mux_t* d)
{
    u_char r;
    if (PP_FAILED(d->parent->rx_byte_data(d->parent, d->i2caddr, REG_CHANNEL_SELECT, &r))) {
	return PP_ERR;
    }

    return d->parent->tx_byte_data(d->parent, d->i2caddr, REG_CHANNEL_SELECT, r & ~(1 << (7 - d->channel)) );
}

void pp_i2c_mux_ltc4305_init(pp_i2c_mux_ltc4305_t* d, const char* id,
			     pp_tp_i2c_comdev_t* parent,
			     u_char i2caddr, u_char channel) {
    /* call super */
    pp_tp_i2c_mux_init((pp_tp_i2c_mux_t*)d, PP_TP_I2C_MUX, id,
		       pp_i2c_mux_ltc4305_dtor,
		       pp_i2c_mux_ltc4305_switch_channel,
		       pp_i2c_mux_ltc4305_disable_channels,
		       parent, i2caddr, channel);
}
