/**
 * max1037_adc_sensor.c
 *
 * MAX1037 4-Channel ADC
 * 
 * (c) 2006 Peppercon AG, mkl@peppecon.de
 */

#include <malloc.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/max1037_adc_sensor.h"

/**
 * private structures
 */

/**
 * The instance of a MAX1037 KIRA sensor.
 */
typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */

    pp_tp_i2c_comdev_t *i2cdev;
    u_char i2caddr;

    u_int channel;
} max1037_adc_sensor_t;

#define MAX1037_ADC_SDR_MTOL  PP_BMC_SDR_M_TOL_2_MTOL(129, 1)

#define MAX1037_CONFIG_BYTE  0x01 /* bit 0   = 1 (single ended input)
				     bit 1-2 = x (channel select)
				     bit 3-4 = 0 (not used for 4-channel adc)
				     bit 5-6 = 0 (scan select bits)
				     bit 7   = 0 (config byte) */
				     

/**
 * private function prototypes
 */
static int  max1037_adc_sensor_init(max1037_adc_sensor_t* this,
				    const char* id,
				    pp_tp_i2c_comdev_t* i2cdev,
				    u_char i2caddr,
				    u_int channel,
				    pp_tp_cond_t* pres_cond);
static void max1037_adc_cleanup(max1037_adc_sensor_t* this);
static void max1037_adc_dtor(pp_tp_obj_t* this);
static ipmi_sdr_header_t* max1037_adc_default_sdr(pp_tp_sensdev_t* s);
static void max1037_adc_update_voltage_reading(pp_sensor_scannable_t* s);


/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_max1037_adc_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[MAX1037 ADC] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    max1037_adc_sensor_t* max1037_adc = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    u_char i2caddr;
    u_int channel;
    pp_tp_cond_t* pres_cond;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d<c>d|o<sc>",
			&i2cdev, &i2caddr, &channel, &pres_cond) != 4) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
            ___F, id, strerror(errno), pp_strstream_buf(&err));
	goto bail;
    } else if (channel > 3) { /* we have 4 channels (0-3) */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, channel);
	goto bail;
    }
    
    max1037_adc = calloc(1, sizeof(max1037_adc_sensor_t));
    if (PP_FAILED(max1037_adc_sensor_init(max1037_adc, id, i2cdev, i2caddr, channel, pres_cond))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(max1037_adc);
	max1037_adc = NULL;
	goto bail;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)max1037_adc;
}

static void
max1037_adc_dtor(pp_tp_obj_t* this)
{
    max1037_adc_cleanup((max1037_adc_sensor_t*)this);
    free(this);
}

/**
 * private functions
 */
static int
max1037_adc_sensor_init(max1037_adc_sensor_t* this,
			const char* id,
			pp_tp_i2c_comdev_t* i2cdev,
			u_char i2caddr,
			u_int channel,
			pp_tp_cond_t* pres_cond)
{  
    /* globals */
    if (PP_SUCCED(pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, id,
					  max1037_adc_dtor,
					  max1037_adc_default_sdr,
					  max1037_adc_update_voltage_reading, pres_cond))) {

	this->i2cdev = (pp_tp_i2c_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)i2cdev);
	this->i2caddr = i2caddr;
	
	this->channel = channel;
	
	/* register for scanning  */
	pp_tp_scan_sensdev_register(&this->base, 0);
    }
    
    return PP_SUC;
}

static ipmi_sdr_header_t*
max1037_adc_default_sdr(pp_tp_sensdev_t* s UNUSED)
{
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_CURRENT,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    IPMI_UNIT_TYPE_AMPS,
				    IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
				    PP_BMC_SDR_M_TOL_2_MTOL(10, 0),
				    PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
				    0, -3,
				    ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
				    ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    
    return (ipmi_sdr_header_t*)sdr;
}

static void
max1037_adc_update_voltage_reading(pp_sensor_scannable_t* s)
{
    max1037_adc_sensor_t* max1037_adc = PP_TP_SCANNABLE_2_OBJ_CAST(s,max1037_adc_sensor_t);
    pp_tp_i2c_comdev_t* i2cdev = max1037_adc->i2cdev;
    u_char i2caddr = max1037_adc->i2caddr;
    u_int channel = max1037_adc->channel;
    u_char value;
    int ret = -2;
    
    if (pp_tp_scan_sensdev_do_scan(&max1037_adc->base)) {
	if (PP_SUCCED(ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {
	    
	    if (PP_FAILED(ret = i2cdev->tx_byte(i2cdev, i2caddr,
						(channel << 1) | MAX1037_CONFIG_BYTE))) {
		goto bail;
	    }
	    if (PP_FAILED(ret = i2cdev->rx_byte(i2cdev, i2caddr, &value))) {
		goto bail;
	    }
	bail:
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	}
	if (PP_SUCCED(ret)) ret = value;	    
    }
    pp_tp_scan_sensdev_update_reading(&max1037_adc->base, ret);
}

static void
max1037_adc_cleanup(max1037_adc_sensor_t* this)
{
    pp_tp_scan_sensdev_unregister(&this->base);
    pp_tp_scan_sensdev_cleanup(&this->base);
}
