/**
 * pp_pca9555_gpio_dev.c
 *
 * Device driver class for the Philips PCA9555 I2C GPIO Extender
 * 
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>

#include "pca9555_gpio_dev.h"

#define GPIO_VALID_MASK	0x0000FFFF

static int
pca9555_get_val(pp_pca955x_gpio_dev_t* d)
{
    int tmpv, value = -1;
    u_char data;

    /* read upper 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  REG_INPUT + 1, &data))) {
	goto bail;
    }
    tmpv = data << 8;

    /* read lower 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  REG_INPUT, &data))) {
	goto bail;
    }
    tmpv |= data;
    value = tmpv & GPIO_VALID_MASK;
    
 bail:
    return value;
}

static int
pca9555_set_reg(pp_pca955x_gpio_dev_t* d, u_char reg, u_long mask, u_long val)
{
    int ret = PP_ERR;
    u_char data;

    mask &= GPIO_VALID_MASK;
    val  &= GPIO_VALID_MASK;
    
    /* set upper 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2 + 1, &data))) {

	goto bail;
    }

    data &= ~(mask >> 8);
    data |= (val & mask) >> 8;
    
    if (PP_FAILED(d->i2cdev->tx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2 + 1, data))) {

	goto bail;
    }

    /* set lower 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2, &data))) {

	goto bail;
    }

    data &= ~(mask & 0xFF);
    data |= val & mask & 0xFF;
    
    if (PP_FAILED(d->i2cdev->tx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2, data))) {

	goto bail;
    }
    
    ret = PP_SUC;
    
 bail:
    return ret;   
}

int pp_pca9555_gpio_dev_init(pp_pca9555_gpio_dev_t* d,
			     pp_tp_i2c_comdev_t* i2cdev,
			     u_char i2caddr, pp_tp_cond_t* pres_cond) {
      
    pp_pca955x_gpio_dev_init((pp_pca955x_gpio_dev_t*)d,
			     PP_PCA955X_GPIO_DEV,
			     "PCA9555 I2C GPIO extender",
			     i2cdev,
			     i2caddr,
			     GPIO_VALID_MASK,
			     pca9555_get_val,
			     pca9555_set_reg,
			     pres_cond);
			     
    return PP_SUC;
}


pp_tp_obj_t* pp_pca9555_gpio_dev_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_pca9555_gpio_dev_t* o = NULL;

    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* pres_cond;
    u_char i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d<c>|o<sc>",
			&i2cdev, &i2caddr, &pres_cond) != 3) {
        pp_bmc_log_error("%s: '%s' failed: %s (%s)", ___F, id,
			 strerror(errno), pp_strstream_buf(&err));
	goto bail;
    }
    
    o = malloc(sizeof(pp_pca9555_gpio_dev_t));
    if (pp_pca9555_gpio_dev_init(o, i2cdev, i2caddr, pres_cond) == PP_ERR) {
        pp_bmc_log_error("%s initialization failed", id);
	free(o); o = NULL;
        goto bail;	
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}
