/**
 * pp_fpga_pwm_dev.c
 *
 * Device driver class for the PWMs in the KIM FPGA
 * 
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 */

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp_pwmtach.h>

#include "drivers/pp_fpga_pwm_dev.h"

static int pwm_set_mode(pp_tp_pwm_act_t* this, pp_tp_pwm_mode_t mode);
static int pwm_get_mode(pp_tp_pwm_act_t* this, pp_tp_pwm_mode_t* mode_ptr);
static int pwm_set_duty_cycle(pp_tp_pwm_act_t* this, u_char duty_cycle);
static int pwm_get_duty_cycle(pp_tp_pwm_act_t* this, u_char* duty_cycle_ptr);

pp_tp_obj_t*
pp_fpga_pwm_dev_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[FpgaPwmDev] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_fpga_pwm_dev_t * fpga_pwm_dev = NULL;
    u_int pwm_num, frequency, invert;

    if (pp_tp_arg_scanf(args, 0, &err, "ddb", &pwm_num, &frequency, &invert) != 3) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    } else if (pwm_num > 7) {
	pp_bmc_log_error(errmsg, ___F, id, "invalid pwm number");
	goto bail;
    }

    fpga_pwm_dev = calloc(1, sizeof(pp_fpga_pwm_dev_t));
    if (PP_FAILED(pp_fpga_pwm_dev_init(fpga_pwm_dev, PP_TP_PWM_ACT, id,
				       pp_fpga_pwm_dev_dtor, pwm_set_mode,
				       pwm_get_mode, pwm_set_duty_cycle,
				       pwm_get_duty_cycle, pwm_num, frequency,
				       invert))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(fpga_pwm_dev);
	fpga_pwm_dev = NULL;
	goto bail;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)fpga_pwm_dev;
}

void
pp_fpga_pwm_dev_dtor(pp_tp_obj_t* this)
{
    pp_fpga_pwm_dev_cleanup((pp_fpga_pwm_dev_t*)this);
    free(this);
}

int
pp_fpga_pwm_dev_init(pp_fpga_pwm_dev_t* this, pp_tp_obj_type_t type, const char* id,
		     pp_tp_obj_dtor_func_t dtor, pp_tp_pwm_act_set_mode_func_t set_mode,
		     pp_tp_pwm_act_get_mode_func_t get_mode,
		     pp_tp_pwm_act_set_duty_cycle_func_t set_duty_cycle,
		     pp_tp_pwm_act_get_duty_cycle_func_t get_duty_cycle,
		     u_int pwm_num, u_int frequency, u_int invert)
{
#define DEV_FILE "/dev/pp_pwmtach-0"
    const char* errmsg = "[FpgaPwmDev] %s: '%s' failed (%s)";

    this->dev_fd = -1;
    if (PP_SUCCED(pp_tp_pwm_act_init(&this->base, type, id, dtor, set_mode,
				     get_mode, set_duty_cycle, get_duty_cycle))) {
	pp_pwm_data_t pwm_data;
	pp_pwm_init_t pwm_init;
	
	this->dev_fd = open(DEV_FILE, O_RDWR);
	if (this->dev_fd < 0) {
	    pp_bmc_log_perror(errmsg, ___F, id, " open(" DEV_FILE ") ");
	    pp_fpga_pwm_dev_cleanup(this);
	    return PP_ERR;
	}

	/* trigger additional pwm init */
	pwm_init.index = pwm_num;
	pwm_init.activate = 1;
	if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_PWM_INIT, &pwm_init) < 0) {
	    pp_bmc_log_perror("%s(): ioctl(PP_PWMTACH_IOCTL_PWM_INIT)", ___F);	    
	    pp_fpga_pwm_dev_cleanup(this);
	    return PP_ERR;
	}

	pwm_data.index = pwm_num;
	pwm_data.frequency = frequency;
	pwm_data.duty_cycle = 100; /* default duty cycle */
	if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_PWM_SET_PARAMS, &pwm_data) < 0) {
	    pp_bmc_log_perror(errmsg, ___F, id, " ioctl(PP_PWMTACH_IOCTL_PWM_SET_PARAMS) ");
	    pp_fpga_pwm_dev_cleanup(this);
	    return PP_ERR;
	}
	
	this->pwm_num = pwm_num;
	this->frequency = frequency;
	this->invert = invert;
	return PP_SUC;
    }

    return PP_ERR;
}

void
pp_fpga_pwm_dev_cleanup(pp_fpga_pwm_dev_t* this)
{
    if (this->dev_fd >= 0) close(this->dev_fd);
    pp_tp_pwm_act_cleanup(&this->base);
}

static int
pwm_set_mode(pp_tp_pwm_act_t* _this, pp_tp_pwm_mode_t mode)
{
    pp_fpga_pwm_dev_t* this = (pp_fpga_pwm_dev_t*)_this;
    const char* errmsg = "[FpgaPwmDev] %s: '%s' failed (%s)";
    int ret = PP_SUC;

    if (mode != PP_TP_PWM_MANUAL) {
	pp_pwm_data_t pwm_data;
	pwm_data.index = this->pwm_num;
	pwm_data.frequency = this->frequency;

	if (mode == PP_TP_PWM_DISABLE) {
	    /* set duty cycle to 0% */
	    pwm_data.duty_cycle = this->invert ? 100 : 0;
	} else if (mode == PP_TP_PWM_AUTO) {
	    /*
	     * not supported: set duty cycle to 99% (100% might not work always
	     * since the output is driven high constantly)
	     */
	    pwm_data.duty_cycle = this->invert ? 1 : 99;
	} else {
	    return PP_ERR;
	}
	if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_PWM_SET_PARAMS, &pwm_data) < 0) {
	    pp_bmc_log_perror(errmsg, ___F, ((pp_tp_obj_t*)this)->id,
			      " ioctl(PP_PWMTACH_IOCTL_PWM_SET_PARAMS) ");
	    return PP_ERR;
	}
    }

    this->mode = mode;

    return ret;
}

static int
pwm_get_mode(pp_tp_pwm_act_t* _this, pp_tp_pwm_mode_t* mode_ptr)
{
    pp_fpga_pwm_dev_t* this = (pp_fpga_pwm_dev_t*)_this;
    assert(mode_ptr);
    *mode_ptr = this->mode;
    return PP_SUC;
}

static int
pwm_set_duty_cycle(pp_tp_pwm_act_t* _this, u_char duty_cycle)
{
    pp_fpga_pwm_dev_t* this = (pp_fpga_pwm_dev_t*)_this;
    const char* errmsg = "[FpgaPwmDev] %s: '%s' failed (%s)";
    pp_pwm_data_t pwm_data;

    pwm_data.index = this->pwm_num;
    pwm_data.frequency = this->frequency;
    pwm_data.duty_cycle = this->invert ? 100 - duty_cycle : duty_cycle;
    if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_PWM_SET_PARAMS, &pwm_data) < 0) {
	pp_bmc_log_perror(errmsg, ___F, ((pp_tp_obj_t*)this)->id,
			  " ioctl(PP_PWMTACH_IOCTL_PWM_SET_PARAMS) ");
	return PP_ERR;
    }

    return PP_SUC;
}

static int
pwm_get_duty_cycle(pp_tp_pwm_act_t* _this, u_char* duty_cycle_ptr)
{
    pp_fpga_pwm_dev_t* this = (pp_fpga_pwm_dev_t*)_this;
    const char* errmsg = "[FpgaPwmDev] %s: '%s' failed (%s)";
    pp_pwm_data_t pwm_data;

    assert(duty_cycle_ptr);

    pwm_data.index = this->pwm_num;
    if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_PWM_GET_PARAMS, &pwm_data) < 0) {
	pp_bmc_log_perror(errmsg, ___F, ((pp_tp_obj_t*)this)->id,
			  " ioctl(PP_PWMTACH_IOCTL_PWM_GET_PARAMS) ");
	return PP_ERR;
    }

    *duty_cycle_ptr = this->invert ? 100 - pwm_data.duty_cycle : pwm_data.duty_cycle;

    return PP_SUC;
}
