/**
 * tp_pp_i2c_dev.c
 *
 * A concrete i2c communication device that uses libpp_base
 * as device driver.
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#include <pp/base.h>
#include <pp/i2c.h>

#include <pp/bmc/topo_factory.h>
#include <pp/bmc/debug.h>

#include "pp_i2c_dev.h"

/**
 * pp i2c device constructor
 */
pp_tp_obj_t* pp_i2c_dev_ctor (const char* id UNUSED, vector_t* args) {
    const char* fn = __FUNCTION__;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_i2c_dev_t* o = NULL;

    char *i2c_devname;

    if (vector_size(args) > 1) {
	pp_bmc_log_error("ERROR: pp_i2c_dev_ctor '%s' failed: "
		"too many arguments", id);
        goto bail;
    }
    
    if (pp_tp_arg_scanf(args, 0, &err, "s", &i2c_devname) != 1) {
	pp_bmc_log_error("%s: '%s' failed: %s (%s)",
                fn, id, strerror(errno), pp_strstream_buf(&err));
	goto bail;
    }
    
    o = malloc(sizeof(pp_i2c_dev_t));

    if (PP_FAILED(pp_i2c_dev_init(o, id, i2c_devname))) {
	free(o); o  = NULL;
	goto bail;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

/**
 * pp i2c device destructor
 */
void pp_i2c_dev_dtor(pp_tp_obj_t* d) {
    pp_i2c_dev_cleanup(d);
    free(d);
}

static int i2c_dev_root_handle(pp_tp_i2c_comdev_t* d) {
    return ((pp_i2c_dev_t*)d)->i2c_handle;
}

static int i2c_dev_pre_com(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr UNUSED)
{
    pp_i2c_lock(((pp_i2c_dev_t*)d)->i2c_handle);
    return PP_SUC;
}


static int i2c_dev_post_com(pp_tp_i2c_comdev_t* d,
			    unsigned char i2caddr UNUSED)
{
    pp_i2c_unlock(((pp_i2c_dev_t*)d)->i2c_handle);
    return PP_SUC;
}

static int i2c_dev_rx_byte(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr,
			   unsigned char* data)
{
    int error;
    u_char ret;
    
    ret = pp_i2c_rx_byte_core(((pp_i2c_dev_t*)d)->i2c_handle,
			      PP_I2C_DONT_LOG_ERR, i2caddr, &error);
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    *data = ret;
    return PP_SUC;
}

static int i2c_dev_tx_byte(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr,
			   unsigned char data) 
{
    int error;
    
    pp_i2c_tx_byte_core(((pp_i2c_dev_t*)d)->i2c_handle,
			PP_I2C_DONT_LOG_ERR, i2caddr, data, &error);
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    return PP_SUC;
}

static int i2c_dev_rx_byte_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned char* data)
{
    int error;
    u_char ret;
    
    ret = pp_i2c_rx_byte_data_core(((pp_i2c_dev_t*)d)->i2c_handle,
				   PP_I2C_DONT_LOG_ERR, i2caddr, reg, &error);
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    *data = ret;
    return PP_SUC;
}

static int i2c_dev_rx_word_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned short* data)
{
    int error;
    u_short ret;

    ret = pp_i2c_rx_word_data_core(((pp_i2c_dev_t*)d)->i2c_handle,
				   PP_I2C_DONT_LOG_ERR, i2caddr, reg, &error);	
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    *data = ret;
    return PP_SUC;
}

static int i2c_dev_tx_byte_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned char data) 
{
    int error;
    
    pp_i2c_tx_byte_data_core(((pp_i2c_dev_t*)d)->i2c_handle,
			     PP_I2C_DONT_LOG_ERR, i2caddr, reg, data, &error);
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    return PP_SUC;
}

static int i2c_dev_tx_word_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned short data) 
{
    int error;
    
    pp_i2c_tx_word_data_core(((pp_i2c_dev_t*)d)->i2c_handle,
			     PP_I2C_DONT_LOG_ERR, i2caddr, reg, data, &error);
    
    if (error != PP_I2C_NO_ERROR) {
	errno = ENODATA;
	return PP_ERR;
    }
    return PP_SUC;
}
int pp_i2c_dev_init(pp_i2c_dev_t* d, const char* id, char* i2c_devname) {
    int error;
    pp_bmc_log_debug("[PP_I2C_DEV] init %s", i2c_devname);
    
    /* open the i2c device */
    d->i2c_handle = pp_i2c_open(i2c_devname, &error);
    if (error != PP_I2C_NO_ERROR) {

	return PP_ERR;
    }

    /* call super */
    pp_tp_i2c_comdev_init((pp_tp_i2c_comdev_t*)d, PP_I2C_DEV, id,
                          pp_i2c_dev_dtor,
			  i2c_dev_root_handle,
                          i2c_dev_pre_com,
                          i2c_dev_post_com,
                          i2c_dev_rx_byte,
                          i2c_dev_tx_byte,
                          i2c_dev_rx_byte_data,
                          i2c_dev_tx_byte_data,
                          i2c_dev_rx_word_data,
                          i2c_dev_tx_word_data);

    return PP_SUC;
}

void pp_i2c_dev_cleanup(pp_tp_obj_t* d) {
    pp_bmc_log_debug("[PP_I2C_DEV] cleanup");
    
    pp_tp_i2c_comdev_cleanup((pp_tp_i2c_comdev_t*)d);

    /* close the i2c device */
    pp_i2c_close(((pp_i2c_dev_t*)d)->i2c_handle);
}
