/**
 * pp_tach_sens.c
 *  
 * Scannable Tacho Sensor reads values from tacho device.
 * Triggered either through scanner thread or an instance
 * of MuxedScanner.
 *
 * (c) 2006 Peppercon AG, <mkl@peppercon.de>
 */ 
    
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
    
#include <pp/base.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/sensor.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_scan_sensdev.h>
    
#include <pp_pwmtach.h>

#include <pp/bmc/ipmi_sdr.h>
#include <drivers/pp_tach_sens.h>

#include <pp/bmc/tp_muxed_scanner.h>

/**
 *  The template of a Peppercon tach sensor.
 */
typedef struct {
    /* the sensor type */
    int sensor_type;
    /* sdr configuration */
    u_short sdr_mtol;
    u_char sdr_b;
    char sdr_bexp;
    char sdr_rexp;
} sensor_config_t;

typedef struct {
    pp_tp_scan_sensdev_t base;
    pp_sensor_scannable_t scannable;
    u_int tach_num;
    u_int pulses_per_rev;
    pp_tp_cond_t* pres_cond;
    int dev_fd;
    pp_tp_muxed_scanner_t *muxed_scanner;
    u_int bank_num;
} pp_tach_sens_t;

/**
 * Initialize ADM1031 sensor templates.
 */

static sensor_config_t sensor_config[] = {
    { /* 0 - tach 0 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 1 - tach 1 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 2 - tach 2 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 3 - tach 3 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 4 - tach 4 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 5 - tach 5 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 6 - tach 6 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 7 - tach 7 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(100, 1),
	.sdr_b = 0,
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }
};

static int pp_tach_sens_init(pp_tach_sens_t* this,
			     const char* id, pp_tp_obj_dtor_func_t dtor,
			     pp_tp_sensdev_default_sdr_func_t default_sdr,
			     u_int tach_num, u_int pulses_per_rev,
			     pp_tp_cond_t* pres_cond, pp_tp_muxed_scanner_t* muxed_scanner, u_int bank_num);
static ipmi_sdr_header_t* pp_tach_sens_default_sdr(pp_tp_sensdev_t *s);
static void pp_tach_sens_update(pp_sensor_scannable_t* s);
static void pp_tach_sens_dtor(pp_tp_obj_t* this);
static void pp_tach_sens_cleanup(pp_tach_sens_t* this);

pp_tp_obj_t*
pp_tach_sens_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[TachSens] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_muxed_scanner_t *muxed_scanner;
    pp_tach_sens_t *tach_sens = NULL;
    sensor_config_t *sc;
    pp_tp_cond_t* pres_cond;
    u_int tach_num, pulses_per_rev, bank_num;

    if (pp_tp_arg_scanf(args, 0, &err, "dd|o<sc>o<m>d", &tach_num, &pulses_per_rev,
			&pres_cond, &muxed_scanner, &bank_num ) != 5){
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    } else if (tach_num > 7) {
	pp_bmc_log_error(errmsg, ___F, id, "invalid tach number");
	goto bail;
    } else if (pulses_per_rev > 8) {
	pp_bmc_log_error(errmsg, ___F, id, "pulses/rev invalid");
	goto bail;
    }
    
    sc = &sensor_config[tach_num];
    
    tach_sens = malloc(sizeof(pp_tach_sens_t));
    
    if (PP_FAILED(pp_tach_sens_init(tach_sens, id,
				     pp_tach_sens_dtor,
				     pp_tach_sens_default_sdr,
				     tach_num, pulses_per_rev, pres_cond,
				     muxed_scanner, bank_num))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(tach_sens);
	tach_sens = NULL;
	goto bail;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)tach_sens;
}

static void pp_tach_sens_dtor(pp_tp_obj_t* this)
{
    pp_tach_sens_cleanup((pp_tach_sens_t*)this);
    free(this);
}

static int pp_tach_sens_init(pp_tach_sens_t* this, 
			     const char* id, pp_tp_obj_dtor_func_t dtor,
			     pp_tp_sensdev_default_sdr_func_t default_sdr,
			     u_int tach_num, u_int pulses_per_rev,
			     pp_tp_cond_t* pres_cond, pp_tp_muxed_scanner_t* muxed_scanner, u_int bank_num) {
#define DEV_FILE "/dev/pp_pwmtach-0"
    const char* errmsg = "[TachSens] %s: '%s' failed (%s)";
    
    this->dev_fd = -1;
    
    if (PP_SUCCED(pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, 
					  id, dtor, default_sdr,
					  pp_tach_sens_update, NULL))) {
	pp_tach_init_t tach_init;
	
	this->dev_fd = open(DEV_FILE, O_RDWR);
	if (this->dev_fd < 0) {
	    pp_bmc_log_perror(errmsg, ___F, id, " open(" DEV_FILE ") ");
	    pp_tp_scan_sensdev_cleanup(&this->base);
	    return PP_ERR;
	}

	/* trigger additional tacho init */
	tach_init.index = tach_num;
	tach_init.activate = 1;
	if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_TACH_INIT, &tach_init) < 0) {
	    pp_bmc_log_perror("%s(): ioctl(PP_PWMTACH_IOCTL_TACH_INIT)", ___F);
	    if (this->dev_fd >= 0) close(this->dev_fd);
	    pp_tp_scan_sensdev_cleanup(&this->base);
	    return PP_ERR;
	}
	
	this->tach_num = tach_num;
	this->pulses_per_rev = pulses_per_rev;
	this->pres_cond = pres_cond == NULL ?
	    NULL : pp_tp_cond_duplicate(pres_cond);
	
	this->bank_num = bank_num;
        
	if ( muxed_scanner ) {
	    /* register to muxed scanner */
	    this->muxed_scanner = pp_tp_muxed_scanner_duplicate(muxed_scanner);

	    if (PP_FAILED(pp_tp_muxed_scanner_register(&this->base,
						       this->muxed_scanner,
						       this->bank_num))) {	
		return PP_ERR;
	    }
	} else {	
	    /* register to scanner thread */
	    pp_tp_scan_sensdev_register(&this->base, 0);
	    this->muxed_scanner = NULL;
	}
	    
	return PP_SUC;
    }
    
    return PP_ERR;
}

static void pp_tach_sens_cleanup(pp_tach_sens_t* this)
{
    if ( !this->muxed_scanner ) {
	pp_tp_scan_sensdev_unregister(&this->base);
    } else {
	pp_tp_muxed_scanner_release(this->muxed_scanner);
    }
    if (this->dev_fd >= 0) close(this->dev_fd);
    if (this->pres_cond) pp_tp_cond_release(this->pres_cond);
    pp_tp_scan_sensdev_cleanup(&this->base);
}

static ipmi_sdr_header_t*
pp_tach_sens_default_sdr(pp_tp_sensdev_t *s) {
    pp_tach_sens_t *this = (pp_tach_sens_t*)s;
    sensor_config_t *sc = &sensor_config[this->tach_num];
    ipmi_sdr_full_sensor_t* sdr;
    
    sdr = ipmi_sdr_full_part_create(sc->sensor_type,
			       IPMI_EVENT_READING_TYPE_THRESHOLD,
			       IPMI_UNIT_TYPE_RPM,
			       IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
			       sc->sdr_mtol, sc->sdr_b,
			       sc->sdr_bexp, sc->sdr_rexp,
			       ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
			       ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    return (ipmi_sdr_header_t*)sdr;
}

static void
pp_tach_sens_update(pp_sensor_scannable_t* s) {

    pp_tach_sens_t * this =
	PP_TP_SCANNABLE_2_OBJ_CAST(s, pp_tach_sens_t);
    pp_tach_data_t tach_data;
    int reading = -1;

    if (this->pres_cond != NULL && !pp_bmc_tp_cond_is_true(this->pres_cond)) {
	goto bail;
    }
    
    tach_data.index = this->tach_num;
    tach_data.pulses_per_rev = this->pulses_per_rev;
    if (ioctl(this->dev_fd, PP_PWMTACH_IOCTL_TACH_GET_RPM, &tach_data) < 0) {
	pp_bmc_log_perror("%s(): ioctl(PP_PWMTACH_IOCTL_TACH_GET_RPM)", ___F);
    } else {
	reading = (((tach_data.revs_per_minute + 50) / 100) & 0xFF);
    }
    
 bail:
    pp_tp_scan_sensdev_update_reading(&this->base, reading);
}
