/**
 * rpc_beep_ctrl.c
 *
 * implements RPC beeper control
 * 
 * (c) 2006 Peppercon AG, 2006/10/26 tbr@raritan.com
 */

#include <pp/vector.h>
#include <pp/bmc/tp_ctrl.h>
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_ipmi_sens.h>

#include "drivers/rpc_beep_ctrl.h"

typedef struct beep_ctrl_s {
    pp_tp_ctrl_t base;
    pp_tp_gpio_act_t*  beeper;
    pp_tp_ipmi_sens_t* temp;
    vector_t cbks;
    pp_tp_ipmi_sens_subscriber_t isscr;
    pp_tp_sensdev_subscriber_t ssscr;
} beep_ctrl_t;

static void beep_ctrl_dtor(pp_tp_obj_t* this);
static void beep_ctrl_isscr_recv_reading(pp_tp_ipmi_sens_subscriber_t* subscriber,
					 pp_tp_ipmi_sens_t* source, int reading);
static void beep_ctrl_ssscr_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					 pp_tp_sensdev_t* source, int reading);
inline static void beep_ctrl_set_state(beep_ctrl_t* this);
static int beep_ctrl_alarm_is_asserted(beep_ctrl_t* this);

pp_tp_obj_t* pp_rpc_beep_ctrl_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[RPCBeepCtrl] %s c'tor: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    beep_ctrl_t* this = NULL;
    pp_tp_gpio_act_t* beeper;
    pp_tp_ipmi_sens_t * temp;
    pp_tp_sensdev_t* cbk;
    unsigned int i, cbk_cnt;

    if (pp_tp_arg_scanf(args, 0, &err, "o<ag>o<p>", &beeper, &temp) != 2) {
        pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(beep_ctrl_t));
	pp_tp_ctrl_init(&this->base, PP_TP_CTRL, id, beep_ctrl_dtor);

	this->beeper = pp_tp_gpio_act_duplicate(beeper);
	this->temp = pp_tp_ipmi_sens_duplicate(temp);
	this->isscr.recv_reading = beep_ctrl_isscr_recv_reading;
	pp_tp_ipmi_sens_subscribe(temp, &this->isscr);

	/* collect circuit breaker sensors and subscribe */
	this->ssscr.recv_reading = beep_ctrl_ssscr_recv_reading;
	vector_new(&this->cbks, 6,
		   (vector_elem_del_func_simple)pp_tp_obj_release);
	cbk_cnt = (vector_size(args) - 2);
	for (i = 0; i < cbk_cnt; ++i) {
	    if (pp_tp_arg_scanf(args, i + 2, &err, "o<s>", &cbk) != 1) {
		pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
		beep_ctrl_dtor((pp_tp_obj_t*)this);
		this = NULL;
		break;
	    }
	    vector_add(&this->cbks, pp_tp_sensdev_duplicate(cbk));
	    pp_bmc_tp_sensdev_subscribe(cbk, &this->ssscr);
	}
	if (this != NULL) { pp_bmc_log_debug(errmsg, id, "initialized"); }
    }	
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

void beep_ctrl_dtor(pp_tp_obj_t* o) {
    beep_ctrl_t* this = (beep_ctrl_t*)o;
    unsigned int i;
    assert(this != NULL);
    pp_bmc_log_debug("[RPCBeepCtrl] %s d'tor", pp_tp_obj_to_string(o));
    for (i = 0; i < vector_size(&this->cbks); ++i) {
	pp_bmc_tp_sensdev_unsubscribe(vector_get(&this->cbks, i),
				      &this->ssscr);
    }
    vector_delete(&this->cbks);
    pp_tp_ipmi_sens_unsubscribe(this->temp, &this->isscr);
    pp_tp_ipmi_sens_release(this->temp);
    pp_tp_gpio_act_release(this->beeper);
    pp_tp_ctrl_cleanup(&this->base);
    free(this);
}

static void beep_ctrl_isscr_recv_reading(pp_tp_ipmi_sens_subscriber_t* s,
					 pp_tp_ipmi_sens_t* source UNUSED, int reading UNUSED) {
    beep_ctrl_t* this = PP_TP_INTF_2_OBJ_CAST(s, beep_ctrl_t, isscr);
    beep_ctrl_set_state(this);
}
    

static void beep_ctrl_ssscr_recv_reading(pp_tp_sensdev_subscriber_t* s,
					 pp_tp_sensdev_t* source UNUSED, int reading UNUSED) {
    beep_ctrl_t* this = PP_TP_INTF_2_OBJ_CAST(s, beep_ctrl_t, ssscr);
    beep_ctrl_set_state(this);
}
    
inline static void beep_ctrl_set_state(beep_ctrl_t* this) {
    int is_alarm = beep_ctrl_alarm_is_asserted(this);
    pp_bmc_log_debug("[RPCBeepCtrl]: alarm %s", is_alarm ? "ON" : "OFF");
    pp_tp_gpio_act_set_gpio(this->beeper, is_alarm);
}

/* returns true, if a single alarm condition is detected (logical OR) */
static int beep_ctrl_alarm_is_asserted(beep_ctrl_t* this) {
    ipmi_sensor_get_event_status_rs_t temp_state;
    unsigned int i, cbcnt;
    pp_tp_sensdev_t* cbk;
    
    /* check state of temp thresholds */
    pp_tp_ipmi_sens_get_event_status(this->temp, &temp_state);

    /* mask in upper thresholds going high
       and     lower thresholds going low   */
    if (temp_state.assertion_events_le16 & 0x0a95) {
	return 1;
    }

    /* check circuit breakers if broken,
       invalid reading will not assert alarm  */
    cbcnt = vector_size(&this->cbks);
    for (i = 0; i < cbcnt; ++i) {
	cbk = vector_get(&this->cbks, i);
	if (pp_tp_sensdev_get_reading(cbk) == 1) {
	    return 1;
	}
    }
    return 0;
}
