/**
 * w83791d_sensor.c
 *
 * implements all W83791D sensors
 * 
 * (c) 2005 Peppercon AG, 02/22/2005, thre@peppecon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/w83791d.h"
#include "drivers/w83791d_sensor.h"


/*
 * private structures
 ************************/

/*
 * W83791D sensor's structure
 */
typedef enum {
    /* sensor directly measures voltage */
    W83791D_VOLTAGE_TYPE_DIRECT,
    /* sensor measures positive voltage with voltage correction */
    W83791D_VOLTAGE_TYPE_POS_CORRECTED,
    /* sensor measures negative voltage with voltage correction */
    W83791D_VOLTAGE_TYPE_NEG_CORRECTED,
} voltage_type_t;

typedef struct sensor_config_s {
    /* the sensor type */
    unsigned char sensor_type;
    unsigned char unit_type;
    unsigned char analog_type;
    /* update function */
    void (*update)(pp_sensor_scannable_t*);
    /* configuration */
    union {
        struct {
            unsigned char reg;
            voltage_type_t type;
            u_int correction_r1;
            u_int correction_r2;
            u_int correction_factor;
        } volt;
        struct {
            unsigned char divisor_low_reg;
            unsigned char divisor_high_reg;
            unsigned char divisor_low_shift;
            unsigned char divisor_high_shift;
            unsigned char reg;
        } fan;
        struct {
            unsigned char reg;
        } temp;
    } config; 
    /* sdr configuration */
    unsigned short sdr_mtol;
    unsigned short sdr_bacc;
    char sdr_bexp;
    char sdr_rexp;
} sensor_config_t;

typedef struct {
    /* the basic sensor description, must be first */
    pp_tp_scan_sensdev_t base;
    
    pp_tp_i2c_chip_t* chip;

    /* sensor configuration, pointer to statically allocated memory */
    sensor_config_t* cfg;

} w83791d_sensor_t;

/*
 * private function prototypes, used by all W83791D sensors
 *************************************************************/

static w83791d_sensor_t* w83791d_sensor_create(/* globals */
					      const char* id,
                                              pp_tp_i2c_chip_t* chip,
                                              /* the sensor idx */
                                              int sensor_idx,
					      pp_tp_cond_t* scan);
//static char* w83791d_to_string(pp_tp_sensdev_t* sensor);
static ipmi_sdr_header_t* w83791d_default_sdr(pp_tp_sensdev_t* s);
static void w83791d_update_voltage_reading(pp_sensor_scannable_t* s);
static void w83791d_update_tacho_reading(pp_sensor_scannable_t* s);
static void w83791d_update_temp_reading(pp_sensor_scannable_t* s);
static void w83791d_destroy(w83791d_sensor_t* sensor);

/**
 * The actual configurations of the W83791D sensors
 *****************************************************************/
static sensor_config_t sensor_config[] = {
    { /* 0 - Vcc */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x20, .type = W83791D_VOLTAGE_TYPE_DIRECT,
                              .correction_r1 = 0, .correction_r2 = 0,
			      .correction_factor = 0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(16, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 1 - Vin R0 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x21, .type = W83791D_VOLTAGE_TYPE_DIRECT,
                              .correction_r1 = 0, .correction_r2 = 0,
			      .correction_factor = 0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(16, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 2 - 3.3V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x22, .type = W83791D_VOLTAGE_TYPE_DIRECT,
                              .correction_r1 = 0, .correction_r2 = 0,
			      .correction_factor = 0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(16, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 3 - Vin R1 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x23,
			      .type = W83791D_VOLTAGE_TYPE_POS_CORRECTED,
                              .correction_r1 = 34, .correction_r2 = 50,
			      .correction_factor = 25, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(25, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 4 - 12V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x24,
			      .type = W83791D_VOLTAGE_TYPE_POS_CORRECTED,
                              .correction_r1 = 28, .correction_r2 = 10,
			      .correction_factor = 100, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(10, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -2,
    }, { /* 5 - -12V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x25,
			      .type = W83791D_VOLTAGE_TYPE_NEG_CORRECTED,
                              .correction_r1 = 56, .correction_r2 = 232,
			      .correction_factor = 100, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(-5, 2), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -2,
    }, { /* 6 - -5V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x26,
			      .type = W83791D_VOLTAGE_TYPE_NEG_CORRECTED,
                              .correction_r1 = 56, .correction_r2 = 120,
			      .correction_factor = 50, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(-25, 2), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 7 - 5Vsb */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0x23,
			      .type = W83791D_VOLTAGE_TYPE_POS_CORRECTED,
                              .correction_r1 = 34, .correction_r2 = 50,
			      .correction_factor = 25, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(25, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 8 - Vbat */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = w83791d_update_voltage_reading,
        .config = { .volt = { .reg = 0xb1, .type = W83791D_VOLTAGE_TYPE_DIRECT,
                              .correction_r1 = 0, .correction_r2 = 0,
			      .correction_factor = 0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(16, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 9 - fan speed 1 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
        .update = w83791d_update_tacho_reading,
        .config = { .fan = { .divisor_low_reg = 0x47, .divisor_high_reg = 0x5d,
                             .divisor_low_shift = 4, .divisor_high_shift = 5,
			     .reg = 0x28, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 10 - fan speed 2 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
        .update = w83791d_update_tacho_reading,
        .config = { .fan = { .divisor_low_reg = 0x47, .divisor_high_reg = 0x5d,
                             .divisor_low_shift = 6, .divisor_high_shift = 6,
			     .reg = 0x29, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 11 - fan speed 3 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
        .update = w83791d_update_tacho_reading,
        .config = { .fan = { .divisor_low_reg = 0x4b, .divisor_high_reg = 0x5d,
                             .divisor_low_shift = 6, .divisor_high_shift = 7,
			     .reg = 0x2a, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 12 - fan speed 4 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
        .update = w83791d_update_tacho_reading,
        /* the the values are guessed, the spec doesn't tell us: */
        .config = { .fan = { .divisor_low_reg = 0x5c, .divisor_high_reg = 0x5c,
                             .divisor_low_shift = 0, .divisor_high_shift = 2,
			     .reg = 0xba, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 13 - fan speed 5 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
        .update = w83791d_update_tacho_reading,
        /* the the values are guessed, the spec doesn't tell us: */
        .config = { .fan = { .divisor_low_reg = 0x5c, .divisor_high_reg = 0x5c,
                             .divisor_low_shift = 3, .divisor_high_shift = 5,
			     .reg = 0xbb, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 14 - temperature 1 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = w83791d_update_temp_reading,
        .config = { .temp = { .reg = 0x27, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 15 - temperature 2 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = w83791d_update_temp_reading,
        .config = { .temp = { .reg = 0xc0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 16 - temperature 3 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = w83791d_update_temp_reading,
        .config = { .temp = { .reg = 0xc8, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }
};

/*
 * implementations
 * -------------------------------
 */

pp_tp_obj_t* pp_sensor_w83791d_sens_ctor(const char* id, vector_t* args) {
    const char* fn = __FUNCTION__;
    const char* errmsg = "%s: '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    w83791d_sensor_t* o = NULL;
    pp_tp_i2c_chip_t* chip;
    pp_tp_cond_t* scan;
    int snum;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d|o<sc>",
			&chip, &snum, &scan) != 3) {
	pp_bmc_log_perror(errmsg, fn, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, W83791D_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, fn, id, "incorrect chip model");
    } else if (snum > PP_SENSOR_W83791D_MAX_SENSOR_NUM) {
	pp_bmc_log_perror(errmsg, fn, id, "invalid sensor number");
    } else {
        o = w83791d_sensor_create(id, chip, snum, scan);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

/*
 * private functions
 ********************************************************************/

static w83791d_sensor_t* w83791d_sensor_create(/* globals */
					      const char* id,
					      pp_tp_i2c_chip_t* chip,
                                              /* the sensor idx */
                                              int sensor_idx,
					      pp_tp_cond_t* scan)
{
    sensor_config_t *cfg = &sensor_config[sensor_idx];
    w83791d_sensor_t* wbs = malloc(sizeof(w83791d_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&wbs->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))w83791d_destroy,
			    w83791d_default_sdr, cfg->update, scan);
    wbs->chip = pp_tp_i2c_chip_duplicate(chip);
    wbs->cfg = cfg;

    /* register for scanning  */
    pp_tp_scan_sensdev_register(&wbs->base, 0);

    return (w83791d_sensor_t*)wbs;
}

static ipmi_sdr_header_t* w83791d_default_sdr(pp_tp_sensdev_t* s) {
    w83791d_sensor_t* this = (w83791d_sensor_t*)s;
    sensor_config_t* cfg = this->cfg;
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(cfg->sensor_type,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    cfg->unit_type, cfg->analog_type,
				    cfg->sdr_mtol, cfg->sdr_bacc,
				    cfg->sdr_bexp, cfg->sdr_rexp,
	    	                    ipmi_sdr_max_by_form(cfg->analog_type),
   		                    ipmi_sdr_min_by_form(cfg->analog_type));
    
    return (ipmi_sdr_header_t*)sdr;
}

/* The w83791d's ADC can only handle voltages up to 4.096V because it has
   a resolution of 16mV. Larger voltages are handled with a voltage divider.
   We calculate the corrected value here. */
static u_char get_corrected_positive_voltage(w83791d_sensor_t* wbs,
					     u_char value) {
    u_int value_mv = value * 16;	// millivolts measured
    u_int really_mv;
    sensor_config_t* cfg = wbs->cfg;
    
    if (cfg->config.volt.correction_r2 == 0) return 0;
    
    // the real value in mV
    really_mv = (value_mv * (cfg->config.volt.correction_r1 +
			     cfg->config.volt.correction_r2))
	/ cfg->config.volt.correction_r2;
    
    return (u_char) (really_mv / cfg->config.volt.correction_factor);
}

static char get_corrected_negative_voltage(w83791d_sensor_t* wbs,
					   u_char value) {
    sensor_config_t* cfg = wbs->cfg;
    u_int value_mv = value * 16;	// millivolts measured
    int really_mv;
    double beta;
    
    beta = (double) cfg->config.volt.correction_r2
	/ (double) (cfg->config.volt.correction_r1
		    + cfg->config.volt.correction_r2);
    
    // the real value in mV
    really_mv = (int) ((value_mv - 3600 * beta) / (1 - beta));

    return (really_mv / (int) cfg->config.volt.correction_factor);
}

static void w83791d_update_voltage_reading(pp_sensor_scannable_t* s) {
    w83791d_sensor_t* wbs = PP_TP_SCANNABLE_2_OBJ_CAST(s, w83791d_sensor_t);
    sensor_config_t* cfg = wbs->cfg;
    int ret = -2;
    pp_tp_i2c_chip_t* chip = wbs->chip;
    u_char value;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&wbs->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->config.volt.reg,
					      &value);
	    pp_tp_i2c_chip_post_com(chip);
	    
	    if (ret == PP_SUC) {
		switch (cfg->config.volt.type) {
		  case W83791D_VOLTAGE_TYPE_POS_CORRECTED:
		      ret = get_corrected_positive_voltage(wbs, value);
		      break;
		  case W83791D_VOLTAGE_TYPE_NEG_CORRECTED:
		      ret = get_corrected_negative_voltage(wbs, value);
		      break;
		  case W83791D_VOLTAGE_TYPE_DIRECT:
		      /* fall through */
		  default:
		      ret = value;
		}
	    }
	}
    }
    pp_tp_scan_sensdev_update_reading(&wbs->base, ret);
}

static void w83791d_update_tacho_reading(pp_sensor_scannable_t* s) {
    w83791d_sensor_t* wbs = PP_TP_SCANNABLE_2_OBJ_CAST(s, w83791d_sensor_t);
    sensor_config_t* cfg = wbs->cfg;
    int ret = PP_ERR;
    pp_tp_i2c_chip_t* chip = wbs->chip;
    u_char div_low, div_high, divs;
    u_char value;
    u_int pulses;
    u_int pulses_per_minute;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&wbs->base) &&
	(PP_SUC == pp_tp_i2c_chip_pre_com(chip))) {
        // read the configuration and fan registers
        ret = pp_tp_i2c_chip_rx_byte_data(chip,
					  cfg->config.fan.divisor_low_reg,
					  &div_low);
        if (ret == PP_SUC) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip,
					      cfg->config.fan.divisor_high_reg,
					      &div_high);
	}
        if (ret == PP_SUC) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->config.fan.reg,
					      &value);
	}
        pp_tp_i2c_chip_post_com(chip);

        if (ret == PP_SUC) {

            // calculate the divisor
            divs = ((div_low >> cfg->config.fan.divisor_low_shift) & 3) |
		(((div_high >> cfg->config.fan.divisor_high_shift) & 1) << 2);
            pulses = 1 << divs;

            if (value == 0xff || value == 0) {
                // fan is not moving, set value to 0 and return
                ret = 0;
            } else {
                // the counter tells us the number of clock cycles (90 kHz)
		// for one fan revolution
                pulses_per_minute = 1350000 / (pulses * value);

                // convert the count into a byte value
                ret = min((u_int)255, pulses_per_minute / 50);
            }

        }
    }
    pp_tp_scan_sensdev_update_reading(&wbs->base, ret);
}

static void w83791d_update_temp_reading(pp_sensor_scannable_t* s) {
    w83791d_sensor_t* wbs = PP_TP_SCANNABLE_2_OBJ_CAST(s, w83791d_sensor_t);
    sensor_config_t* cfg = wbs->cfg;
    int ret = PP_ERR;
    pp_tp_i2c_chip_t* chip = wbs->chip;
    u_char value;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&wbs->base)
	&& (PP_SUC == pp_tp_i2c_chip_pre_com(chip))) {
        ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->config.temp.reg,
					  &value);
        pp_tp_i2c_chip_post_com(chip);

        if (ret == PP_SUC) {
            // TODO: check for correct values --> 0x80 means no value
            if (value == 0x80) {
                errno = ENODATA;
                ret = PP_ERR;
            } else {
                ret = value;
            }
        }
    }
    pp_tp_scan_sensdev_update_reading(&wbs->base, ret);
}

static void w83791d_destroy(w83791d_sensor_t* sensor) {
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_chip_release(sensor->chip);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}
