/**
 * w83792d_clr_intrusion.c
 *
 * implements W83792D clear intrusion device
 * TODO: could also even implement the actor itself!
 * 
 * (c) 2006 Peppercon AG, 2006/04/30, rgue@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_gpio_dev.h>
#include <pp/bmc/tp_cond.h>

#include "drivers/w83792d.h"
#include "drivers/w83792d_clr_intrusion.h"

/*
 * actor
 **************************/

typedef struct w83792d_clr_intrusion_s {
    pp_tp_gpio_dev_t base;
    pp_tp_i2c_chip_t* chip;
    pp_tp_cond_t* acc_cond;
} w83792d_clr_intrusion_t;

static void w83792d_clr_intrusion_destroy(pp_tp_obj_t* o);
static int w83792d_clr_intrusion_get_val(pp_tp_gpio_dev_t* d, unsigned long mask);
static int w83792d_clr_intrusion_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
					 unsigned long val, unsigned long tristate);

pp_tp_obj_t* pp_sensor_w83792d_clr_intrusion_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    w83792d_clr_intrusion_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;
    pp_tp_cond_t* acc_cond;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>|o<sc>", &chip, &acc_cond) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, W83792D_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else {
	this = malloc(sizeof(w83792d_clr_intrusion_t));
	pp_tp_gpio_dev_init(&this->base, PP_FPGA_GPIO_DEV,
			    id, 0x00000001 /* just one bit */,
			    w83792d_clr_intrusion_destroy,
			    NULL, /* no init necessary */
			    w83792d_clr_intrusion_get_val, w83792d_clr_intrusion_set_val,
			    NULL);
	this->chip     = pp_tp_i2c_chip_duplicate(chip);
	this->acc_cond = acc_cond;

	// no need to register at sensor scanner, since read out value is always 0
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void w83792d_clr_intrusion_destroy(pp_tp_obj_t* o) {
    w83792d_clr_intrusion_t* this = (w83792d_clr_intrusion_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_gpio_dev_cleanup(&this->base);
    pp_tp_cond_cleanup(this->acc_cond);
    free(this);
}

static int w83792d_clr_intrusion_get_val(pp_tp_gpio_dev_t* d, unsigned long mask) {
    //w83792d_clr_intrusion_t* this = (w83792d_clr_intrusion_t*)d;
    assert((mask & ~d->valid_mask) == 0);
    (void)mask;
    (void)d;
    return 0; // value 
}

static int w83792d_clr_intrusion_set_val(pp_tp_gpio_dev_t* d,
					 unsigned long mask,
					 unsigned long val,
					 unsigned long tristate UNUSED) {
    w83792d_clr_intrusion_t* this = (w83792d_clr_intrusion_t*)d;
    pp_tp_i2c_chip_t* chip = this->chip;
    int ret = PP_SUC;
    PP_ASSERT(mask, (mask & ~d->valid_mask) == 0);

    if (pp_bmc_tp_cond_is_true(this->chip->init_cond)
	&& pp_bmc_tp_cond_is_true(this->acc_cond)) {

	// set to 1 to clear chassis intrusion bit
	if (val & 0x00000001) {
	    ret = pp_tp_i2c_chip_pre_com(chip);
	    if (ret == PP_SUC) {
		ret = pp_tp_i2c_chip_tx_byte_data(chip, 0x44, 0x80);
		pp_tp_i2c_chip_post_com(chip);
	    }
	}
    } else {
	ret = PP_ERR;
    }

    return ret;
}
