/**
 * sensor_scanner.c
 *
 * scans scanables peridically within another thread
 * 
 * (c) 2005 Peppercon AG, thomas@peppercon.de
 */

#include <sys/time.h>
#include <pthread.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/bmc/sensor.h>
#include <pp/bmc/debug.h>
#include "sensor_scanner.h"

// TODO remove
#include <pp/bmc/tp_scan_sensdev.h>

/*
 * local types
 * ----------------------
 */
typedef struct scan_item_s {
    struct list_head node;
    pp_sensor_scannable_t* scanable;
    int scangroup;
} scan_item_t;

typedef enum {
    SCAN_START,
    SCAN_RUNNING,
    SCAN_HALT,
    SCAN_STOPPED
} scan_state_t;

typedef struct start_item_s {
    struct list_head node;
    pp_sensor_startable_t* startable;
} start_item_t;

/*
 * local declarations and definitions
 * ------------------------------------
 */
static  LIST_HEAD(scan_items);
static volatile scan_item_t* scan_item_cur = NULL; // current scan item
static volatile scan_state_t scan_state = SCAN_STOPPED;
static pthread_mutex_t scan_items_mtx =PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static pthread_cond_t scan_items_cond = PTHREAD_COND_INITIALIZER;
static pthread_cond_t scan_term_cond  = PTHREAD_COND_INITIALIZER;
static pthread_t scan_thread;
static unsigned int scan_interval = 0;

#if !defined(NDEBUG)
static volatile scan_item_t* scan_item_del = NULL;
#endif /* !NDEBUG */

static scan_item_t* scan_item_create(pp_sensor_scannable_t* scanable,
				     int scangroup);
static inline void scan_item_destroy(scan_item_t* si);
static void* scan_thread_func(void* arg);

static LIST_HEAD(start_items);
static start_item_t* start_item_create(pp_sensor_startable_t* startable);
static inline void start_item_destroy(start_item_t* si);

/*
 * public implementation
 * ---------------------
 */

void pp_sensor_scanner_init() {
    int scan_int;
    assert(scan_interval == 0);
    pp_cfg_get_int(&scan_int, "bmc.scan_interval");
    assert(scan_int > 0);
    scan_interval = scan_int;
}

void pp_sensor_scanner_cleanup() {
    struct list_head *i, *n;
    assert(scan_interval != 0);
    
    assert(scan_state == SCAN_STOPPED);
    list_for_each_safe(i, n, &scan_items) {
	list_del(i);
	scan_item_destroy(list_entry(i, scan_item_t, node));
    }
    list_for_each_safe(i, n, &start_items) {
	list_del(i);
	start_item_destroy(list_entry(i, start_item_t, node));
    }
    scan_interval = 0;
}

int pp_bmc_sensor_scanner_start() {
    int err, ret = PP_SUC;
    struct list_head *i, *n;
    assert(scan_interval != 0);

    // FIXME
    //return PP_SUC;
    
    /* callback startables */
    list_for_each_safe(i, n, &start_items) {
	start_item_t* si = list_entry(i, start_item_t, node);
	si->startable->start(si->startable);
	list_del(i);
	start_item_destroy(si);
    }

    /* start scanner thread */
    MUTEX_LOCK(&scan_items_mtx);
    assert(scan_state == SCAN_STOPPED);
    
    if (0 > (err = pthread_create(&scan_thread, NULL, scan_thread_func, 0))) {
	errno = err;
	ret = PP_ERR;
    } else {
	scan_state = SCAN_START;
    }
    
    MUTEX_UNLOCK(&scan_items_mtx);
    return ret;
}

int pp_bmc_sensor_scanner_stop() {
    int err, ret = PP_SUC;
    assert(scan_interval != 0);
    
    MUTEX_LOCK(&scan_items_mtx);
    if (scan_state == SCAN_RUNNING || scan_state == SCAN_START) {
	scan_state = SCAN_HALT;
	pthread_cond_signal(&scan_term_cond);
	MUTEX_UNLOCK(&scan_items_mtx);
	if (0 > (err = pthread_join(scan_thread, NULL))) {
	    errno = err;
	    pp_bmc_log_perror("pp_bmc_sensor_scanner_stop: pthread_join");
	    ret = PP_ERR;
	}
    } else {
	MUTEX_UNLOCK(&scan_items_mtx);
    }
    return PP_SUC;
}

void pp_sensor_scanner_add(pp_sensor_scannable_t* s, int sg) {
    scan_item_t* si = scan_item_create(s, sg);
    assert(scan_interval != 0);
    MUTEX_LOCK(&scan_items_mtx);
    list_add_tail(&si->node, &scan_items);
    MUTEX_UNLOCK(&scan_items_mtx);
}

int pp_sensor_scanner_remove(pp_sensor_scannable_t* s) {
    struct list_head* i;
    scan_item_t* si;
    int ret = PP_ERR;
    assert(scan_interval != 0);

    MUTEX_LOCK(&scan_items_mtx);
    list_for_each(i, &scan_items) {
	si = list_entry(i, scan_item_t, node);
	if (si->scanable == s) {      // found
#if !defined(NDEBUG)
	    // paranoia test, no other thread should be doing the same
	    assert(scan_item_del != si);
	    scan_item_del = si;
#endif /* !NDEBUG */
	    while ((scan_state == SCAN_RUNNING || scan_state == SCAN_HALT)
		   && si == scan_item_cur) {
		pthread_cond_wait(&scan_items_cond, &scan_items_mtx);
	    }
#if !defined(NDEBUG)
	    scan_item_del = NULL;
#endif /* !NDEBUG */
	    list_del(i);
	    scan_item_destroy(si);
	    ret = PP_SUC;
	    break;
	}
    }
    MUTEX_UNLOCK(&scan_items_mtx);
    return ret;
}

void pp_sensor_scanner_add_startable(pp_sensor_startable_t* s) {
    start_item_t* si = start_item_create(s);
    list_add_tail(&si->node, &start_items);
}

int pp_sensor_scanner_remove_startable(pp_sensor_startable_t* s) {
    struct list_head* i;
    start_item_t* si;
    int ret = PP_ERR;
    
    list_for_each(i, &start_items) {
	si = list_entry(i, start_item_t, node);
	if (si->startable == s) { // found
	    list_del(i);
	    start_item_destroy(si);
	    ret = PP_SUC;
	    break;
	}
    }
    return ret;
}

/*
 * local functions
 * ----------------
 */
static void* scan_thread_func(void* arg UNUSED) {
    struct list_head* i;
    scan_item_t* si;
    struct timespec to;

    MUTEX_LOCK(&scan_items_mtx);
    scan_state = SCAN_RUNNING;
    
    // TODO: implement scangroups 

    /* sleep a bit, initially */
    to = pp_get_wakeup_time(1000 /* in ms */);
    pthread_cond_timedwait(&scan_term_cond, &scan_items_mtx, &to);
    
    while(scan_state != SCAN_HALT) {
	/*
	 * determine overall time of our loop: we need to stretch
	 * the inner loop to the length of the desired scan-intervall
	 * in case all the update functions consume less time
	 */
	to = pp_get_wakeup_time(scan_interval /* in ms */);
	
	for (i = scan_items.next; i != &scan_items; i = i->next) {
	    si = list_entry(i, scan_item_t, node);
	    scan_item_cur = si;
	    MUTEX_UNLOCK(&scan_items_mtx);
	    si->scanable->update(si->scanable);
	    MUTEX_LOCK(&scan_items_mtx);
	    scan_item_cur = NULL;
	    pthread_cond_signal(&scan_items_cond);
	    assert(scan_state != SCAN_STOPPED && scan_state != SCAN_START);
	    if (scan_state == SCAN_HALT) {
		goto bailout;
	    }
#if defined (PP_FEAT_IPMI_SERVER_SENS_SCAN_DELAY)
	    /* PP_FEAT_IPMI_SERVER_SENS_SCAN_DELAY in milli-seconds */
	    struct timespec to1;
	    to1 = pp_get_wakeup_time(PP_FEAT_IPMI_SERVER_SENS_SCAN_DELAY);
	    pthread_cond_timedwait(&scan_term_cond, &scan_items_mtx, &to1);
	    if (scan_state == SCAN_HALT) {
		goto bailout;
	    }
#endif /* PP_FEAT_IPMI_SERVER_SENS_SCAN_DELAY */
	}
	/* sleeep until interval is finished */
	pthread_cond_timedwait(&scan_term_cond, &scan_items_mtx, &to);
    }

 bailout:
    scan_state = SCAN_STOPPED; 
    MUTEX_UNLOCK(&scan_items_mtx);	    
    return NULL;
}

static scan_item_t* scan_item_create(pp_sensor_scannable_t* scanable,
				     int scangroup) {
    scan_item_t* si = malloc(sizeof(scan_item_t));
    si->scanable = scanable;
    si->scangroup = scangroup;
    return si;
}

static inline void scan_item_destroy(scan_item_t* si) {
    free(si);
}
	
static start_item_t* start_item_create(pp_sensor_startable_t* startable) {
    start_item_t* si = malloc(sizeof(start_item_t));
    si->startable = startable;
    return si;
}

static inline void start_item_destroy(start_item_t* si) {
    free(si);
}
