/**
 * tp_acpi_power_state_opma.c
 *
 * A concrete implementation of the ACPI power state sensor for
 * OPMA compliant hardware.
 * 
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 */

#include <pp/base.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_acpi_power_state_opma.h>

static void
acpi_power_state_opma_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
				   pp_tp_sensdev_t* source, int reading);
static int value_to_state(int value);

pp_tp_obj_t*
pp_tp_acpi_power_state_opma_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[ACPI_POWER_STATE_OPMA] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_tp_acpi_power_state_opma_t* acpi_power_state = NULL;
    pp_tp_gpio_multi_sens_t * acpi_state_multi_gpio;

    if (pp_tp_arg_scanf(args, 0, &err, "o<sm>", &acpi_state_multi_gpio) != 1) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
    } else {
	acpi_power_state = calloc(1, sizeof(pp_tp_acpi_power_state_opma_t));
	if (PP_FAILED(pp_tp_acpi_power_state_opma_init(acpi_power_state,
					      PP_TP_ACPI_POWER_STATE, id,
					      pp_tp_acpi_power_state_opma_dtor,
					      acpi_state_multi_gpio))) {
	    pp_bmc_log_error(errmsg, ___F, id, "init");
	    free(acpi_power_state);
	    acpi_power_state = NULL;
	}
    }

    pp_strstream_free(&err);
    return (pp_tp_obj_t*)acpi_power_state;
}

void
pp_tp_acpi_power_state_opma_dtor(pp_tp_obj_t* this)
{
    if (this != NULL) {
	pp_tp_acpi_power_state_opma_cleanup
	    ((pp_tp_acpi_power_state_opma_t *)this);
	free(this);
    }
}

int
pp_tp_acpi_power_state_opma_init(pp_tp_acpi_power_state_opma_t* this,
				 pp_tp_obj_type_t type, const char* id,
				 pp_tp_obj_dtor_func_t dtor,
			      pp_tp_gpio_multi_sens_t * acpi_state_multi_gpio)
{
    if (PP_SUCCED(pp_tp_acpi_power_state_init(&this->base, type, id, dtor))) {
	this->acpi_state_multi_gpio =
	    pp_tp_gpio_multi_sens_duplicate(acpi_state_multi_gpio);
	this->gpio_subscriber.recv_reading =acpi_power_state_opma_recv_reading;
	pp_bmc_tp_sensdev_subscribe(&this->acpi_state_multi_gpio->base,
				&this->gpio_subscriber);
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_tp_acpi_power_state_opma_cleanup(pp_tp_acpi_power_state_opma_t* this)
{
    pp_bmc_tp_sensdev_unsubscribe(&this->acpi_state_multi_gpio->base,
			      &this->gpio_subscriber);
    pp_tp_gpio_multi_sens_release(this->acpi_state_multi_gpio);
    pp_tp_acpi_power_state_cleanup(&this->base);
}

static void
acpi_power_state_opma_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
				   pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_tp_acpi_power_state_opma_t* this =
	PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_acpi_power_state_opma_t,
			      gpio_subscriber);
    int newstate = value_to_state(reading);
    this->base.base.reading = newstate;
    pp_bmc_tp_sensdev_notify_subscribers(&this->base.base, newstate);
}

static int
value_to_state(int value)
{
    if (value == 0x0) {
	return (1<<HOST_POWER_STATE_S0);
    } else if (value == 0x1) {
	return (1<<HOST_POWER_STATE_S1);
    } else if (value == 0x2) {
	return (1<<HOST_POWER_STATE_S2);
    } else if (value == 0x3) {
	return (1<<HOST_POWER_STATE_S3);
    } else if (value == 0x4) {
	return (1<<HOST_POWER_STATE_S45);
    } else {
	if (value < 0) {
	    pp_bmc_log_debug("[ACPI_POWER_STATE_OPMA] %s(): failed to read multi GPIO (ignored)", ___F);
	}
	return (1<<HOST_POWER_STATE_UNKNOWN);
    }
}
