/**
 * tp_gpio_bulk_sens.c
 *
 * A sensor for multiple gpio pins of a gpio-device
 * 
 * (c) 2005 Peppercon AG, 2005/11/23, thomas@peppercon.de
 */

#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/tp_gpio_bulk_sens.h>
#include <pp/bmc/debug.h>

static void gpio_bulk_sens_dtor(pp_tp_obj_t* o);
static ipmi_sdr_header_t* gpio_bulk_sens_default_sdr(pp_tp_sensdev_t* s);
static void gpio_changed(pp_tp_gpio_subscriber_t* s, unsigned long value, int valid UNUSED);
static unsigned long get_gpio_mask(pp_tp_gpio_subscriber_t* s);

pp_tp_obj_t* pp_tp_gpio_bulk_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[GpioBulkSens] %s: '%s' failed (%s)";
    pp_tp_gpio_bulk_sens_t* this = NULL;
    pp_tp_gpio_dev_t* gpio_dev;
    unsigned long gpio_mask;
    unsigned char sens_type;
    unsigned char reading_type;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    if (pp_tp_arg_scanf(args, 0, &err, "o<g>dd<c>d<c>",
			&gpio_dev, &gpio_mask,
			&sens_type, &reading_type) != 4) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	errno = EINVAL;
    } else if (((gpio_dev->valid_mask | gpio_mask) ^ gpio_dev->valid_mask)
	       != 0) {
	pp_bmc_log_error(errmsg, ___F, id, "gpio mask not supported by dev");
	errno = EINVAL;
    } else {
	this = malloc(sizeof(pp_tp_gpio_bulk_sens_t));
	pp_tp_sensdev_init(&this->base, PP_TP_GPIO_SENS, id,
			   gpio_bulk_sens_dtor, gpio_bulk_sens_default_sdr);
	this->gpio_dev = pp_tp_gpio_dev_duplicate(gpio_dev);
	this->gpio_mask = gpio_mask;
	this->sens_type = sens_type;
	this->reading_type = reading_type;

	pp_tp_gpio_dev_init_gpio(this->gpio_dev, this->gpio_mask);
	
	// subscribe as gpio listener
	this->gpio_subs.gpio_changed = gpio_changed;
	this->gpio_subs.get_gpio_mask = get_gpio_mask;
	pp_tp_gpio_dev_register_subscriber(this->gpio_dev, &this->gpio_subs);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void gpio_bulk_sens_dtor(pp_tp_obj_t* o) {
    pp_tp_gpio_bulk_sens_t* this = (pp_tp_gpio_bulk_sens_t*)o;
    assert(this != NULL);
    pp_tp_gpio_dev_unregister_subscriber(this->gpio_dev, &this->gpio_subs);
    pp_tp_gpio_dev_release(this->gpio_dev);
    pp_tp_sensdev_cleanup(&this->base);
    free(this);
}
 
static ipmi_sdr_header_t* gpio_bulk_sens_default_sdr(pp_tp_sensdev_t* s) {
    pp_tp_gpio_bulk_sens_t* this = (pp_tp_gpio_bulk_sens_t*)s;
    ipmi_sdr_compact_sensor_t* sdr;
    sdr = ipmi_sdr_compact_part_create(this->sens_type, this->reading_type);
    return &sdr->header;
}

static void gpio_changed(pp_tp_gpio_subscriber_t* s, unsigned long value, int valid UNUSED) {
    int old_reading, new_reading;
    pp_tp_gpio_bulk_sens_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(s, pp_tp_gpio_bulk_sens_t, gpio_subs);

    old_reading = this->base.reading;
    new_reading = (int)value & this->gpio_mask;
    if (new_reading != old_reading) {
	pp_bmc_log_debug("[GpioBulkSens] '%s' has new reading %d", 
			 this->base.base.id, new_reading);
	this->base.reading = new_reading;
	pp_tp_gpio_bulk_sens_notify_subscribers(this, new_reading);
    }
}

static u_long get_gpio_mask(pp_tp_gpio_subscriber_t* subscriber) {
    pp_tp_gpio_bulk_sens_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_gpio_bulk_sens_t,
				 gpio_subs);
    return this->gpio_mask;
}


