/**
 * tp_gpio_multi_sens.h
 *
 * A sensor for a collection of concrete gpio input signals
 * 
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 */

#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/tp_gpio_multi_sens.h>
#include <pp/bmc/debug.h>
#include <pp/selector.h>

typedef struct subsc_entry_s {
    pp_tp_gpio_multi_sens_t* multi_sens;
    pp_tp_sensdev_subscriber_t  subscriber;
    size_t                   index;
} subsc_entry_t;

static void gpio_multi_sens_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					 pp_tp_sensdev_t* source, int reading);
static void apply_new_value(pp_tp_gpio_multi_sens_t* this);
static int debounce_handler(int item_id UNUSED, void* ctx);

pp_tp_obj_t*
pp_tp_gpio_multi_sens_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    const char* errmsg = "[GPIO_multi_in] %s: '%s' failed (%s)";
    pp_tp_gpio_multi_sens_t* gpio_multi_sens = NULL;
    vector_t * gpios;
    int i, debounce, arg_cnt, gpio_cnt;

    arg_cnt = vector_size(args);

    if (arg_cnt < 1) {
	pp_bmc_log_error(errmsg, ___F, id, "missing argument(s)");
	goto bail;
    } else if (arg_cnt > 32) {
	pp_bmc_log_error(errmsg, ___F, id, "too many arguments");
	goto bail;
    }

    if (pp_tp_arg_scanf(args, 0, &err, "d", &debounce) != 1) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }

    gpio_cnt = arg_cnt - 1;
    gpios = vector_new(NULL, gpio_cnt,
		       (vector_elem_del_func_simple)pp_tp_obj_release);
    for (i = 0; i < gpio_cnt; ++i) {
	pp_tp_gpio_sens_t * gpio_sens;
        if (pp_tp_arg_scanf(args, 1 + i, &err, "o<sg>", &gpio_sens) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	    vector_delete(gpios);
            goto bail;
        }
	vector_add(gpios, pp_tp_gpio_sens_duplicate(gpio_sens));
    }

    gpio_multi_sens = calloc(1, sizeof(pp_tp_gpio_multi_sens_t));
    if (PP_FAILED(pp_tp_gpio_multi_sens_init(gpio_multi_sens,
					     PP_TP_GPIO_MULTI_SENS, id,
					     pp_tp_gpio_multi_sens_dtor,
					     NULL, gpios, debounce))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	vector_delete(gpios);
	free(gpio_multi_sens);
	gpio_multi_sens = NULL;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)gpio_multi_sens;
}

void
pp_tp_gpio_multi_sens_dtor(pp_tp_obj_t* this)
{
    pp_tp_gpio_multi_sens_cleanup((pp_tp_gpio_multi_sens_t *)this);
    free(this);
}

int
pp_tp_gpio_multi_sens_init(pp_tp_gpio_multi_sens_t* this,
			   pp_tp_obj_type_t type, const char* id,
			   pp_tp_obj_dtor_func_t dtor,
			   pp_tp_sensdev_default_sdr_func_t default_sdr,
			   vector_t* gpios, int debounce)
{
    pp_tp_gpio_sens_t* gpio;
    subsc_entry_t* subsc_entry;
    size_t i, gpio_cnt = vector_size(gpios);
    
    if (PP_SUCCED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				     default_sdr))) {
	this->debounce = debounce;
	this->debounce_hdl = 0;
	this->value_pending = 0;
	this->gpios = gpios; /* gpios are already duplicates */
	this->gpio_subscribers = vector_new(NULL, gpio_cnt, free);
	for (i = 0; i < gpio_cnt; ++i) {
	    gpio = (pp_tp_gpio_sens_t*)vector_get(this->gpios, i);
	    subsc_entry = calloc(1, sizeof(subsc_entry_t));
	    subsc_entry->multi_sens = this;
	    subsc_entry->index = i;
	    subsc_entry->subscriber.recv_reading =gpio_multi_sens_recv_reading;
	    vector_add(this->gpio_subscribers, subsc_entry);
	    pp_tp_gpio_sens_subscribe(gpio, &subsc_entry->subscriber);
	}
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_tp_gpio_multi_sens_cleanup(pp_tp_gpio_multi_sens_t* this)
{
    size_t i, gpio_cnt = vector_size(this->gpios);
    pp_tp_gpio_sens_t* gpio;
    pp_tp_sensdev_subscriber_t* subscriber;

    for (i = 0; i < gpio_cnt; ++i) {
	gpio = (pp_tp_gpio_sens_t*)vector_get(this->gpios, i);
	subscriber = (pp_tp_sensdev_subscriber_t*)vector_get(this->
							  gpio_subscribers, i);
	pp_tp_gpio_sens_unsubscribe(gpio, subscriber);
    }
    vector_delete(this->gpio_subscribers);
    vector_delete(this->gpios);
    pp_tp_sensdev_cleanup(&this->base);
}

static void
gpio_multi_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc, 
			     pp_tp_sensdev_t* source UNUSED, int reading)
{
    subsc_entry_t* subsc_entry =
	PP_TP_INTF_2_OBJ_CAST(subsc, subsc_entry_t, subscriber);
    pp_tp_gpio_multi_sens_t* this = subsc_entry->multi_sens;
    u_int32_t new_value;

    // in case of invalid readings, we remember the last known state,
    // may not be correct in all cases
    if (reading < 0) return;	

    if (reading) {
	new_value = this->value_pending | (1 << subsc_entry->index);
    } else {
	new_value = this->value_pending & ~(1 << subsc_entry->index);
    }

    if (this->debounce == 0) {
	/* use the new value directly */
	this->value_pending = new_value;
	apply_new_value(this);
    } else {
	/* setup timer but don't change value yet */
	this->value_pending = new_value;
	if (this->debounce_hdl != 0) pp_select_remove_to(this->debounce_hdl);
	this->debounce_hdl = pp_select_add_to(this->debounce, 0,
					      debounce_handler, (void*)this);
    }
}

static void
apply_new_value(pp_tp_gpio_multi_sens_t* this)
{
    if (this->value_pending != this->base.reading) {
	this->base.reading = this->value_pending;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, this->base.reading);
    }
}

static int
debounce_handler(int item_id UNUSED, void* ctx)
{
    pp_tp_gpio_multi_sens_t* this = (pp_tp_gpio_multi_sens_t*)ctx;
    apply_new_value(this);
    this->debounce_hdl = 0;
    return PP_SUC;
}
