/**
 * pp_tp_scan_sensdev.c
 *
 * Abstract sensor device object
 * 
 * (c) 2005 Peppercon AG, 7/25/2005, tbr@peppecon.de
 */

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/tp_scan_sensdev.h>

static int scannable_sensdev_notify(void* ctx);

int pp_tp_scan_sensdev_init(pp_tp_scan_sensdev_t* this, pp_tp_obj_type_t type,
			    const char* id, pp_tp_obj_dtor_func_t dtor,
			    pp_tp_sensdev_default_sdr_func_t default_sdr,
			    void (*update)(pp_sensor_scannable_t*),
			    pp_tp_cond_t* scan) {
    pp_tp_sensdev_init(&this->base, type, id, dtor, default_sdr);
    this->scan_cond = (scan == NULL) ? NULL : pp_tp_cond_duplicate(scan);
    this->scannable.update = update;
    this->update_sf_id = 0;
    MUTEX_CREATE_ERRORCHECKING(&this->mtx);
    return PP_SUC;
}


/*
 * ATTENTION: this function is meant to be used by the
 *            scanner thread only! Do not call it from
 *            any other context than the scanner thread
 *            and only if the sensor implements scannable
 */
int pp_tp_scan_sensdev_update_reading(pp_tp_scan_sensdev_t* this,
				      int reading) {
    int ret = 0;
    MUTEX_LOCK(&this->mtx);
    
    /*
     * theoretically it may happen that the scanner is faster calling
     * update reading than BMC thread is executing the scheduled function
     * for the notfication handling. In this case, we do not update the
     * reading and do not schedule another notification function,
     * since the first one is still supposed to run.
     */
    if (this->update_sf_id == 0) {
	if (this->base.reading != reading) {
	    this->base.reading = reading;
	    this->update_sf_id = pp_select_add_sf(scannable_sensdev_notify,
						  this);
	}
    }
    MUTEX_UNLOCK(&this->mtx);
    return ret;
}

void pp_tp_scan_sensdev_unregister(pp_tp_scan_sensdev_t* this) {
    pp_sensor_scanner_remove(&this->scannable);

    /*
     * no need to lock anything here, after scanner_remove,
     * the BMC-thread is the only thread working with the object
     */
    if (this->update_sf_id != 0) {
	pp_select_remove_sf(this->update_sf_id);
    }
}

static int scannable_sensdev_notify(void* ctx) {
    pp_tp_scan_sensdev_t* this = (pp_tp_scan_sensdev_t*)ctx;
    MUTEX_LOCK(&this->mtx);
    pp_bmc_tp_sensdev_notify_subscribers(&this->base, this->base.reading);
    this->update_sf_id = 0;
    MUTEX_UNLOCK(&this->mtx);
    return 0;
}
