/**
 * tp_sdr
 *
 * implement various objects for SDR configuration
 * 
 * (c) 2005 Peppercon AG, 2005/06/04, tbr@peppecon.de
 */

#include <string.h>
#include <pp/base.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/tp_sdr.h>
#include <pp/bmc/ipmi_sdr.h>

static void thresh_sdr_dtor(pp_tp_obj_t* o);
static void disc_sdr_dtor(pp_tp_obj_t* o);

/* 
 * refer to header and pp_tp_thresh_sdr_ctor for docu
 */
int pp_tp_cfg_sdr_init(pp_tp_cfg_sdr_t* this, pp_tp_obj_type_t type,
		       const char* id, void (*dtor)(pp_tp_obj_t*),
		       vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    if (pp_tp_arg_scanf(args, 0, &err, "d<c>", &this->caps.byteval) != 1) {
	pp_bmc_log_error("[CfgSDR] (%s) failed: %s", id, 
			 pp_strstream_buf(&err));
	return -1;
    }
    pp_tp_obj_init(&this->base, type, id, dtor);
    return 1; // return the current argument vector index
}

/* 
 * refer to header and pp_tp_thresh_sdr_ctor for docu
 */
int pp_tp_thresh_sdr_init(pp_tp_thresh_sdr_t* this, pp_tp_obj_type_t type,
			 const char* id, void (*dtor)(pp_tp_obj_t*), 
			 vector_t* args) {
    const char* cname = "[ThreshSDR]";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    int argidx, n;
    short assevt, deassevt;

    // empty out things
    memset(this, 0, sizeof(pp_tp_thresh_sdr_t));
	
    if (0 > (argidx = pp_tp_cfg_sdr_init(&this->base, type, id, dtor, 
					 args))) {
	pp_bmc_log_error("%s (%s) failed", cname, id);
	goto bailout;
    }

    if (this->base.caps.bits.threshold == 0 &&
	this->base.caps.bits.hysteresis == 0) {
	pp_bmc_log_error("%s (%s) failed: neither threshold nor "
			 "hysteresis support", cname, id);
	argidx = -1;
	goto bailout;
    }
	
    // read in threshold values
    if (this->base.caps.bits.threshold != 0) { 
	if ((n = pp_tp_arg_scanf(args, argidx, &err, 
		 "d<s>d<s>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>",
				 &assevt, &deassevt,
				 &this->mask.threshold.set,
				 &this->mask.threshold.read,
				 &this->nominal_read,
				 &this->normal_max,
				 &this->normal_min,
				 &this->threshold.upper.non_recover,
				 &this->threshold.upper.critical,
				 &this->threshold.upper.non_critical,
				 &this->threshold.lower.non_recover,
				 &this->threshold.lower.critical,
				 &this->threshold.lower.non_critical)) 
	    != 13) {
	    goto bailout_argscanf;
	} else {
	    this->mask.threshold.assert_event_le16 = cpu_to_le16(assevt);
	    this->mask.threshold.deassert_event_le16 = cpu_to_le16(deassevt);
	    argidx += n;
	}
    }

    // read in hysteresis values
    if (this->base.caps.bits.hysteresis != 0) {
	if ((n = pp_tp_arg_scanf(args, argidx, &err, "d<c>d<c>",
				 &this->threshold.hysteresis.positive,
				 &this->threshold.hysteresis.negative)) != 2) {
	    goto bailout_argscanf;
	} else {
	    argidx += n;
	}
    }
bailout:
    // no need to free err-strstream here, hasn't been used!
    return argidx;

bailout_argscanf:
    pp_bmc_log_error("[ThreshSDR] (%s) failed: %s", id, 
		     pp_strstream_buf(&err));
    pp_tp_cfg_sdr_cleanup(&this->base);
    pp_strstream_free(&err);
    return -1;
}

/* 
 * refer to header for docu
 */
pp_tp_obj_t* pp_tp_thresh_sdr_ctor(const char* id, vector_t* args) {
    pp_tp_thresh_sdr_t* this;
    
    this = malloc(sizeof(pp_tp_thresh_sdr_t));
    if (0 > pp_tp_thresh_sdr_init(this, PP_TP_THRESH_SDR, id, thresh_sdr_dtor, 
				 args)) {
	free(this);
	this = NULL;
    }
    return (pp_tp_obj_t*)this;
}

int pp_tp_thresh_cfg_sdr_fill_sdr(pp_tp_thresh_sdr_t* sdrcfg,
				   unsigned char snum,
                                   const char* desc,
                                   unsigned char entity_id,
                                   unsigned char entity_instance,
                                   ipmi_sdr_header_t* sdr)
{
    ipmi_sdr_id_t* id;
    ipmi_sdr_key_sensor_t* key;
    ipmi_sdr_full_sensor_t* fsdr;
    int len;
    int ret = PP_SUC;
    
    // check description
    if ((len = strlen(desc)) > (IPMI_SENSOR_ID_STRING_LEN - 1))
	len = IPMI_SENSOR_ID_STRING_LEN - 1;

    // set sensor number
    if (NULL != (key = ipmi_sdr_get_key_sensor_ptr(sdr))) {
        key->owner_id = PP_BMC_IPMB_ADDR;
        key->lun = 0;
        key->channel = IPMI_CHAN_PRIMARY_IPMB;
        key->sensor_num = snum;
    }
    
    // set sensor id
    if (NULL != (id = ipmi_sdr_get_id_ptr(sdr))) {
        id->code = IPMI_STR_ASCII8;
        id->length = len;
        strncpy(id->string, desc, len);
        id->string[len] = '\0';
    }

    // fill in SDR config
    // TODO: we rely on a full SDR type. We could also write
    //       our information to a compact SDR but this would
    //       require additional code ...
    if (sdr->type == IPMI_SDR_FULL_SENSOR) {
        fsdr = (ipmi_sdr_full_sensor_t*)sdr;

        fsdr->entity.id = entity_id;
        fsdr->entity.instance = entity_instance & 0x7f;
        fsdr->entity.is_logical = (entity_instance & 0x80) >> 7;
    
        *(unsigned char*)&fsdr->sensor.caps = sdrcfg->base.caps.byteval;
        fsdr->mask.threshold.assert_event_le16 =
            sdrcfg->mask.threshold.assert_event_le16;
        fsdr->mask.threshold.deassert_event_le16 =  
            sdrcfg->mask.threshold.deassert_event_le16;
        fsdr->mask.threshold.set = sdrcfg->mask.threshold.set;
        fsdr->mask.threshold.read = sdrcfg->mask.threshold.read;
        fsdr->nominal_read = sdrcfg->nominal_read;
        fsdr->normal_max = sdrcfg->normal_max;
        fsdr->normal_min = sdrcfg->normal_min;
        *(unsigned char*)&fsdr->analog_flag = 0x7; // nominal, max, min specif.
        fsdr->threshold = sdrcfg->threshold;
    } else {
        // unsupported SDR type
        pp_bmc_log_error("[ThreshSDR] SDR type %d not supported", sdr->type);
        ret = PP_ERR;
    }
    
    return ret;
}


static void thresh_sdr_dtor(pp_tp_obj_t* o) {
    if (o != NULL) {
	pp_tp_thresh_sdr_t* this = (pp_tp_thresh_sdr_t*)o;
	pp_tp_thresh_sdr_cleanup(this);
	free(this);
    }
}

/*
 * discrete sensor SDR
 */

/* 
 * refer to header and pp_tp_thresh_sdr_ctor for docu
 */
int pp_tp_disc_sdr_init(pp_tp_disc_sdr_t* this, pp_tp_obj_type_t type,
			const char* id, void (*dtor)(pp_tp_obj_t*), 
			vector_t* args) {
    const char* cname = "[DiscSDR]";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    int argidx, n;
    short assevt, deassevt, readm;
   
    // empty out things
    memset(this, 0, sizeof(pp_tp_disc_sdr_t));
	
    if (0 > (argidx = pp_tp_cfg_sdr_init(&this->base, type, id, dtor, 
					 args))) {
	pp_bmc_log_error("%s (%s) failed", cname, id);
	goto bailout;
    }
  
    // read in entity information and event/reading masks
    if ((n = pp_tp_arg_scanf(args, argidx, &err, "d<s>d<s>d<s>",
			     &assevt, &deassevt, &readm)) != 3) {
	goto bailout_argscanf;
    } else {
	this->mask.discrete.assert_event_le16 = cpu_to_le16(assevt);
	this->mask.discrete.deassert_event_le16 = cpu_to_le16(deassevt);
	this->mask.discrete.read_le16 = cpu_to_le16(readm);
	argidx += n;
    }


bailout:
    // no need to free err-strstream here, hasn't been used!
    return argidx;

bailout_argscanf:
    pp_bmc_log_error("[DiscSDR] (%s) failed: %s", id, 
		     pp_strstream_buf(&err));
    pp_tp_cfg_sdr_cleanup(&this->base);
    pp_strstream_free(&err);
    return -1;
}

pp_tp_obj_t* pp_tp_disc_sdr_ctor(const char* id, vector_t* args) {
    pp_tp_disc_sdr_t* this;
    
    this = malloc(sizeof(pp_tp_disc_sdr_t));
    if (0 > pp_tp_disc_sdr_init(this, PP_TP_DISC_SDR, id, disc_sdr_dtor, 
				args)) {
	free(this);
	this = NULL;
    }
    return (pp_tp_obj_t*)this;
}

int pp_tp_disc_cfg_sdr_fill_sdr(pp_tp_disc_sdr_t* sdrcfg,
				 unsigned char snum,
				 const char* desc,
                                 unsigned char entity_id,
                                 unsigned char entity_instance,
                                 ipmi_sdr_header_t* sdr)
{
    ipmi_sdr_id_t*     id;
    ipmi_sdr_key_sensor_t* key;
    int len;
    int ret = PP_SUC;

    if ((len = strlen(desc)) > (IPMI_SENSOR_ID_STRING_LEN - 1))
	len = IPMI_SENSOR_ID_STRING_LEN - 1;

    // set sensor number
    if (NULL != (key = ipmi_sdr_get_key_sensor_ptr(sdr))) {
        key->owner_id = PP_BMC_IPMB_ADDR;
        key->lun = 0;
        key->channel = IPMI_CHAN_PRIMARY_IPMB;
        key->sensor_num = snum;
    }
    
    // set sensor id
    if (NULL != (id = ipmi_sdr_get_id_ptr(sdr))) {
        id->code = IPMI_STR_ASCII8;
        id->length = len;
        strncpy(id->string, desc, len);
        id->string[len] = '\0';
    }
  
    // fill in SDR config
    if (sdr->type == IPMI_SDR_COMPACT_SENSOR) {
        ipmi_sdr_compact_sensor_t* csdr;
        csdr = (ipmi_sdr_compact_sensor_t*)sdr;

        csdr->entity.id = entity_id;
        csdr->entity.instance = entity_instance & 0x7f;
        csdr->entity.is_logical = (entity_instance & 0x80) >> 7;

	*(unsigned char*)&csdr->sensor.caps = sdrcfg->base.caps.byteval;
	csdr->mask.discrete.assert_event_le16 =
            sdrcfg->mask.discrete.assert_event_le16;
        csdr->mask.discrete.deassert_event_le16 =
            sdrcfg->mask.discrete.deassert_event_le16;
        csdr->mask.discrete.read_le16 =
            sdrcfg->mask.discrete.read_le16;
    } else if (sdr->type == IPMI_SDR_FULL_SENSOR) {
        ipmi_sdr_full_sensor_t* fsdr;
        fsdr = (ipmi_sdr_full_sensor_t*)sdr;

        fsdr->entity.id = entity_id;
        fsdr->entity.instance = entity_instance & 0x7f;
        fsdr->entity.is_logical = (entity_instance & 0x80) >> 7;

	*(unsigned char*)&fsdr->sensor.caps = sdrcfg->base.caps.byteval;
	fsdr->mask.discrete.assert_event_le16 =
            sdrcfg->mask.discrete.assert_event_le16;
        fsdr->mask.discrete.deassert_event_le16 =
            sdrcfg->mask.discrete.deassert_event_le16;
        fsdr->mask.discrete.read_le16 =
            sdrcfg->mask.discrete.read_le16;

        // TODO rgue: do we have to init some more fields here???
        *(unsigned char*)&fsdr->analog_flag = 0x0; // nominal, max, min NOT specif.
    } else {
        // unsupported sdr type
        pp_bmc_log_error("[DiscSDR] SDR type %d not supported", sdr->type);
        ret = PP_ERR;
    }
    
    return ret;
}

static void disc_sdr_dtor(pp_tp_obj_t* o) {
    if (o != NULL) {
	pp_tp_disc_sdr_t* this = (pp_tp_disc_sdr_t*)o;
	pp_tp_disc_sdr_cleanup(this);
	free(this);
    }
}
