/**
 * serial_adapter.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * A serial port channel adapter for the bmc. This channel
 * currently supports direct connections in basic mode only.
 * 
 * FIXME: muxe konfigurierbar machen
 * FIXME: test DCD polling on MSI board
 * 
 * Note: This implementation polls DCD every 1000ms instead of using
 * Signals. The use of signals for monitoring DCD in an interupt like 
 * manner should be possible but is hardly documented.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/ioctl.h>
#include <sys/ioctl.h>
#include <termios.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/termios.h>
#include <lara.h>

#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/session_manager.h>


#include "pp/bmc/debug.h"
#include "pp/bmc/serial_adapter.h"      // the functions that we offer
#include "pp/bmc/serial_lowlevel.h"     // the functions that we need

#include "serial_modem_cmds.h"
#include "serial_snoop.h"

#define MAX_IPMI_MSG_LEN 172     // same value as in openIPMI

typedef enum {
    PP_SER_ACTIVATING,
    PP_SER_ACTIVATED,
    PP_SER_DEACTIVATING,
    PP_SER_DEACTIVATED
} serial_adapter_state_t;
serial_adapter_state_t serial_adapter_state = PP_SER_DEACTIVATED;

static pp_bmc_router_chan_adapter_config_t* serial_channel_config = NULL;

/* variables for PP_SER_ACTIVATED, PP_SER_DEACTIVATING */
typedef struct {
    // incoming bytes are added to this buffer until the msg is complete
    int inbuffer_len;   // -1 if no incoming message
                        // else index for next byte in incoming_buffer
    int serial_hndl;
    int serial_fd;
    int ping_hndl;      // timeout handler for serial ping msg
    int dcd_hndl;       // timeout handler for dcd monitoring
    unsigned char dcd_state;   // current dcd state
    
    unsigned char inbuffer[MAX_IPMI_MSG_LEN];
    unsigned char last_byte_escaped;
    
    // only valid for deactivation
    int deactivate_count;      // number of times that the deactivate_ping message has been sent
    int deactivate_3s_hndl;    // 3 second timeout handler for deactivate
    int deactivate_20ms_hndl;  // message repetition timeout handler
    pp_bmc_serial_mux_state_t deactivate_newstate;   // new new state after deactivation
    
    unsigned int session_id;   // active session id, set to 0 if no active session
} sa_globals_t;


sa_globals_t sa_globals;
sa_globals_t *sag = NULL;


typedef struct {
    unsigned char ipmi_msg_sequence; // 6 bits only 
} serial_channeldata_t;


/**
 * Some internal function blueprints
 */
static int serial_read(const int item_id UNUSED, const int fd,
                       const short event UNUSED, void* ctx UNUSED);
static int serial_receive_bytes(unsigned char* buffer, int length);
static int serial_receive_msg(void);
static int serial_send_ping(unsigned char session_state);
static int serial_send_ping_timeout(const int id, void* ctx);
static void serial_deactivate_final(void);
static int serial_dcd_timeout(const int id, void* ctx);

void bmc_serial_adapter_set_fd(int fd) {
    sag->serial_fd = fd;
}

#define PING_NO_SESSION        0x00
#define PING_SESSION_ACTIVE    0x01
#define PING_MUX_SWITCH        0x02

static int serial_send_ping(unsigned char session_state) {
    /* format and send a _command_ to remote console software */
    imsg_t* imsg;
    imsg = pp_bmc_imsg_new(2);
    
    imsg->netfn = IPMI_NETFN_TRANSPORT;
    imsg->cmd = IPMI_CMD_SERIAL_MODEM_CONNECTION_ALIVE;

    // responder ( = serial connection partner)
    imsg->rs_addr = 0x81;        // serial console 0 management software
    imsg->rs_lun = 0;
    
    // requester (= bmc)
    imsg->rq_addr = 0x20;
    imsg->rq_lun = 0;
    imsg->rq_seq = 42;           // we dont check the response, so any seq is fine
    
    imsg->data[0] = session_state & 0x7;
    imsg->data[1] = 0x51;
    
    imsg->buf = malloc(sizeof(serial_channeldata_t));
    imsg->buf_size = sizeof(serial_channeldata_t);
    imsg->buf[0] = 0;  // serial msg sequence
    
    imsg->chan = IPMI_CHAN_SERIAL;
    
    return pp_bmc_router_send_msg(imsg);
}

static int serial_send_ping_timeout(const int id UNUSED, void* ctx UNUSED) {
    if (sag->session_id == 0) {
        serial_send_ping(PING_NO_SESSION);
    } else {
        serial_send_ping(PING_SESSION_ACTIVE);
    }
    return PP_SUC;
}

static int serial_dcd_timeout(const int id UNUSED, void* ctx UNUSED) {
    int status;
    unsigned char val;

    if (ioctl(sag->serial_fd, TIOCMGET, &status) < 0) {
        // something is wrong - better assume that dcd is low
        pp_bmc_log_debug("[SER] could not read DCD status line");
        val = 0;
    } else {
        if ((status & TIOCM_CAR) == TIOCM_CAR) {
            val = 1;
        } else {
            val = 0;
        }
    }
    
    if (val != sag->dcd_state) {
        if (sag->dcd_state == 1) {
            // DCD dropped
            pp_bmc_serial_mux_switch(PP_SMUX_HOST_CHASSIS);
            pp_bmc_log_info("[SER] DCD dropped");
        } else {
            // DCD now present
            // not of interest right now
            pp_bmc_log_info("[SER] DCD going high");
        }
        sag->dcd_state = val;
    }
    
    return PP_SUC;
}

/**
 * Activate the serial channel. Called by serial_lowlevel after the
 * mux has switched.
 */
static void serial_smux_activate_hndl(pp_bmc_serial_mux_state_t oldstate) {
    assert((serial_adapter_state == PP_SER_ACTIVATING) ||
           (serial_adapter_state == PP_SER_DEACTIVATED));
    
    /* reset internal buffers */
    sag->inbuffer_len = -1;
    sag->last_byte_escaped = 0;
    sag->session_id = 0;
    
    if (serial_channel_config->access_mode == IPMI_CHAN_ACCESS_DISABLE) {
        /* channel is deactivated, do not activate */
        return;
    }
    
    if (oldstate == PP_SMUX_HOST_CHASSIS) {
        // assume that serial_fd is correct open fd from serial_snoop
    } else {
        sag->serial_fd = -1;
    }
    
    if (sag->serial_fd < 0) {
        sag->serial_fd = open(PP_BMC_SERIAL_TTY, O_RDWR | O_NONBLOCK);

        if (sag->serial_fd < 0) {
            pp_bmc_log_perror("[SER] could not open %s", PP_BMC_SERIAL_TTY);
            serial_adapter_state = PP_SER_DEACTIVATED;
            return;
        }
    }
    
    // FIXME use IPMI configuration for serial channels line speed setup
    if (pp_base_set_tty_params(sag->serial_fd, 115200, "", 8, 1, 0, 0) < 0) {
        pp_bmc_log_pwarn("[SER] failed to set tty params");
        // ignore this and try to go without correct tty params
    }
    
    pp_bmc_log_debug("[SER] register serial fd (%d)", sag->serial_fd);
    sag->serial_hndl = pp_select_add_fd(sag->serial_fd, POLLIN,
                                        serial_read, NULL);
    if (sag->serial_hndl < 0) {
        pp_bmc_log_warn("[SER] could not register serial fd in selector");
        return;
    }
    pp_bmc_log_notice("[SER] channel activated");
    serial_adapter_state = PP_SER_ACTIVATED;
    
    serial_send_ping(PING_NO_SESSION);
    sag->ping_hndl = pp_select_add_to(1900, 1, serial_send_ping_timeout, NULL);
    
    sag->dcd_hndl = pp_select_add_to(1000, 1, serial_dcd_timeout, NULL);
}

void bmc_serial_mux_response_received() {
    if ((sag->deactivate_3s_hndl != -1) &
        (serial_adapter_state == PP_SER_DEACTIVATING))
    {
        serial_deactivate_final();
    }
}

static int deactivate_3s_timeout(int id UNUSED, void* ctx UNUSED) {
    // no response, timeout
    serial_deactivate_final();
    sag->deactivate_3s_hndl = -1;
    return PP_SUC;
}

static int deactivate_20ms_timeout(int id UNUSED, void* ctx UNUSED) {
    // repeat mux switch message 3 times
    serial_send_ping(PING_MUX_SWITCH);
    sag->deactivate_count++;
    if (sag->deactivate_count < 3) {
        sag->deactivate_20ms_hndl = pp_select_add_to(20, 0, deactivate_20ms_timeout, NULL);
    } else {
        sag->deactivate_20ms_hndl = -1;
    }
    return PP_SUC;
}

// FIXME: call this function if deactivating and message has been received
// FIXME: what about configuration options

static void serial_smux_deactivate_hndl(pp_bmc_serial_mux_state_t newstate) {
    assert((serial_adapter_state == PP_SER_ACTIVATED) ||
           (serial_adapter_state == PP_SER_DEACTIVATING));
    
    pp_bmc_log_notice("[SER] serial channel deactivated");
    
    if (serial_adapter_state == PP_SER_DEACTIVATING) {
        // we are already deactivating
        return;
    }
    sag->deactivate_newstate = newstate;

    serial_adapter_state = PP_SER_DEACTIVATING;
    
    // FIXME: replace if(true) by correct condition (DCD present ?)
    if (1) {
        // send deactivation message
        sag->deactivate_count = 1;
        sag->deactivate_3s_hndl = pp_select_add_to(3000, 0, deactivate_3s_timeout, NULL);
        sag->deactivate_20ms_hndl = pp_select_add_to(20, 0, deactivate_20ms_timeout, NULL);
        serial_send_ping(PING_MUX_SWITCH);
    } else {
        // deactivate directly
        sag->deactivate_3s_hndl = -1;
        sag->deactivate_20ms_hndl = -1;
        serial_deactivate_final();
    }
}

static void serial_deactivate_final() {
    pp_bmc_serial_mux_state_t newstate = sag->deactivate_newstate;
    pp_select_remove_fd(sag->serial_hndl);
    
    if (sag->deactivate_3s_hndl != -1) {
        pp_select_remove_to(sag->deactivate_3s_hndl);
        sag->deactivate_3s_hndl = -1;
    }
    
    if (sag->deactivate_20ms_hndl != -1) {
        pp_select_remove_to(sag->deactivate_20ms_hndl);
        sag->deactivate_20ms_hndl = -1;
    }

    if (sag->ping_hndl != -1) {
        pp_select_remove_to(sag->ping_hndl);
        sag->ping_hndl = -1;
    }

    if (sag->dcd_hndl != -1) {
        pp_select_remove_to(sag->dcd_hndl);
        sag->dcd_hndl = -1;
    }
    
    if (newstate == PP_SMUX_HOST_CHASSIS) {
        // do not close fd, pass fd to serial_snooop
        bmc_serial_snoop_set_fd(sag->serial_fd);
    } else {
        close(sag->serial_fd);
    }
    sag->serial_fd = -1;
    
    if (sag->session_id != 0) {
        /* we are deactivated but the session might still be open              *
         * close right now and ignore any errors if the session is not present */
        pp_bmc_close_session(sag->session_id);
        sag->session_id = 0;
    }
    
    pp_bmc_serial_mux_deactivate_finished();
    
    pp_bmc_log_notice("[SER] channel deactivated");
    serial_adapter_state = PP_SER_DEACTIVATED;
}


static int serial_read(const int item_id UNUSED, const int fd,
		       const short event UNUSED, void* ctx UNUSED) {
    char buf[60];
    int n;

    if (0 > (n = read(sag->serial_fd, buf, sizeof(buf)))) {
	// read went wrong
	if (errno == EINTR || errno == EAGAIN) {
	    return PP_SUC;
	} else {
	    pp_bmc_log_perror("[SER] read error(fd=%d)", fd);
	    return PP_ERR;
	}
    } else {
	serial_receive_bytes(buf, n);
    }
    return PP_SUC;
}

static int serial_write(char* buf, size_t len) {
    ssize_t n, sent = 0, rmdr = len;
    
    // pp_bmc_printbuf("[SER] sending", buf, len);

    while (rmdr > 0) {
	if (0 > (n = write(sag->serial_fd, &buf[sent], rmdr))) {
	    if (errno == EINTR) {
		continue;
	    } else if (errno == EAGAIN) {
		break;
	    } else {
		pp_bmc_log_perror("[SER] write error(fd=%d)", sag->serial_fd);
		break;
	    }
	}
	rmdr -= n;
	sent += n;
    }
    return sent;
}

/**
 * Low level handler, called everytime that bytes have been received
 * 
 * This functions is responsible for handling the packet framing and
 * restoring escaped characters. It also sends the handshake character
 * once the message is completely received (as we can always handle
 * messages and can receive the next message immediately)
 */
static int serial_receive_bytes(unsigned char* buffer, int length) {
    int i;
    int rv;
    unsigned char c;
    
    // process byte by byte
    for (i=0; i<length; i++) {
        // handle byte buffer[i]

        rv = PP_SUC;
        
        // ignore next character after <ESC>
        if (sag->last_byte_escaped == 2) {
            sag->last_byte_escaped = 0;
        } else
        
        switch (buffer[i]) {   // switch special characters
        
        case 0xa0:
            // start character, always allowed, always restarts message
            if (sag->inbuffer_len != -1) {
                // for protocol debugging only
                pp_bmc_log_debug("[SER] incoming message incomplete. new message started");
            }
            // this is a new message
            sag->inbuffer_len = 0;
            // forget any escape characters - this is a new message
            sag->last_byte_escaped = 0; 
            break;
            
        case 0xa5:
            // stop character
            if (sag->last_byte_escaped == 1) {
                // for protocol debugging only
                pp_bmc_log_debug("[SER] <ESC><StopFrame> sequence found. message ignored");
                sag->inbuffer_len = -1;
                sag->last_byte_escaped = 0;
            }
            else
            {
                if (sag->inbuffer == 0) {
                    // for protocol debugging only
                    pp_bmc_log_debug("[SER] received empty frame - ignored");
                } else {
                    // stop character, msg is complete

                    // send handshake
                    c = 0xa6;
                    serial_write(&c, 1);

                    rv = serial_receive_msg();
                    // reset inbuffer
                    sag->inbuffer_len = -1;
                    //sag->last_byte_escaped = 0 (precondition)
                    
                    break;
                }
            }

        case 0xa6:
            // packet handshake character
            if (sag->last_byte_escaped == 1) {
                // for protocol debugging only
                pp_bmc_log_debug("[SER] <escape><handshake> sequence found. message ignored");
                sag->inbuffer_len = -1;
                sag->last_byte_escaped = 0;
            } else {
                // Handshakes go from BMC to terminal, but it should be save to ignore this
            }
            break;

        case 0xaa:
            // escape character
            if (sag->last_byte_escaped == 1) {
                // for protocol debugging only
                pp_bmc_log_debug("[SER] <escape><escape> sequence found. message ignored");
                sag->inbuffer_len = -1;
                sag->last_byte_escaped = 0;
            } else {
                sag->last_byte_escaped = 1;
            }
            break;
            
        case 0x1b:
            // microsoft console escape character, ignore next character
            if (sag->last_byte_escaped == 1) {
                // for protocol debugging only
                pp_bmc_log_debug("[SER] <escape><ESC> sequence found. message ignored");
                sag->inbuffer_len = -1;
                sag->last_byte_escaped = 0;
            } else {
                sag->last_byte_escaped = 2;
            }
            break;

        default:
            // regular character
            
            if (sag->inbuffer_len == -1) {
                // very strange, characters without frame
                // for protocol debugging only
                pp_bmc_log_debug("[SER] received character outside framed message - ignored");
            }
            else
            if (sag->last_byte_escaped == 1) {
                // allow escaped characters only
                switch(buffer[i]) {
                case 0xb0:
                    sag->inbuffer[sag->inbuffer_len] = 0xa0;
                    sag->inbuffer_len++;
                    break;

                case 0xb5:
                    sag->inbuffer[sag->inbuffer_len] = 0xa5;
                    sag->inbuffer_len++;
                    break;

                case 0xb6:
                    sag->inbuffer[sag->inbuffer_len] = 0xa6;
                    sag->inbuffer_len++;
                    break;

                case 0xba:
                    sag->inbuffer[sag->inbuffer_len] = 0xaa;
                    sag->inbuffer_len++;
                    break;
                
                case 0x3b:
                    sag->inbuffer[sag->inbuffer_len] = 0x1b;
                    sag->inbuffer_len++;
                    break;

                default:
                    // for protocol debugging only
                    sag->inbuffer_len = -1;
                    pp_bmc_log_debug("[SEL] invalid <escape><0x%.02x> sequence received. frame dropped", buffer[i]);
                    break;
                }
                
                sag->last_byte_escaped = 0;
            }
            else
            {
                // last character was neither special nor escaped
                sag->inbuffer[sag->inbuffer_len] = buffer[i];
                sag->inbuffer_len ++;
            } // end if (msgexists) else (escaped) else (regular char) 
        } // end switch (controlchar)
        
        if (sag->inbuffer_len == MAX_IPMI_MSG_LEN) {
            // our message buffer is full but the message did not end. drop msg.
            // for protocol debugging only
            pp_bmc_log_debug("[SEL] dropped unfinished message due to maximum size constraints");
            sag->inbuffer_len = -1;
            sag->last_byte_escaped = 0;
        }
        
    } // end for all received bytes
    
    return PP_SUC;
}

int pp_bmc_serial_receive_msg(unsigned char* buf, int len) {
    int rv;
    
    /* Watch out, this will overwrite any pending message */
    assert(len < MAX_IPMI_MSG_LEN);
    
    memcpy(sag->inbuffer, buf, len);
    sag->inbuffer_len = len;
    
    rv = serial_receive_msg();
    sag->inbuffer_len = -1;
    
    return rv;
}

/**
 * Called whenever a new message is available in inbuffer
 * 
 * Extract the message contents, build imsg and forward it to the core.
 */
static int serial_receive_msg() {
    // minimum size of request messages
    if (sag->inbuffer_len < 7) {
        pp_bmc_log_debug("[SER] incoming buffer to short for imsg");
        return PP_ERR;
    }
    
    /*
    int i;
    printf("received <"); 
    for (i=0; i<sag->inbuffer_len; i++) {
        printf("%.2x ", sag->inbuffer[i]);
    } printf(">\n");
    */
    
    // check checksums
    if (sag->inbuffer[2] != pp_bmc_calc_ipmi_checksum(sag->inbuffer,0,1)) {
        // for protocol debugging only
        //pp_bmc_log_info("[SER] received msg with invalid checksum 1");
        return PP_ERR;
    }
    if (sag->inbuffer[sag->inbuffer_len-1] != pp_bmc_calc_ipmi_checksum(sag->inbuffer,3,sag->inbuffer_len-2)) {
        // for protocol debugging only
        //pp_bmc_log_info("[SER] received msg with invalid checksum 2");
        return PP_ERR;
    }

    // create imsg
    imsg_t *imsg;
    serial_channeldata_t *scdata;

    imsg = pp_bmc_imsg_new(sag->inbuffer_len - 7);

    imsg->buf = malloc(sizeof(serial_channeldata_t));
    imsg->buf_size = sizeof(serial_channeldata_t);
    scdata = (serial_channeldata_t*)(imsg->buf);
    
    imsg->rs_addr =  sag->inbuffer[0];
    imsg->netfn   = (sag->inbuffer[1] >> 2) & 0x3F;
    imsg->rs_lun  = (sag->inbuffer[1]) & 0x03;
    imsg->rq_addr =  sag->inbuffer[3];
    scdata->ipmi_msg_sequence = (sag->inbuffer[4] >> 2) & 0x3F;
    imsg->rq_lun  = (sag->inbuffer[4]) & 0x03;
    
    imsg->cmd     =  sag->inbuffer[5];
    memcpy(imsg->data, sag->inbuffer+6, imsg->data_size);
    
    // other values in imsg
    imsg->chan = IPMI_CHAN_SERIAL;
    
    if (sag->session_id != 0) {
        imsg->session = pp_bmc_get_session(sag->session_id);
        if (imsg->session == NULL) {
            // it seems that our session timed out
            sag->session_id = 0;
        }
    }
    if (imsg->session != NULL) {
        imsg->priv_level = imsg->session->cur_priv_level;
    } else {
        imsg->priv_level = IPMI_PRIV_UNSPEC;
    }

    pp_bmc_core_handle_msg(imsg);
    
    return PP_SUC;
}


/**
 * Low level handler for framing and escaping message buffers before sending.
 */
static int serial_send_msg_buffer(unsigned char* msgbuffer, int msglen) {
    // escape characters and frame packet
    unsigned char buffer[MAX_IPMI_MSG_LEN*2];
    int len;
    int i;
    
    len = 0;
    buffer[0] = 0xa0; // packet start
    len++;
    for (i=0; i<msglen; i++) {
        if ((msgbuffer[i] == 0xa0) || (msgbuffer[i] == 0xa5) ||
            (msgbuffer[i] == 0xa6) || (msgbuffer[i] == 0xaa) || 
            (msgbuffer[i] == 0x1b))
        {
            buffer[len] = 0xaa;
            len++;

            if (msgbuffer[i] == 0xa0)
                buffer[len] = 0xb0;
            if (msgbuffer[i] == 0xa5)
                buffer[len] = 0xb5;
            if (msgbuffer[i] == 0xa6)
                buffer[len] = 0xb6;
            if (msgbuffer[i] == 0xaa)
                buffer[len] = 0xba;
            if (msgbuffer[i] == 0x1b)
                buffer[len] = 0x3b;

            len++;
        }
        else
        {
            buffer[len] = msgbuffer[i];
            len++;
        }
        
    }
    buffer[len] = 0xa5; // packet stop
    len++;

    // FIXME: serial fd is non-blocking, so serial_write may return without having written all data
    return serial_write(buffer, len);
}

/**
 * High level handler, called by the bmccore/msgrouter to send
 * messages to this channel.
 * 
 * Constructs the message form the imsg and uses serial_send_msg_buffer
 * to send the message.
 */
static int serial_send_imsg(imsg_t *imsg) {
    unsigned char msgbuffer[MAX_IPMI_MSG_LEN];
    serial_channeldata_t *scdata;
    int msglen;
    unsigned int* uip;

    /* snoop if this is a successful activate session response *
     * and extract the new session id if this is the case      */
    if ( (imsg->cmd == IPMI_CMD_GET_SESSION_CHALLENGE) &&
         (imsg->netfn == (IPMI_NETFN_APP | 0x01)) && 
         (imsg->data[0] == IPMI_ERR_SUCCESS) )
    {
        assert(imsg->data_size == 21);
        uip = (unsigned int*)(imsg->data+1);
        sag->session_id = le32_to_cpu(*uip);
        pp_bmc_log_debug("[SER] retrieving new session id %u from activate session command", sag->session_id);
    }
     
    /* snoop if this is a successful close session response */
    if ( (imsg->cmd == IPMI_CMD_CLOSE_SESSION) &&
         (imsg->netfn == (IPMI_NETFN_APP | 0x01)) && 
         (imsg->data[0] == IPMI_ERR_SUCCESS) )
    {
        sag->session_id = 0;
        pp_bmc_log_debug("[SER] session closed");
    }
     
    scdata = (serial_channeldata_t*) imsg->buf;
    
    // build imsg into msgbuffer
    // caveat: the completion code is part of imsg->data
    msglen = imsg->data_size+7;
    msgbuffer[0] = imsg->rq_addr;
    msgbuffer[1] = (imsg->netfn << 2) | imsg->rq_lun;
    msgbuffer[2] = pp_bmc_calc_ipmi_checksum(msgbuffer,0,1);
    msgbuffer[3] = imsg->rs_addr;
    msgbuffer[4] = (scdata->ipmi_msg_sequence << 2) | imsg->rs_lun;
    msgbuffer[5] = imsg->cmd;
    // msgbuffer[6] = imsg->data[0];   // this byte is copied with data
    memcpy(msgbuffer+6, imsg->data, imsg->data_size);
    msgbuffer[imsg->data_size+6] = pp_bmc_calc_ipmi_checksum(msgbuffer,3,imsg->data_size+5);
        
    pp_bmc_imsg_delete(imsg);
    
    return serial_send_msg_buffer(msgbuffer, msglen);
}

static const pp_bmc_router_chan_adapter_info_t ser_chan_info = {
    .medium =       IPMI_CHAN_MED_SERIAL,
    .proto =        IPMI_CHAN_PROT_IPMB,
    .sess_supp =    IPMI_CHAN_SESS_SINGLE,
};

/**
 * Set serial access mode. Valid values are 'disabled' and 
 * 'shared'. 'pre-boot' and 'always' are not supported.
 * This function must not be called with a invalid access mode.
 */
static void bmc_serial_set_access_mode(unsigned char mode) {
    if (pp_bmc_serial_mux_get_state() == PP_SMUX_BMC_CHASSIS) {
        // mux is switched to this channel, (de)activate
        
        switch (mode) {
        case IPMI_CHAN_ACCESS_DISABLE:
            if (sag->serial_fd >= 0) {
                // we have to deactivate
                // do it the hard way - no notifications
                serial_adapter_state = PP_SER_DEACTIVATED;
                pp_select_remove_fd(sag->serial_hndl);
                sag->serial_hndl = -1;
                pp_bmc_close_session(sag->session_id);
                sag->session_id = 0;
                close(sag->serial_fd);
                sag->serial_fd = -1;
                if (sag->ping_hndl != -1) {
                    pp_select_remove_to(sag->ping_hndl);
                    sag->ping_hndl = -1;
                }
                if (sag->dcd_hndl != -1) {
                    pp_select_remove_to(sag->dcd_hndl);
                    sag->dcd_hndl = -1;
                }
            }
            // else: mux is switched, but we have no open serial port - strange
            break;
        case IPMI_CHAN_ACCESS_SHARED:
            if ((serial_adapter_state == PP_SER_DEACTIVATED) ||
                (serial_adapter_state == PP_SER_ACTIVATING))
            {
                // we have to activate
                serial_smux_activate_hndl(PP_SMUX_HOST_CHASSIS);
            }
            break;
        default:
            // unsupported mode, must not happen
            assert(0);
        }
    }
    // else: no immediate action needed
}

/**
 * Set channel configuration parameters. 
 * - PEF alerting disable is ignored (not applicable)
 * - per message authentication must be disabled (this is a secure channel),
 *   ignore if enabled
 * - per user authentication is ignored (as per_msg_auth applies)
 * - access mode: ses bmc_serial_set_access_mode()
 * - channel_priv_limit can be applied regularly
 */
static void bmc_serial_config(void) {
    bmc_serial_set_access_mode(serial_channel_config->access_mode);
    bmc_serial_snoop_set_access_mode(serial_channel_config->access_mode);
}

int pp_bmc_serial_init() {
    assert(sag == NULL);
    sag = &sa_globals;
    sag->inbuffer_len = -1;
    sag->last_byte_escaped = 0;
    sag->serial_fd = -1;
    sag->serial_hndl = -1;
    sag->session_id = 0;
    sag->ping_hndl = -1;
    sag->dcd_hndl = -1;
    sag->dcd_state = 0;

    // register the channel
    serial_channel_config = pp_bmc_router_reg_chan(
                                IPMI_CHAN_SERIAL,
                                serial_send_imsg,
                                NULL,
                                bmc_serial_config,
                                &ser_chan_info);
    if (serial_channel_config == NULL) {
        return PP_ERR;
    }

    if (pp_serial_modem_cmds_init() == PP_ERR) {
        return PP_ERR;
    }
    
    if (pp_serial_snoop_init() == PP_ERR) {
        return PP_ERR;
    }
    
    /* register smux-change handler and setup for initial state */
    pp_bmc_serial_mux_register_handler(PP_SMUX_BMC_CHASSIS,
                                       serial_smux_activate_hndl,
                                       serial_smux_deactivate_hndl);
    pp_bmc_log_info("[SER] channel initialized");

    return PP_SUC;
}

void pp_bmc_serial_cleanup()  {
    pp_serial_modem_cmds_cleanup();
    
    pp_bmc_router_unreg_chan(IPMI_CHAN_SERIAL);
    
    pp_serial_snoop_cleanup();
    
    sag = NULL;
    pp_bmc_log_info("[SER] channel shut down");
}
