/**
 * serial_modem_cmds.c
 *
 * IPMI Serial/Modem Commands
 *
 * (c) 2005 Peppercon AG, Thomas Weber <tweb@peppercon.de>
 */

#include <pp/base.h>

#include <pp/cfg.h>
#include <pp/xdefs.h>

#include <pp/bmc/debug.h>

#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/serial_adapter.h>

#include "serial_modem_cmds.h"

static unsigned char serial_modem_set_in_progress = 0x00; /* set complete */

static const char *bmc_serial_active_message = "bmc.serial.active_message";
static const char *bmc_serial_auto_mux_to_bmc = "bmc.serial.auto_mux_to_bmc";
static const char *bmc_serial_gcac_snoop = "bmc.serial.gcac_snoop";
static const char *bmc_serial_mux_acknowledge = "bmc.serial.mux_acknowledge";
static const char *bmc_serial_port_sharing = "bmc.serial.port_sharing";

/* implemented configuration types */
#define SERIAL_MODEM_AUTH_IMPLEMENTED 0x17 /* none, MD2, MD5, plain */

/**
 * MUX switch control settings
 * mask out reserved / unsupported bits
 *
 * byte 1:
 * [7]:   reserved
 * [6]:   disable system power-up / wakeup via MSVT 
 * [5]:   disable hard reset on MSVT
 * [3]:   disable switch to BMC on PPP-RMCP pattern
 * [2]:   disable BMC-to-baseboard switch on MSVT
 * [1]:   disable baseboard-to-BMC switch on MSVT
 * byte 2: 
 * [7:4]: reserved
 * [2]:   disable serial / modem connection alive message during callback
 */
#define SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_1 0x11
#define SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_2 0x0B

#define CHECK_PDATA_SIZE(s) \
    if (psize < s) { ierr = IPMI_ERR_REQ_DATA_LEN_INVALID; break; }

#define SET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[SER] can't write param #%d to cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int set_serial_modem_config(imsg_t *imsg) {
    /* IPMI message data */
    unsigned char channel = imsg->data[0] & 0x0f;
    unsigned char param = imsg->data[1];
    unsigned char *pdata = &imsg->data[2];
    int psize = imsg->data_size - 2;

    unsigned char ierr = IPMI_ERR_SUCCESS;
    int dirty = 0;
    pp_cfg_tx_t* cfg_trans;

    if(channel != IPMI_CHAN_SERIAL) {
        pp_bmc_log_warn("[SER] invalid serial channel id %u, "
                        "should be %u", channel, IPMI_CHAN_SERIAL);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    cfg_trans = pp_cfg_tx_begin(1);
    if (!cfg_trans) {
        pp_bmc_log_error("[SER] can't create cfg transaction");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    pp_bmc_log_info("[SER] set param %u", param);
    switch (param) {
        case  0: /* set in progress */
            CHECK_PDATA_SIZE(1);
            switch (pdata[0] & 0x3) {
                case 0: /* set complete */
                    serial_modem_set_in_progress = 0;
                    break;
                case 1: /* set in progress */
                    if (serial_modem_set_in_progress == 1) {
                        ierr = IPMI_ERR_SERIAL_MODEM_SET_IN_PROGRESS;
                    }
                    serial_modem_set_in_progress = 1;
                    break;
                default:
                    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            }
            break;
        case  1: /* authentication type support */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
        case  2: /* authentication type enables */
            CHECK_PDATA_SIZE(5);

            /* check pdata for correctness */
            if ((pdata[0] & (~SERIAL_MODEM_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[1] & (~SERIAL_MODEM_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[2] & (~SERIAL_MODEM_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[3] & (~SERIAL_MODEM_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if (pdata[4] != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if (ierr != IPMI_ERR_SUCCESS) break;
            
            /* pdata is correct, write to config fs */
            SET_CFG(pp_cfg_set_short(pdata[0], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR));
            SET_CFG(pp_cfg_set_short(pdata[1], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_USER_STR));
            SET_CFG(pp_cfg_set_short(pdata[2], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR));
            SET_CFG(pp_cfg_set_short(pdata[3], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR));
            /* imsg->pdata[4] (OEM) is ignored */
            dirty = 1;
            break;
        case  3: /* connection mode */
	    /* read only parameter (not specified!) */
#ifdef PP_BMC_ICTS_CONFORM
	    ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
#else
	    ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
#endif
            break;
//        case  4: /* session inactivity timeout */
            /* parameter not supported (at the moment) */
        case  5: // Fallthrough /* channel callback control */
        case  6: /* session termination */
            /* read only parameter (not specified!) */
#ifdef PP_BMC_ICTS_CONFORM
	    ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
#else
	    ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
#endif
            break;
//        case  7: /* IPMI messaging comm settings */
            /* parameter not supported (at the moment) */
        case  8: /* mux switch control */
            CHECK_PDATA_SIZE(2);
            
            SET_CFG(pp_cfg_set_enabled(pdata[0] & 0x10, bmc_serial_gcac_snoop));
            SET_CFG(pp_cfg_set_enabled(pdata[0] & 0x01, bmc_serial_auto_mux_to_bmc));
            
            //  Enable this bit once set_serial_modem_mux() is implemented!
            //pp_cfg_set_enabled(pdata[1] & 0x08, bmc_serial_port_sharing);
            SET_CFG(pp_cfg_set_enabled(pdata[1] & 0x02, bmc_serial_active_message));
            SET_CFG(pp_cfg_set_enabled(pdata[1] & 0x01, bmc_serial_mux_acknowledge));
            dirty = 1;
            break;
//        case  9: /* modem ring time */
            /* parameter not supported (at the moment) */
//        case 10: /* modem init string */
            /* parameter not supported (at the moment) */
//        case 11: /* modem escape sequence */
            /* parameter not supported (at the moment) */
//        case 12: /* modem hang-up sequence */
            /* parameter not supported (at the moment) */
//        case 13: /* modem dial command */
            /* parameter not supported (at the moment) */
//        case 14: /* page blackout interval */
            /* parameter not supported (at the moment) */
//        case 15: /* community string */
            /* parameter not supported (at the moment) */
        case 16: /* number of alert destinations */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 17: /* destination info */
            /* parameter not supported (at the moment) */
//        case 18: /* call retry interval */
            /* parameter not supported (at the moment) */
//        case 19: /* destination comm settings */
            /* parameter not supported (at the moment) */
        case 20: /* number of dial strings */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 21: /* destination dial strings */
            /* parameter not supported (at the moment) */
        case 22: /* number of alert destination IP addresses */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 23: /* destination IP addresses */
            /* parameter not supported (at the moment) */
        case 24: /* number of TAP accounts */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 25: /* TAP account */
            /* parameter not supported (at the moment) */
//        case 26: /* TAP passwords */
            /* parameter not supported (at the moment) */
//        case 27: /* TAP pager ID strings */
            /* parameter not supported (at the moment) */
//        case 28: /* TAP service settings */
            /* parameter not supported (at the moment) */
//        case 29: /* terminal mode configuration */
            /* parameter not supported (at the moment) */
//        case 30: /* PPP protocol options */
            /* parameter not supported (at the moment) */
//        case 31: /* PPP primary RMCP port number */
            /* parameter not supported (at the moment) */
//        case 32: /* PPP secondary RMCP port number */
            /* parameter not supported (at the moment) */
//        case 33: /* PPP link authentication */
            /* parameter not supported (at the moment) */
//        case 34: /* CHAP name */
            /* parameter not supported (at the moment) */
//        case 35: /* PPP ACCM */
            /* parameter not supported (at the moment) */
//        case 36: /* PPP snoop ACCM */
            /* parameter not supported (at the moment) */
        case 37: /* number of PPP accounts */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 38: /* PPP account dial string selector */
            /* parameter not supported (at the moment) */
//        case 39: /* PPP account IP addresses, BMC IP address */
            /* parameter not supported (at the moment) */
//        case 40: /* PPP account user names */
            /* parameter not supported (at the moment) */
//        case 41: /* PPP account user domains */
            /* parameter not supported (at the moment) */
//        case 42: /* PPP account user passwords */
            /* parameter not supported (at the moment) */
//        case 43: /* PPP account authentication settings */
            /* parameter not supported (at the moment) */
//        case 44: /* PPP account connection hold times */
            /* parameter not supported (at the moment) */
//        case 45: /* PPP UDP proxy IP header pdata */
            /* parameter not supported (at the moment) */
        case 46: /* PPP UDP proxy transmit buffer size */
        case 47: /* PPP UDP proxy receive buffer size */
            ierr = IPMI_ERR_SERIAL_MODEM_READ_ONLY;
            break;
//        case 48: /* PPP remote console IP address */
            /* parameter not supported (at the moment) */
//        case 49: /* system phone number */
            /* parameter not supported (at the moment) */
        default:
            ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
    }

    if (dirty && ierr == IPMI_ERR_SUCCESS) {
        if (PP_FAILED(pp_cfg_tx_commit(cfg_trans))) {
            pp_bmc_log_error("[SER] can't commit cfg transaction");
            ierr = IPMI_ERR_UNSPECIFIED;
        }
        pp_cfg_save(DO_FLUSH);
    } else {
        pp_cfg_tx_rollback(cfg_trans);
    }

    return pp_bmc_router_resp_err(imsg, ierr);
}

#define GET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[SER] can't read param #%d from cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int get_serial_modem_config(imsg_t *imsg) {
     /* local data buffer */
    unsigned char pdata[256];
    int psize = 0;

    /* IPMI message data */
    unsigned char channel = imsg->data[0] & 0x0f;
    unsigned char param = imsg->data[1];
    char *resp;
    unsigned char ierr = IPMI_ERR_SUCCESS;

    unsigned short us;
    int i;

    if(channel != IPMI_CHAN_SERIAL) {
        pp_bmc_log_warn("[SER] invalid serial channel id %u, "
                        "should be %u", channel, IPMI_CHAN_SERIAL);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    if(imsg->data[0] & 0x80) {
        // return param rev only
        psize = 0;
    } else {
        pp_bmc_log_info("[SER] get param %u", param);
        switch (param) {
            case  0: /* set in progress */
                psize = 1;
                pdata[0] = serial_modem_set_in_progress & 0x03;
                break;
            case  1: /* authentication type support */
                psize = 1;
                pdata[0] = SERIAL_MODEM_AUTH_IMPLEMENTED;
                break;
            case  2: /* authentication type enables */
                psize = 5;
                pp_cfg_get_short_nodflt(&us, "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR);
                pdata[0] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us, "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_USER_STR);
                pdata[1] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us, "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR);
                pdata[2] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us, "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR);
                pdata[3] = 0x37 & us;
                pdata[4] = 0; /* OEM login not supported, maybe we should store dummy values here */
                break;
            case  3: /* connection mode */
#ifdef PP_BMC_ICTS_CONFORM
                ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
#else
                psize = 1;
                pdata[0] = 0x81; /* we only support direct connect / basic mode */
#endif
                break;
//            case  4: /* session inactivity timeout */
                /* parameter not supported (at the moment) */
            case  5: /* channel callback control */
#ifdef PP_BMC_ICTS_CONFORM
                // ICTS can't deal with read-only supported parameters
                ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
#else
                psize = 5;
                pdata[0] = 0x00; /* callback disabled */
                pdata[1] = 0x01; /* only enable "no callback" */
                pdata[2] = 0xff; /* callback destination 1 unspecified */
                pdata[3] = 0xff; /* callback destination 2 unspecified */
                pdata[4] = 0xff; /* callback destination 3 unspecified */
#endif
                break;
            case  6: /* session termination */
#ifdef PP_BMC_ICTS_CONFORM
                // ICTS can't deal with read-only supported parameters
                ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
#else
                psize = 1;
                pdata[0] = 0x01; /* no inactivity timeout, close on loss of DCD */
#endif
                break;
//            case  7: /* IPMI messaging comm settings */
                /* parameter not supported (at the moment) */
            case  8: /* mux switch control */
                psize = 2;

                memset(pdata, 0, 2); // init

                /* mask out reserved / unsupported bits (see description) */
                pp_cfg_is_enabled(&i, bmc_serial_gcac_snoop);
                pdata[0] |= (i << 4) & SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_1;
                pp_cfg_is_enabled(&i, bmc_serial_auto_mux_to_bmc);
                pdata[0] |= i & SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_1;

                pp_cfg_is_enabled(&i, bmc_serial_port_sharing);
                pdata[1] |= (i << 3) & SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_2;
                pp_cfg_is_enabled(&i, bmc_serial_active_message);
                pdata[1] |= (i << 1) & SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_2;
                pp_cfg_is_enabled(&i, bmc_serial_mux_acknowledge);
                pdata[1] |= i & SERIAL_MODEM_MUX_SWITCH_CONTROL_MASK_2;

                break;
//            case  9: /* modem ring time */
                /* parameter not supported (at the moment) */
//            case 10: /* modem init string */
                /* parameter not supported (at the moment) */
//            case 11: /* modem escape sequence */
                /* parameter not supported (at the moment) */
//            case 12: /* modem hang-up sequence */
                /* parameter not supported (at the moment) */
//            case 13: /* modem dial command */
                /* parameter not supported (at the moment) */
//            case 14: /* page blackout interval */
                /* parameter not supported (at the moment) */
//            case 15: /* community string */
                /* parameter not supported (at the moment) */
            case 16: /* number of alert destinations */
                psize = 1;
                pdata[0] = 0x00; /* page alerting not supported */
                break;
//            case 17: /* destination info */
                /* parameter not supported (at the moment) */
//            case 18: /* call retry interval */
                /* parameter not supported (at the moment) */
//            case 19: /* destination comm settings */
                /* parameter not supported (at the moment) */
            case 20: /* number of dial strings */
                psize = 1;
                /* serial / modem alerting and callback not supported */
                pdata[0] = 0x00;
                break;
//            case 21: /* destination dial strings */
                /* parameter not supported (at the moment) */
            case 22: /* number of alert destination IP addresses */
                psize = 1;
                pdata[0] = 0x00; /* PPP alerting and callback not supported */
                break;
//            case 23: /* destination IP addresses */
                /* parameter not supported (at the moment) */
            case 24: /* number of TAP accounts */
                psize = 1;
                pdata[0] = 0x00; /* TAP not supported */
                break;
//            case 25: /* TAP account */
                /* parameter not supported (at the moment) */
            case 26: /* TAP passwords */
                ierr = IPMI_ERR_SERIAL_MODEM_WRITE_ONLY;
                break;
//            case 27: /* TAP pager ID strings */
                /* parameter not supported (at the moment) */
//            case 28: /* TAP service settings */
                /* parameter not supported (at the moment) */
//            case 29: /* terminal mode configuration */
                /* parameter not supported (at the moment) */
//            case 30: /* PPP protocol options */
                /* parameter not supported (at the moment) */
//            case 31: /* PPP primary RMCP port number */
                /* parameter not supported (at the moment) */
//            case 32: /* PPP secondary RMCP port number */
                /* parameter not supported (at the moment) */
//            case 33: /* PPP link authentication */
                /* parameter not supported (at the moment) */
//            case 34: /* CHAP name */
                /* parameter not supported (at the moment) */
//            case 35: /* PPP ACCM */
                /* parameter not supported (at the moment) */
//            case 36: /* PPP snoop ACCM */
                /* parameter not supported (at the moment) */
            case 37: /* number of PPP accounts */
                psize = 1;
                pdata[0] = 0x00; /* PPP alerting and callback not supported */
                break;
//            case 38: /* PPP account dial string selector */
                /* parameter not supported (at the moment) */
//            case 39: /* PPP account IP addresses, BMC IP address */
                /* parameter not supported (at the moment) */
//            case 40: /* PPP account user names */
                /* parameter not supported (at the moment) */
//            case 41: /* PPP account user domains */
                /* parameter not supported (at the moment) */
            case 42: /* PPP account user passwords */
                ierr = IPMI_ERR_SERIAL_MODEM_WRITE_ONLY;
                break;
//            case 43: /* PPP account authentication settings */
                /* parameter not supported (at the moment) */
//            case 44: /* PPP account connection hold times */
                /* parameter not supported (at the moment) */
//            case 45: /* PPP UDP proxy IP header pdata */
                /* parameter not supported (at the moment) */
            case 46: /* PPP UDP proxy transmit buffer size */
            case 47: /* PPP UDP proxy receive buffer size */
                psize = 2;
                pdata[0] = 0x00; /* PPP UDP proxy not supported */
                pdata[1] = 0x00;
                break;
//            case 48: /* PPP remote console IP address */
                /* parameter not supported (at the moment) */
//            case 49: /* system phone number */
                /* parameter not supported (at the moment) */
            default:
                ierr = IPMI_ERR_SERIAL_MODEM_PARAM_NOT_SUPPORTED;
        }
    }

    if (ierr != IPMI_ERR_SUCCESS) {
        return pp_bmc_router_resp_err(imsg, ierr);
    }

    resp = pp_bmc_imsg_resp(imsg, ierr, psize + 1);
    resp[0] = 0x11; // parameter revision
    memcpy(resp + 1, pdata, psize);

    return pp_bmc_router_send_msg(imsg);
}

static int set_serial_modem_mux(imsg_t *imsg) { /* 25.3 */
//    unsigned char channel = imsg->data[1] & 0x0f; // mask out reserved bits
//    unsigned char setting = imsg->data[2] & 0x0f; // mask out reserved bits
    
    pp_bmc_log_debug("[SER] set_serial_modem_mux called, "
                     "but not yet implemented");
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
}

static int serial_modem_connection_alive(imsg_t *imsg) { /* 25.9 */
    // FIXME: this command is a response, does the bmc_router handle this correctly ?
    
    /* this is a _response_ command _to_ the bmc (bmc sent the request) */
    bmc_serial_mux_response_received();
    
    /* we dont have to return a response */
    pp_bmc_imsg_delete(imsg);
    return PP_SUC;
}

/********************************************************************
 * App device channel command table
 */

static const dev_cmd_entry_t serial_modem_cmd_tab[] = {
    {
        .cmd_hndlr = set_serial_modem_config,
        .netfn = IPMI_NETFN_TRANSPORT, 
        .cmd = IPMI_CMD_SET_SERIAL_MODEM_CONF,
        .min_data_size = 2, 
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = get_serial_modem_config,
        .netfn = IPMI_NETFN_TRANSPORT, 
        .cmd = IPMI_CMD_GET_SERIAL_MODEM_CONF,
        .min_data_size = 4, 
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = set_serial_modem_mux,
        .netfn = IPMI_NETFN_TRANSPORT, 
        .cmd = IPMI_CMD_SET_SERIAL_MODEM_MUX,
        .min_data_size = 2, 
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = serial_modem_connection_alive,
        .netfn = IPMI_NETFN_TRANSPORT, 
        .cmd = IPMI_CMD_SERIAL_MODEM_CONNECTION_ALIVE,
        .min_data_size = 2, 
        .min_priv_level = IPMI_PRIV_UNSPEC
    },

    { .cmd_hndlr = NULL }
};

/********************************************************************
 * App device c'tor/d'tor
 */

int pp_serial_modem_cmds_init(void)
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(serial_modem_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[SER] device started");
    return PP_SUC;
}

void pp_serial_modem_cmds_cleanup(void)
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(serial_modem_cmd_tab);

    pp_bmc_log_info("[SER] device shut down");
}
