/**
 * smi_loop_adapter.c
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Uses the bmccore-msgrouter and the ipmi_bmc_loop.ko Kernelmodule
 * to connect the bmccore to the openIPMI lib over the SMI.
 *
 * Please load the ipmi_bmc_loop.ko module before initializing this adapter.
 *
 * Please create a character device /dev/ipmi_bmc_si before initializing.
 * The major number can be queried with 'grep ipmi_bmc_si /proc/device', 
 * the minor numer is 0.
 */

#include "pp/bmc/debug.h"

#include "pp/base.h"
#include "pp/selector.h"
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/smi_adapter.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/ioctl.h>
#include <sys/ioctl.h>

#include "ipmi_bmc_si.h"               // the functions that we need
#include "smi_loop_adapter.h"   // the functions that we offer
#include "lpc.h"                // PP_BMC_SMI_MAX_MSG_LEN
/**
 * the global variables
 */
typedef struct {
   int fd;            // the file descriptor of our si connection
   int selID;         // our selector entry
} smi_loop_adapter_globals;

static smi_loop_adapter_globals *sla_globals;
static smi_loop_adapter_globals my_sla_globals;


/**
 * Called by the bmccore/msgrouter to send messages to the smi
 */
static int  smi_receive_bmc_msg(imsg_t *imsg) {
    char receiveData[PP_BMC_SMI_MAX_MSG_LEN];
    bmc_si_msg_t msg;
    int rv;
    
    msg.data = receiveData;
    msg.data_size = PP_BMC_SMI_MAX_MSG_LEN;
    msg.data[0] = (imsg->netfn<<2) | (imsg->rs_lun & 0x03);
    msg.data[1] = imsg->cmd;
    memcpy(((msg.data)+2), imsg->data, imsg->data_size);
    msg.data_size = imsg->data_size+2;
            
#if PP_BMC_DEBUG_LEVEL >= 8
    {
        int i;
        printf("Sending <");
        for (i=0; i<msg.data_size; i++) {
        printf("%2x ", msg.data[i]);
        }
        printf(">\n");
    }
#endif

    rv = ioctl(sla_globals->fd, IPMI_BMC_SEND, (void*) &msg);
    // We cannot recover from problems here, so at least print an error
    if (rv != 0) {
        pp_bmc_log_error("[smi_loop_adapter] could not send message, IOctl failed - %d", rv);
    }
    
    pp_bmc_imsg_delete(imsg);
    
    return PP_SUC;
 }

/**
 * called by the selector to indicate that new messages are available on the
 * si. The message is read and forwarded to the core.
 */
static int smi_receive_si_msg(UNUSED const int item_id, const int fd, UNUSED const short event, UNUSED void *context) {
    char receiveData[PP_BMC_SMI_MAX_MSG_LEN];
    bmc_si_msg_t msg;
    int rv;
    imsg_t *msgBMC;
    
    msg.data = receiveData;
    msg.data_size = PP_BMC_SMI_MAX_MSG_LEN;
    rv = ioctl(fd, IPMI_BMC_RECEIVE, (void*) &msg);

    if (rv != 0) {
        pp_bmc_log_warn("[SMI_loop] receive-ioctl failed: %d", rv);
	return PP_ERR;
    }
#if PP_BMC_DEBUG_LEVEL >= 8
    if ((msg.data_size > 0) & (msg.data_size < PP_BMC_SMI_MAX_MSG_LEN)) {
        int i;
        printf("Received <");
        for (i=0; i<msg.data_size; i++) {
            printf("%2x ", msg.data[i]);
        }
        printf(">\n");
    }
#endif
    
    // convert internal message from si data
    if (msg.data_size < 2) {
      pp_bmc_log_warn("[SMI_loop] ignored too short message");
      return PP_ERR;
    }
    
    msgBMC = pp_bmc_imsg_new(msg.data_size - 2);
    msgBMC->netfn = msg.data[0] >> 2;
    msgBMC->rq_lun = msg.data[0] & 3;
    msgBMC->cmd = msg.data[1];
    memcpy (msgBMC->data, (msg.data + 2), msgBMC->data_size);
    // add other information
    msgBMC->chan = IPMI_CHAN_SI;
    msgBMC->priv_level = IPMI_PRIV_ADMIN;
    msgBMC->buf = NULL;
    msgBMC->buf_size = 0;
    msgBMC->rs_lun = 0;
    msgBMC->session = NULL;
    
    pp_bmc_core_handle_msg(msgBMC);
    
    return 0;
}

static const pp_bmc_router_chan_adapter_info_t smi_loop_chan_info = {
    .medium =       IPMI_CHAN_MED_SI,
    .proto =        IPMI_CHAN_PROT_KCS,
    .sess_supp =    IPMI_CHAN_SESS_LESS,
};

int pp_smi_loop_init() {
    // use malloc here if structure gets too big
    sla_globals = &my_sla_globals;
    
    pp_bmc_log_debug("[SMI_loop] Initializing openIpmi SI loop adapter");
    
    if (pp_bmc_router_reg_chan(IPMI_CHAN_SI, smi_receive_bmc_msg, NULL, NULL, &smi_loop_chan_info) == NULL) {
        pp_bmc_log_error("[SMI_loop] could not register channel");
        return PP_ERR;
    }
    
    sla_globals->fd = open("/dev/ipmi_bmc_si",O_RDONLY);
    if (sla_globals->fd <= 0) {
        pp_bmc_log_error("[SMI_loop] cannot open /dev/ipmi_bmc_si");
        return PP_ERR;
    }
    sla_globals->selID = pp_select_add_fd(sla_globals->fd, POLLIN, smi_receive_si_msg, NULL);
    if (sla_globals->selID < 0) {
        pp_bmc_log_error("[SMI_loop] Could not register /dev/ipmi_bmc_si in selector");
        return PP_ERR;
   }

    return PP_SUC;
 }

void pp_smi_loop_cleanup()  {
   pp_bmc_router_unreg_chan(IPMI_CHAN_SI);
   close(sla_globals->fd);
   pp_select_remove_fd(sla_globals->selID);
}
