/**
 * ipmi_bmc_loop.c
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 * 
 * This file implements a kernel module that creates a loop between
 * the openIPMI msghandler kernel module and the Peppercon bmccore.
 * It can be used to test the bmccore insystem with the openIPMI lib.
 *
 * TODO: use locks to avoid race conditions
 */

// needed to compile modules
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/vermagic.h>
#if defined(MODVERSIONS)
#include <linux/modversions.h>
#endif

// openIPMI specific stuff
#include <linux/ipmi_smi.h>
#include <linux/ipmi_msgdefs.h>

// pp bmccore specific stuff
#include "ipmi_bmc_si.h"

// general utilities
#include <linux/list.h>
#include <linux/poll.h>


 
 /**
  * The global variables of this module
  */
  
typedef struct {
    ipmi_smi_t openIPMI_intf;          // The openIPMI smi context

    // openIPMI specific stuff
    unsigned char ipmi_maj;
    unsigned char ipmi_min;
    struct ipmi_smi_msg *cur_msg;
    struct ipmi_smi_msg *rsp_msg;
    int ipmiInit;  // are we registered at the msghandler ?

    // bmccore specific stuff
    int df_major;  // our device file major number
    int cdevInit;  // are we already registered as chardev ?
    int useCnt;    // do we have connected users (is a bmccore present ?)
    wait_queue_head_t receiverWaitQ;
    
} ipmi_bmc_loop_global_t;

/** Store all global variables in this structure */
static ipmi_bmc_loop_global_t *ibl_globals;
/** static structure to avoid a kalloc */
static ipmi_bmc_loop_global_t ipmi_bmc_loop_globals;


/**
 * BMC-side file operations (via ioctl. on our char device)
 */ 

static int bmc_ioctl_receive(unsigned long data) {
    bmc_si_msg_t umsg;
    struct ipmi_smi_msg *msg;

    // copy umsg from user to get valid data handle
    if (copy_from_user(&umsg, (void*)data, sizeof(bmc_si_msg_t))) {
        printk("ipmi_loop: bmc tried to receive msg with invalid msghandle\n");
        return -EFAULT;   // correct errortype ?
    }

    // check if we have a msg to receive
    msg = ibl_globals->cur_msg;
    if (msg == NULL) {
        printk("ipmi_loop: bmc queried for non-existant receive data\n");
        return -EFAULT;   // correct errortype ?
    }

    // check the msg size
    if (umsg.data_size < msg->data_size) {
        printk("ipmi_loop: bmc receive - message is truncated\n");
        msg->data_size = umsg.data_size;
    } else
        umsg.data_size = msg->data_size;

    // this cannot fail, we already copied it in the other direction
    copy_to_user((void*)data, &umsg, sizeof(bmc_si_msg_t));
    // if this copy fails just leave the msg where it is and return an error
    if (copy_to_user(umsg.data, msg->data, msg->data_size)) {
        printk("ipmi_loop: bmc tried to received msg with invalid msg.data handle\n");
        return -EFAULT;   // correct errortype ?
    }

    ibl_globals->rsp_msg = ibl_globals->cur_msg;
    ibl_globals->cur_msg = NULL;
    
    return 0;
}

static int bmc_ioctl_send(unsigned long data) {
    struct ipmi_smi_msg *msg;
    bmc_si_msg_t umsg;
    int rv;

    msg = NULL;
    umsg.data = NULL;
    rv = -EFAULT;

    /** use the old msg - is this correct ? */
    if (ibl_globals->rsp_msg == NULL) {
        printk(KERN_WARNING "ipmi_bmc_loop: rejected bmc message: no request from openIPMI\n");
/*        
        msg = ipmi_alloc_smi_msg();
        if (msg <= 0) {
            printk(KERN_WARNING "ipmi_bmc_loop: no mem - rejected message from bmc\n");
            return -EFAULT;
        }
        msg->data_size = 2;
        msg->data[0] = (IPMI_NETFN_APP_REQUEST << 2);
        msg->data[1] = IPMI_GET_MSG_CMD;
        */
        return -EFAULT;
    } else {
       msg = ibl_globals->rsp_msg;
    }

    if (copy_from_user(&umsg, (void*)data, sizeof(bmc_si_msg_t)) > 0) {
        printk("ipmi_loop: bmc tried to send msg with invalid msghandle\n");
        rv = -EFAULT;   // correct errortype ?
        goto out_err;
    }

    if (umsg.data_size > IPMI_MAX_MSG_LENGTH) {
        printk("ipmi_loop: bmc tried to send more bytes than supported\n");
        rv = -EFAULT;   // correct errortype ?
        goto out_err;
    }
    msg->rsp_size = umsg.data_size;

    rv = copy_from_user(msg->rsp, umsg.data, umsg.data_size);
    if (rv > 0) {
        printk("ipmi_loop: bmc tried to send msg with invalid data handle: %d\n", rv);
        rv = -EFAULT;   // correct errortype ?
    goto out_err;
    }
    ibl_globals->rsp_msg = NULL;
    
    // Everything should be ready. Now deliver the message.
    ipmi_smi_msg_received(ibl_globals->openIPMI_intf, msg);

    return 0;

out_err:
    return rv;
}

static int bmc_ioctl(struct inode  *inode,
                     struct file   *file,
                     unsigned int  cmd,
                     unsigned long data)
{
    if (cmd == IPMI_BMC_RECEIVE) {
        return bmc_ioctl_receive(data);
    } else if (cmd == IPMI_BMC_SEND) {
        return bmc_ioctl_send(data);
    } else {
        printk(KERN_NOTICE "ipmi_bmc_loop: received unknown IOCTL %ud\n", cmd);
        return -EFAULT;  // correct errortype ?
    }
}

static int bmc_open(struct inode *inode, struct file *file)
{
    ibl_globals->useCnt++;
    
    if (ibl_globals->useCnt > 1) {
        printk(KERN_NOTICE "ipmi_loop: Too many BMCs registered. Only one BMC is supported.\n");
    }
    
    return 0;
}

static int bmc_release(struct inode *inode, struct file *file)
{
    ibl_globals->useCnt--;
    
    // If the last (and hopefully the only) bmc disconnects, the
    // we should reject any pending messages
    if ((ibl_globals->useCnt == 0) && (ibl_globals->cur_msg != NULL)) {
        struct ipmi_smi_msg *msg;
        msg = ibl_globals->cur_msg;
        ibl_globals->cur_msg = NULL;
        msg->rsp[0] = msg->data[0] | 4;  // netfn: make response from command
        msg->rsp[1] = msg->data[1];      // cmd stays the same
        msg->rsp[2] = 0xCB;              // IPMI_ERROR_NOT_PRESENT
        msg->rsp_size = 3;
        ipmi_smi_msg_received(ibl_globals->openIPMI_intf, msg);
    }
    
    return 0;
}

static unsigned int bmc_poll(struct file *file, poll_table *wait)
{
    unsigned int mask = 0;

    poll_wait(file, &(ibl_globals->receiverWaitQ), wait);

    if (!(ibl_globals->cur_msg == NULL))
        mask |= (POLLIN | POLLRDNORM);

    return mask;
}

static struct file_operations bmc_fops = {
	.owner    = THIS_MODULE,
	.ioctl    = bmc_ioctl,
	.open     = bmc_open,
	.release  = bmc_release,
	.poll     = bmc_poll,
};


/**
 * Functions implementing the openIPMI SI dd (ipmi_smi.h)
 */

void openIPMI_send(void *send_info,
                   struct ipmi_smi_msg *msg,
                   int priority)
{
    if (ibl_globals->useCnt <= 0) {
        // no BMC registered, send error message
        msg->rsp[0] = msg->data[0] | 4;  // netfn: make response from command
        msg->rsp[1] = msg->data[1];      // cmd stays the same
        msg->rsp[2] = 0xCB;              // IPMI_ERROR_NOT_PRESENT
        msg->rsp_size = 3;
        ipmi_smi_msg_received(ibl_globals->openIPMI_intf, msg);
        return;
    }

    // There should be no need to copy the message. We
    // still need it to deliver the response to openIPMI.
    if (ibl_globals->cur_msg != NULL)
        printk(KERN_ERR "ipmi_bmc_loop: receive msg buffer full, msg dropped\n");
    else
        ibl_globals->cur_msg = msg;
    
    wake_up(&(ibl_globals->receiverWaitQ));
}

void openIPMI_request_events(void *send_info)
{
    // Does this apply to us ?
    // If yes, what exactly do we have to do ?
}

void openIPMI_set_run_to_completion(void *send_info, int run_to_completion)
{
    // This does not apply to us - we're always in run_to_completion mode
    // as the messages are always written to the bmccore buffer directly
}

static struct ipmi_smi_handlers openIPMI_handlers = {
	.owner                  = THIS_MODULE,
	.sender			= openIPMI_send,
	.request_events		= openIPMI_request_events,
	.set_run_to_completion  = openIPMI_set_run_to_completion,
};



int init_ipmi_bmc_loop(void)
{
    int rv = 0;
    printk("ipmi_bmc_loop: Version 1.0 starting\n");
    
    // we can still use a kalloc() here if the structure gets too big
    ibl_globals = &ipmi_bmc_loop_globals;
    
    ibl_globals->ipmi_maj = 2;
    ibl_globals->ipmi_min = 0;
    ibl_globals->cur_msg = NULL;
    ibl_globals->rsp_msg = NULL;
    ibl_globals->ipmiInit = 0;
    
    ibl_globals->df_major = 249; // this one should be free ...
    ibl_globals->useCnt =   0;   // no bmccore connected yet
    ibl_globals->cdevInit = 0;
    init_waitqueue_head(&(ibl_globals->receiverWaitQ));
    
    // Register as chrdev
    rv = register_chrdev(ibl_globals->df_major, "ipmi_bmc_si", &bmc_fops);
    if (rv < 0) {
        printk(KERN_ERR "ipmi_bmc_loop: can't register as char device: %d\n", rv);
        goto out_err;
    } else {
        ibl_globals->cdevInit = 1;
        ibl_globals->df_major = rv;
    }

    rv = ipmi_register_smi(&openIPMI_handlers,
        NULL,
        ibl_globals->ipmi_maj, ibl_globals->ipmi_min,
        &(ibl_globals->openIPMI_intf));
    if (rv) {
        printk(KERN_ERR "ipmi_bmc_loop: unable to register as openIPMI device: error %d\n", rv);
        goto out_err;
    } else {
      ibl_globals->ipmiInit = 1;
    }
    
    // everything is okay, init finished
    return 0;
    
    /** release everything in case of errors */
out_err:
    if (ibl_globals->cdevInit == 1) {
        rv = unregister_chrdev(ibl_globals->df_major, "ipmi_bmc_si");
        if (rv) {
            printk(KERN_ERR "ipmi_bmc_loop: can't unregister char device: %d\n", rv);
        }
    }

    if (ibl_globals->ipmiInit == 1) {
        rv = ipmi_unregister_smi(ibl_globals->openIPMI_intf);
        if (rv) {
            printk(KERN_ERR "ipmi_bmc_loop: can't unregister char device: %d\n", rv);
        }
    }
    
    return -1;
}
module_init(init_ipmi_bmc_loop);


void cleanup_ipmi_bmc_loop(void)
{
    printk("ipmi_bmc_loop: unloading\n");
    int rv=0;    

    if (ibl_globals->cdevInit == 1)
      rv = unregister_chrdev(ibl_globals->df_major, "ipmi_bmc_si");
    if (rv) {
        printk(KERN_ERR "ipmi_bmc_loop: can't unregister char device: %d\n", rv);
        // There is not much we can do ...
    }

    rv=0;
    if (ibl_globals->ipmiInit == 1)
      rv = ipmi_unregister_smi(ibl_globals->openIPMI_intf);
    if (rv) {
        printk(KERN_ERR "ipmi_bmc_loop: can't unregister openIPMI device: %d\n", rv);
        // There is not much we can do ...
    }
}
module_exit(cleanup_ipmi_bmc_loop);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("IPMI-BMC loop for the Peppercon BMCcore");
