package PP_VCS;

use strict;
use Carp;
use Getopt::Long;
use Data::Dumper;
use pp_build_lib;
use pp_hw_data;
use base qw( PP_VCS_Base );

sub new {
    my $class = shift;
    my $typ = shift;
    my $self = {};

    eval ("require PP_VCS_" . $typ) or croak $@;
    my $subclass = "PP_VCS_" . $typ;
    bless $self, $subclass;

    $self->{vcs}{mode} = $typ;
    $self->init();

    return $self;
}


sub init {
    my $self = shift;
    my $opts = {};

    # defaults
    $self->{_operation} = 'none';
    $self->debug($DEBUG{ALL});
    $self->use_log_file(1);

    $self->{conflicts_by_component} = {};
    $self->{uncommitted_by_component} = {};
    $self->{updated_by_component} = {};
    $self->{tag_not_moving_by_component} = {};
    $self->{vcserror_by_component} = {};

    $self->{components}{all} = [];
    $self->{components}{package} = [];
    $self->{components}{existing} = [];
    $self->{components}{current} = [];

    $self->{components}{processed} = {};

    # determine operation - this must be called after determining the firmware root
    Getopt::Long::Configure("no_ignore_case", "pass_through", "require_order");
    GetOptions($opts, "S", "A=s", "M", "T=s");
    Getopt::Long::Configure("no_ignore_case", "no_pass_through", "no_require_order");

    # check operation mode
    if ( defined($opts->{'S'}) ) {
	$self->operation('setview');
    }
    if ( defined($opts->{'A'}) ) {
	$self->operation('add');
    }
    if ( defined($opts->{'T'}) ) {
	$self->operation('tag');
    }
    if ( defined($opts->{'M'}) ) {
	$self->operation('merge');
    }
    if ($self->operation() eq 'none') {
	print STDERR "You must specify at least one operation\n";
	$self->usage();
	exit 1;
    }

    if ($self->operation() eq 'setview') {
	GetOptions($opts, "a", "b=s", "c", "d", "l", "n", "r");
	# setview command-line options
	$self->{options}{checkout_all} = defined($opts->{"a"});
	$self->{options}{batch_mode} = defined($opts->{"b"});
	$self->{options}{cleanup} = defined($opts->{"c"});
	$self->{options}{dont_checkout} = defined($opts->{"d"});
	$self->{options}{list} = defined($opts->{"l"});
	$self->{options}{checkout_new_only} = defined($opts->{"n"});
	$self->{options}{reconf_view} = defined($opts->{"r"});

	$self->{target_config} = {
				'oem' => '',
				'lightoem' => '',
				'preserve_install_root' => 'no',
				};

	if ( $self->{options}{batch_mode} ) {
	    ($self->{target_config}{fw_type},
	     $self->{target_config}{board},
	     $self->{target_config}{subboard},
	     $self->{target_config}{product},
	     $self->{target_config}{oem},
	     $self->{target_config}{lightoem},
	     $self->{target_config}{build_type}
	    ) = split /[,;:]/, $opts->{"b"};
	}
    }
    elsif ($self->operation() eq 'add') {
	$self->{options}{new_component} = $opts->{"A"};
	if ($self->{options}{new_component} =~ /^[^A-Za-z0-9]+/) {
	    die "Error: Component name must start with an alphanumeric character!\n";
	}

	# get operation options
	GetOptions($opts, "f");
	$self->{options}{create_framework} = defined($opts->{"f"});
    }
    elsif ($self->operation() eq 'tag') {
	$self->{options}{tag} = $opts->{"T"};
	if ($self->{options}{tag} =~ /^[^A-Za-z0-9]+/) {
	    die "Error: The tag must start with an alphanumeric character!\n";
	}

	# get operation options
	GetOptions($opts, "a", "b", "r=s");

	# check merge command-line options
	if (defined($opts->{"r"}) and $opts->{"r"} =~ /^[^A-Za-z0-9]+/) {
	    die "Error: The reference tag must start with an alphanumeric character!\n";
	}
	$self->{options}{tag_all} = defined($opts->{"a"});
	$self->{options}{branch_tag} = defined($opts->{"b"});
	$self->{options}{ref_tag} = defined($opts->{"r"}) ? $opts->{"r"} : '';
	$self->{target_config} = {};

	# read target configuration
	$self->read_target_config();

	# check firmware target
	if (!defined($self->{target_config}{fw_type})
	    and not $self->{vcs}{mode} eq 'SVN'
	    and defined($self->{options}{with_externals})) {
	    die "\nFirmware target is not yet configured.\n";
	}

	$self->{components}{remaining} = [];
    }
    elsif ($self->operation() eq 'merge') {
	# get operation options
	GetOptions($opts, "t=s", "s=s", "e=s");

	# check merge command-line options
	$self->{options}{fw_type} = $opts->{"t"};
	$self->{options}{start_tag} = defined($opts->{"s"}) ? $opts->{"s"} : "";
	$self->{options}{end_tag} = $opts->{"e"};

	if ($self->{vcs}{mode} eq 'SVN' 
	    and not $self->{options}{start_tag}) 
	{
	    die "\nOption -s <tag/branch> must be specified!\n\n";
	}
	if (!defined($self->{options}{end_tag})) {
	    die "\nOption -e <tag/branch> must be specified!\n\n";
	}

    }
    else {
	die "Don't know how to initialize this operation!\n";
    }
}


sub chooser {
    my $self		= shift;
    my %options		= @_;
    my $list		= $options{list};
    my $default		= $options{default};
    my $prompt		= $options{prompt};
    my $list_elem_cnt	= scalar @{$list};
    my $menu		= '';
    my $cnt		= 0;
    my $val;
    my $defval;

    # add all available choices to menu
    # and highlight default if defined
    foreach my $list_elem (@{$list}) {
	my $sel = " ";
	if (defined $default and $list_elem eq $default) {
	    $defval = $cnt;
	    $sel = ">";
	}
	$menu .= sprintf("  %s %2u - %s\n", $sel, $cnt++,
		    ($list_elem ne "") ? $list_elem : "<none>");
    }

    if (not defined $defval and $self->{options}{batch_mode}) {
	if ($list_elem_cnt > 0) {
	    die "\nInvalid value '" . (defined $default ? $default : "") 
		. "' entered on prompt '$prompt'.\n\n";
	}
	return "";
    }
    elsif (not defined $defval or $self->{options}{reconf_view}) {
	if ($list_elem_cnt > 0) {
	    print "\n$menu\n";
	    do {
		print "$prompt ";
		$val = <STDIN>;
		$val =~ s/\s*(\S*)\s*/$1/;
		$val = $defval if ( $val =~ /^$/ and defined($defval) ) ;
	    } while ($val !~ /^[0-9]+$/ or $val >= $cnt);

	    if ($val < $list_elem_cnt) {
		return $list->[$val];
	    }
	}
	return "";
    }
    else {
	if ($defval < $list_elem_cnt) {
	    return $list->[$defval];
	}
    }

}


sub choose_target_fw_type {
    my $self = shift;
    my @fw_types = sort keys %{$hw_id_map};
    $self->{target_config}{"fw_type"} = $self->chooser(list => \@fw_types,
	    default => $self->{target_config}{"fw_type"},
	    prompt => "Choose firmware type>");
}


sub choose_target_board {
    my $self = shift;
    my @boards = sort keys %{$hw_id_map->{ $self->{target_config}{"fw_type"} }};
    $self->{target_config}{"board"} = $self->chooser(list => \@boards,
	    default => $self->{target_config}{"board"},
	    prompt => "Choose board>");
}


sub choose_target_subboard {
    my $self = shift;
    my @subboards = sort keys %{$hw_id_map->{
	    $self->{target_config}{"fw_type"} }{ $self->{target_config}{"board"} }};
    $self->{target_config}{"subboard"} = $self->chooser(list => \@subboards,
	    default => $self->{target_config}{"subboard"},
	    prompt => "Choose sub-board>");

}


sub choose_target_product {
    my $self = shift;
    my @products = sort keys %{$hw_id_map->{ $self->{target_config}{"fw_type"} }
	    { $self->{target_config}{"board"} }{ $self->{target_config}{"subboard"} }};
    $self->{target_config}{"product"} = $self->chooser(list => \@products,
	    default => $self->{target_config}{"product"},
	    prompt => "Choose product>");
}


sub choose_target_oem {
    my $self = shift;
    croak "Method choose_target_oem() not implemented by class " . ref($self) ."!";
}


sub choose_target_build_type {
    my $self = shift;
    $self->{target_config}{"build_type"} = $self->chooser(list => \@build_types,
	    default => $self->{target_config}{"build_type"},
	    prompt => "Choose build type>");
}


sub setview_endcheck {
    my $self = shift;

    if (scalar(@{ $self->{components}{existing} }) > 0) {
	print "Already checked out components that are left untouched:\n\n";
	foreach my $component (sort @{ $self->{components}{existing} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %{ $self->{uncommitted_by_component} }) > 0) {
	print "Components with uncommitted files:\n\n";
	foreach my $component (sort keys %{ $self->{uncommitted_by_component} } ) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %{ $self->{updated_by_component} }) > 0) {
	print "Components with updated files:\n\n";
	foreach my $component (sort keys %{ $self->{updated_by_component} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %{ $self->{conflicts_by_component} }) > 0) {
	print "Components with conflicts:\n\n";
	foreach my $component (sort keys %{ $self->{conflicts_by_component} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %{ $self->{vcserror_by_component} }) > 0) {
	print "VCS errors:\n\n";
	foreach my $component (sort keys %{ $self->{vcserror_by_component} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }
    if (scalar(keys %{ $self->{conflicts_by_component} }) > 0 
	or scalar(keys %{ $self->{vcserror_by_component} }) > 0) {
	die "\nYou should check cvs.log for errors and/or merge conflicts!\n\n";
    }

}


sub setview_external {
    my $self	= shift;
    my %options	= @_;

    my $command = "(cd $options{ext_path}; ./ppsetview -S"
	    . ($self->{options}{checkout_new_only} ? " -n" : "")
	    . " -b $options{fw_type}:$options{board}:$options{subboard}"
	    . ":$options{product}:$options{oem}:$options{light_oem}"
	    . ":$options{build_type})";

    $self->do_log($DEBUG{INFO}, "Setting view for external "
		. "'$options{ext_name}', check "
		. "$options{ext_path}/cvs.log for errors.\n" .
		"$command\n");

    system($command);

    $self->do_log($DEBUG{INFO}, "done...\n");

}


sub remove_hidden_fwcomponents_root {
    my $self = shift;

    if (-d '.fwcomponents') {
	print "Removing hidden fwcomponents root: ";
	system("rm -rf .fwcomponents");
	if ($? == 0) {
	    print "done\n";
	} else {
	    print "failed (ignored)\n";
	}
    }
    return 0;
}


sub check_new_component_local {
    my $self = shift;

    if ( -d "$self->{options}{new_component}" ) {
	if ( -d "$self->{options}{new_component}/CVS" or
	     -d "$self->{options}{new_component}/.svn")
	{
	    $self->remove_hidden_fwcomponents_root();
	    die "\nThe component $self->{options}{new_component} seems to be in VCS already.\n\n"

	} elsif ($self->{options}{create_framework}) {
	    if (opendir my $fh, "$self->{options}{new_component}") {
		my @dir_contents = grep { !/^\.\.?$/ } readdir($fh);
		if (scalar(@dir_contents) > 0) {
		    $self->remove_hidden_fwcomponents_root();
		    die "\nThe component directory must be empty to add a framework!\n\n"
		}
		closedir $fh;
	    }
	}
    }

}


sub filter_and_uniquify_components {
    my $self = shift;
    my $what = shift;

    return 0 unless exists $self->{components}{$what};

    # cut of any subtrees from components
    # (always checkout full component)
    for (0 .. $#{ $self->{components}{$what} }) {
	$self->{components}{$what}[$_] =~ s/\/.*//;
    }

    # skip duplicate and the build_sys component
    # from perl cookbook 4.7.2.4
    my %seen = ();
    foreach my $component ( @{ $self->{components}{$what} } ) {
	$component =~ /^build_sys$/ and next; # build_sys is not a real component
	$component =~ /^Source(\/.*|)$/ and next; # "Source" is a special Raritan component.
	$seen{$component}++;
    }
    @{ $self->{components}{$what} } = keys %seen;

}


sub check_components_before_tagging {
    my $self = shift;

    # determine required components (do *not* skip missing directories)
    my $skip_missing_dirs = 0;
    get_components(	$self->{components}{all},
			$self->{components}{current},
			$skip_missing_dirs,
			$self->{target_config}{fw_type},
			$self->{target_config}{board},
			$self->{target_config}{subboard},
			$self->{target_config}{product},
			"none")
    or die "\nError during determining the required components\n";

    $self->filter_and_uniquify_components('all');
    $self->filter_and_uniquify_components('current');

    # get difference between all_components and components
    # from perl cookbook 4.8.2.3
    my %seen = ();
    @seen{ @{ $self->{components}{all} } } = ( );
    delete @seen{ @{ $self->{components}{current} } };

    @{ $self->{components}{remaining} } = keys %seen;

}


sub tagging_endcheck {
    my $self = shift;

    if (scalar(keys %{ $self->{tag_not_moving_by_component} }) > 0) {
	print "Components that are already tagged and would require tag moving:\n\n";
	foreach my $component (sort keys %{ $self->{tag_not_moving_by_component} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %{ $self->{vcserror_by_component} }) > 0) {
	print "VCS errors:\n\n";
	foreach my $component (sort keys %{ $self->{vcserror_by_component} }) {
	    # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }
}

sub operation {
    my $self = shift;

    # get
    unless (@_ == 1 ) {
	if (ref($self)) {
	    return $self->{_operation};
	}
	else {
	    confess "You cannot use this as a class method!";
	}
    }

    # set
    my $operation = shift;

    if (ref($self)) {
	if ($self->{_operation} ne 'none' ) {
	    die "Error: conflicting operations specified\n";
	}
	else {
	    $self->{_operation} = $operation;
	}
    }
    else {
	confess "You need an object here!";
    }
}


sub checkout_component {
    my $self = shift;
    croak "Method checkout_component() not implemented by class " . ref($self) ."!";
}


sub update_base {
    my $self = shift;
    croak "Method update_base() not implemented by class " . ref($self) ."!";
}

sub setview {
    my $self = shift;
    croak "Method setview() not implemented by class " . ref($self) ."!";
}

sub add {
    my $self = shift;
    croak "Method add() not implemented by class " . ref($self) ."!";
}

sub tag {
    my $self = shift;
    croak "Method tag() not implemented by class " . ref($self) ."!";
}

sub merge {
    my $self = shift;
    croak "Method merge() not implemented by class " . ref($self) ."!";
}

sub usage {
    print <<EOF;

Usage: ppsetview <operation> <options>

    <operations> are selected using capitalized command-line switches.
    <options> are specified using lower case command-line switches.

    -S		 Setting a view by checking out the required components from
		 CVS. The view is determined by various configuration items
		 that are specified interactively once.

	-a	 Checkout all possible components for a particular firmware
		 type. This must be used for branching and merging.

	-b <cfg> Batch mode. <cfg> is a colon/comma/semicolon seperated list

		 fwtype:board:sub-board:product:oem:light-oem:build-type

	-n	 Only checkout new components and don't touch existing. This
		 may lead to inconsistencies on the local CVS tree but reduces
		 CVS traffic quite a lot. Use this option with care!

	-r	 Reconfigure the view parameters.
	
	-d	 Only set the target configuration, don't check out.
		 This is needed if e.g. configuring a new product with no
		 featureset specified yet.

	-l	 List unnecessary components (e.g. after branching).

	-c	 Clean up, i.e. remove unnecessary components.

    -A <name>	 Add a new component called <name>.

	-f	 Create the component framework. The framework depends on the
		 name of the component. If the name starts with "libpp_" it
		 will create a framework for libraries. Otherwise a framework
		 for executables is created.
		 NOTE: The framework is created only if the component directory
		       does not yet exist on the local side or if it is empty!

    -T <tag>	 Create a tag.

	-a	 Tag the all-view.

	-b	 Create a branch tag.

	-r <tag> Specify a reference tag.

    -M		 Merge a branch.

	-t <typ> This specifies the firmware type (e.g. erla). It is used for
		 determining the components to merge. This option has a second
		 meaning: If ommitted the build system only is merged.
		 Otherwise the components only.

	-s <tag> The tag in the branch from where to start the merge. On
		 consecutive merges you use the end tag of the last merge as
		 start tag for the current one. This tag may be ommitted if a
		 branch is merged for the first time. In this case the branch
		 point is used by default. However, in SVN this option
		 becomes mandatory.

	-e <tag> The tag in the branch up to which changes should be merged.

EOF

}

1;
