package PP_VCS_CVS;

use strict;
use Carp;
use Cwd;
use Data::Dumper;
use File::Path;
use File::Find;
use pp_build_lib;
use pp_hw_data;
use base qw(PP_VCS);

sub determine_repo_information {
    my $self = shift;
    if (-r "CVS/Root") {
	open(my $vc_info, "CVS/Root");
	$self->{vcs}{root} = <$vc_info>;
	$self->{vcs}{root} =~ s/\s*(\S*)\s*/$1/;
	close $vc_info;
    }
    else {
	die "\nCannot open CVS/Root ($!).\n\n";
    }

    # determine the CVS module
    if (-r "CVS/Repository") {
	open (my $vc_info, "CVS/Repository");
	$self->{vcs}{module} = <$vc_info>;
	$self->{vcs}{module} =~ s/\s*(\S*)\s*/$1/;
	$self->{vcs}{module} =~ s/\/.*//;
	close $vc_info;
    }
    else {
	die "\nCannot open CVS/Repository ($!).\n\n";
    }

    # determine branch
    $self->{vcs}{is_branch} = 0;
    if (-e "CVS/Tag") {
	if (open my $vc_info, "CVS/Tag") {
	    $self->{vcs}{tag} = <$vc_info>;
	    # The first character is a single letter indicating the type
	    # of tag: T, N, or D, for branch tag, nonbranch tag, or date
	    # respectively.
	    $self->{vcs}{is_branch} = 1
		if ($self->{vcs}{tag} =~ /\s*T\S*\s*/);
	    $self->{vcs}{tag} =~ s/\s*\S(\S*)\s*/$1/;
	    close $vc_info;
	}
	else {
	    # possible permission problem
	    die "\nCannot read CVS/Tag ($!).\n\n";
	}
    } else {
	$self->{vcs}{tag} = ""
    }

    # check for r/o mirror server
    $self->{vcs}{mirror} = "";
    if (open my $cvs_mirror_fh, "$ENV{HOME}/.ppsetview/mirror") {
        $self->{vcs}{mirror} = <$cvs_mirror_fh>;
        $self->{vcs}{mirror} =~ s/\s*(\S*)\s*/$1/;
    }

    # check if we have authentication
    if ($self->{vcs}{mirror} ne '') {
	# just try to get the status of any file, if we don't have
	# access this will fail regardless of whether the file exists or
	# not
	system "cvs -d '$self->{vcs}{mirror}' status dummy 2>&1 | grep -q 'cvs login'";
	if ($? == 0) {
	    # user must enter valid authentication data
	    do {
		system "cvs -d '$self->{vcs}{mirror}' login";
	    } while ($? != 0);
	}
    }
}


sub cvs_operation {
    my $self = shift;
    my %options = @_;
    my $component = $options{component} || 'dummy';
    my $message = $options{message} || '';
    my $cvs_operation = $options{operation} or croak "No operation given!";

    print "$message: ";
    $self->do_log($DEBUG{INFO}, "---- CVS: $message ...\n");

    my $cvsfh;
    # start CVS operation
    if (not open $cvsfh, "$cvs_operation 2>&1 |") {
	print "failed\n";
	$self->do_log($DEBUG{WARN},
	    "!!!! CVS: Executing cvs command ($cvs_operation) failed ($!).\n\n");
	$self->{vcsserror_by_component}{$component} = 1;
	return 0;
    }

    # copy CVS output to log file
    while (<$cvsfh>) {
	if ($_ =~ /Merging differences /) {
	    $self->{updated_by_component}{$component} = 1;
	}
	if ($_ =~ /^C / or $_ =~ /conflicts during merge/) {
	    $self->{conflicts_by_component}{$component} = 1;
	} elsif ($_ =~ /^M / or $_ =~ /^A / or $_ =~ /^R /) {
	    $self->{uncommitted_by_component}{$component} = 1;
	} elsif ($_ =~ /^U / or $_ =~ /^P /) {
	    $self->{updated_by_component}{$component} = 1;
	} elsif ($_ =~ /^W /) {
	    $self->{tag_not_moving_by_component}{$component} = 1;
	}

	$self->do_log($DEBUG{INFO}, "$_");
    }
    close $cvsfh;

    # evaluate exit code
    if ($? == 0) {
	print "done\n";
	$self->do_log($DEBUG{INFO}, "---- CVS: $message finished.\n\n");
	return 1;
    } else {
	print "failed\n";
	$self->do_log($DEBUG{WARN}, "!!!! CVS: $message failed.\n\n");
	$self->{vcserror_by_component}{$component} = 1;
	return 0;
    }
}


sub update_base {
    my $self = shift;

    # before doing an update, create md5sum of our precious self
    $self->create_checksum();

    # update the base directory (NOTE: subdirectories must be specified here manually!)
    my @base_subdirs = ( "build_sys" );
    if (not $self->cvs_operation(component => "<base>", message => "Updating base directory", 
		operation => "cvs -z3 update -l -P") ) {
	exit 1;
    }

    foreach my $base_subdir (@base_subdirs) {
	if (not $self->cvs_operation(component => "<base>",
	    message => "Updating directory '$base_subdir'",
	    operation => "cvs -z3 update -dP $base_subdir")) {
		exit 1;
	}
    }

    # check if some bad guy modified us and restart in this case
    if ($self->verify_checksum() != 0) {
	if ( defined($self->{conflicts_by_component}{"<base>"}) ) {
	    die "\nWe were modified and a conflict occured. Check cvs.log! Aborting ...\n\n";
	}
	elsif ( defined($self->{vcserror_by_component}{"<base>"}) ) {
	    die "\nWe were modified and a CVS error occured. Check cvs.log! Aborting ...\n\n";
	}
	else {
#	    print "\nWe were modified. Restarting ourself ...\n\n";
	    # We were modified. Someone should restart us
	    return 1;
	}
    }

    return 0;
}


sub checkout_component {
    my $self		    = shift;
    my %options		    = @_;
    my $mirror_allowed	    = $options{mirror_allowed};
    my $component	    = $options{component};

    if ( -d $component ) {
	# determine the CVS repository of the component
	open(my $cvsfh, "$component/CVS/Repository")
	    or die "\nCannot open $component/CVS/Repository ($!).\n\n";
	my $cvs_module_comp = <$cvsfh>;
	$cvs_module_comp =~ s/\s*(\S*)\s*/$1/;
	$cvs_module_comp =~ s/\/.*//;
	close $cvsfh;

	# components have a different cvs module - so if we have a component
	# with the same module as our base directory then someone has
	# added a component to the base directory. This is not allowed but
	# cannot entirely prevented administratively with CVS. (SVN does
	# not have this problem as directories are also versioned)
	# Since file checkins are prevented we can assume that only
	# empty directories are in the way that can be pruned using "cvs
	# update -dP".
	if ($self->{vcs}{module} eq $cvs_module_comp) {
	    if (not $self->cvs_operation(component => $component,
		message => "Pruning '$component'",
		operation => "cvs -z3 update -dP \"$component\"") ) 
	    {
		return 0;
	    }
	}
    }

    if ( $self->{options}{checkout_new_only} ) {
	if ( -d $component ) {
	    push @{ $self->{components}{existing} }, $component;
	    return 1;
	}
    }

    my $branch_spec = $self->{vcs}{tag} ne '' ? "-r $self->{vcs}{tag}" : "-A";
    my $mirror_failed = 0;
    if ($mirror_allowed
	and $self->{vcs}{mirror} ne ''
	and not $self->cvs_operation(component => $component,
	    message => "Checking out '$component'",
	    operation => "cvs -z3 -d \"$self->{vcs}{mirror}\" co -P "
	    . "-d \"$component\"  $branch_spec "
	    . "\"fwcomponents/$component\" ") ) {
	delete $self->{vcserror_by_component}{$component};
	$mirror_failed = 1;
    }
    if ($mirror_allowed and $self->{vcs}{mirror} ne ''
	    and not -d "$component/CVS") {
	$mirror_failed = 1;
    }
    if ( (not $mirror_allowed or $self->{vcs}{mirror} eq '' or
	    $mirror_failed) 
	and not $self->cvs_operation(component=> $component,
	    message => "Checking out '$component'"
		. ($mirror_failed ? " from master server" : ""),
	    operation => "cvs -z3 -d \"$self->{vcs}{root}\" co -P "
	    . "-d \"$component\" $branch_spec "
	    . "\"fwcomponents/$component\"") ){
	return 0;
    }

    return 1;
}


sub merge_component {
    my $self = shift;
    my $component = shift;

    if ( -d $component ) {
	# determine the CVS repository of the component
	open(my $cvsfh, "$component/CVS/Repository")
	    or die "\nCannot open $component/CVS/Repository ($!).\n\n";
	my $cvs_module_comp = <$cvsfh>;
	$cvs_module_comp =~ s/\s*(\S*)\s*/$1/;
	$cvs_module_comp =~ s/\/.*//;
	close $cvsfh;

	# components have a different cvs module - so if we have a component
	# with the same module as our base directory then someone has
	# added a component to the base directory. This is not allowed but
	# cannot entirely prevented administratively with CVS. (SVN does
	# not have this problem as directories are also versioned)
	# Since file checkins are prevented we can assume that only
	# empty directories are in the way that can be pruned using "cvs
	# update -dP".
	if ($self->{vcs}{module} eq $cvs_module_comp) {
	    if (not $self->cvs_operation(component => $component,
		message => "Pruning '$component'",
		operation => "cvs -z3 update -dP \"$component\"") ) 
	    {
		return 0;
	    }
	}
    }

    my $start_tag_spec = $self->{options}{start_tag} ne "" 
			? "-j $self->{options}{start_tag}" : "";
    my $end_tag_spec = $self->{options}{end_tag} ne "" 
			? "-j $self->{options}{end_tag}" : "";
    my $branch_spec = $self->{vcs}{tag} ne "" 
			? "-r $self->{vcs}{tag}" : "-A";

    if (not $self->cvs_operation(component => $component,
		message => "Merging '$component'",
		operation => "cvs -z3 -d \"$self->{vcs}{root}\" co -P "
		    ."-d \"$component\" $branch_spec $start_tag_spec "
		    ."$end_tag_spec \"fwcomponents/$component\"")) {
	return 0;
    }

    return 1;
}


sub checkout_external {
    my $self		= shift;
    my %options		= @_;
    my $mirror_allowed	= $options{mirror_allowed};
    my $component	= $options{component};
    my $ext_name	= determine_ext_name($component);
    my $current_tag	= $self->{vcs}{tag};
    my($ext_tag, $tag, $fw_type, $board, $subboard, $product, $OEM,
       $light_OEM, $build_type) = split ",", $component;
    my $ext_base_path = "externals";
    my $ext_path = "$ext_base_path/$ext_name";

    # do not use external cvs tag if it is a branch tag and
    # firmware tag is a non-branch tag
    if ($current_tag eq '') {
	#save some co time: MAIN is a branch for sure
	$current_tag = $tag;
    }
    else {
	# instead of doing some pattern matching magic, just look at the
	# symbols in the repo - a branch has a zero as the second last
	# number like: 1.35._0_.6, while a tag is always something like
	# 1.35 or 1.30.2.1 or even 1.27.4.1.2.1, but no zero
	my $cvsfh;
	# we check the cvs log of ppsetview, as it should always be present
	if (not open $cvsfh, "cvs -n -z3 log -h ppsetview 2>&1 |") {
	    print "failed\n";
	    die "cannot verify branchness of component for external checkout";
	}

	my $fw_is_tag = 0;
	my $ext_is_tag = 0;
	# special tag for externals
	my $fw_ext_tag = $current_tag . '_EXT_' . $ext_name;
	my $fw_ext_tag_found = 0;
	while (<$cvsfh>) {
	    chomp;
	    next if ($_ !~ /^\t.+$/);
	    my @line = split(/\./, $_);
	    if ($line[-2] ne 0) {
		$fw_is_tag = 1 if ($_ =~ /^\t$current_tag:.+/);
		$ext_is_tag = 1 if ($_ =~ /^\t$tag:.+/);
		$fw_ext_tag_found = 1 if ($_ =~ /^\t$fw_ext_tag:.+/);
	    }
	    # if external is a tag, exit early
	    last if $ext_is_tag;
	}
	close $cvsfh;

	if ($fw_is_tag and not $ext_is_tag) {
	    if ($fw_ext_tag_found) {
		$current_tag = $fw_ext_tag;
	    } else {
		die "\nCannot checkout external '$ext_name', because it "
		. "has not been tagged correctly!\"n"
	    }
	}
	else {
	    $current_tag = $tag;
	}
    }

    # check out base system
    my $branch_spec = $current_tag ne "" ? "-r $current_tag" : "-A";

    # checkout external base system if not already there
    if(! -d $ext_path) {
	# create externals path if necessary
	! -d $ext_base_path and mkpath($ext_base_path);

	# store current work directory
	my $cwd = cwd();
	chdir($ext_base_path);

	# never use mirror to checkout fwbase!
	if (not $self->cvs_operation(component => $ext_path,
		message => "Checking out external '$ext_name'",
		operation => "cvs -z3 -d \"$self->{vcs}{root}\" co -P "
		    . "-d \"$ext_name\" $branch_spec \"fwbase\"") )
	{
	    $self->{vcserror_by_component}{$ext_path} = 1;
	    chdir($cwd);
	    return 0;
	}
	chdir($cwd);
    }
    else {
	# external path already existing - make sure to update to the correct branch

	# store current work directory
	my $cwd = cwd();
	chdir($ext_path);

	# update fwbase (non-recursive)
	if (not $self->cvs_operation(component => "<base>",
		message => "Updating base directory",
		operation => "cvs -z3 update -l -P $branch_spec") ) 
	{
	    $self->{vcserror_by_component}{$ext_path} = 1;
	    chdir($cwd);
	    return 0;
	}

	# update build_sys
	my @base_subdirs = ( "build_sys" );
	foreach my $base_subdir (@base_subdirs) {
	    if (not $self->cvs_operation(component => "<base>",
		    message => "Updating directory '$base_subdir'",
		    operation => "cvs -z3 update -dP $branch_spec $base_subdir") )
	    {
		$self->{vcserror_by_component}{$ext_path} = 1;
		chdir($cwd);
		return 0;
	    }
	}
	chdir($cwd);
    }

    # set the external view
    $self->setview_external(fw_type => $fw_type, 
			    board => $board,
			    subboard => $subboard,
			    product => $product,
			    oem => $OEM,
			    lightoem => (defined $light_OEM ? $light_OEM : ''),
			    build_type => ($build_type 
					    ? $build_type 
					    : $self->{target_config}{build_type}),
			    ext_name => $ext_name,
			    ext_path => $ext_path);

    return 1;
}


# has to be made vcs specific because we need to call checkout_component
sub choose_target_oem {
    my $self = shift;

    my %oems;
    my $find_oems = sub {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)$/
		and $self->{target_config}{"board"} eq
		    $self->{target_config}{"product"}
		and $1 eq $self->{target_config}{"board"}) {

		$oems{$2} = 1;
	    } elsif ($_ =~ /^([^_]+)_([^_]+)_([^_]+)/
		     and $1 eq $self->{target_config}{"board"}
		     and $2 eq $self->{target_config}{"product"}) {

		$oems{$3} = 1;
	    }
	}
    };

    my %lightoems;
    my $find_lightoems = sub {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)_([^_]+)_([^_]+)$/
		and $1 eq $self->{target_config}{"board"}
		and $2 eq $self->{target_config}{"product"}
		and $3 eq $self->{target_config}{"oem"}) {

		$lightoems{$4} = 1;
	    }
	}
    };

    # check out OEM. this is currently needed for all firmware types.
    # find magic to decide once we have a non OEM firmware
    print "\n";
    if ($self->checkout_component(mirror_allowed => 0, component => "OEM")) {
	# scan OEM directory
	find({ wanted => $find_oems, no_chdir => 0 }, "OEM");
	my @oems = sort keys %oems;
	# ask for OEMs
	$self->{target_config}{"oem"} = $self->chooser( list => \@oems,
	    list_hidden => [], default => $self->{target_config}{"oem"},
	    prompt => "Choose OEM>");

	# scan OEM directory
	find({ wanted => $find_lightoems, no_chdir => 0 }, "OEM");
	my @lightoems = sort keys %lightoems;
	if (scalar(@lightoems) > 0) {
	    unshift @lightoems, "";
	}
	# ask for light-OEMs
	$self->{target_config}{"lightoem"} = $self->chooser( list => \@lightoems,
	    list_hidden=> [], default => $self->{target_config}{"lightoem"},
	    prompt => "Choose light-OEM>");
    }

}


sub setview {
    my $self = shift;
    my %seen;

    if (not $self->{options}{batch_mode}) {
	$self->read_target_config();
    }

    # ask for firmware type
    $self->choose_target_fw_type();
    # ask for board
    $self->choose_target_board();
    # ask for subboard
    $self->choose_target_subboard();
    # ask for product
    $self->choose_target_product();
    # ask for oem
    $self->choose_target_oem();
    # ask for the build type
    $self->choose_target_build_type();

    # write target configuration to file
    $self->write_target_config();

    $self->print_target_config();

    # we need Config.[sh|mk], generated by build system. generate fake if it isnt there
    $self->generate_config_sh_mk();

    # do we need to go further?
    if($self->{options}{dont_checkout}) {
	print "don't checkout selected\n";
	exit 0;
    }


    # determine required components (do *not* skip missing directories)
    # non resolvable dependencies will be ignored (as possibly fixed in CVS)
    my $skip_missing_dirs = 0;
    get_components( $self->{components}{all},
		    $self->{components}{current},
		    $skip_missing_dirs,
		    $self->{target_config}{fw_type},
		    $self->{target_config}{board},
		    $self->{target_config}{subboard},
		    $self->{target_config}{product},
		   "none")
	or die "\nError during determining the required components\n";

    # list or cleanup mode
    if ( ($self->{options}{list} || $self->{options}{cleanup})
	&& ! $self->{options}{checkout_all}
	&& ! $self->{options}{checkout_pkg})
    {
	# cut of any subtrees from components
	# (always checkout full component)
	for (0 .. $#{ $self->{components}{all} }) {
	    $self->{components}{all}[$_] =~ s/\/.*//;
	}
	for (0 .. $#{ $self->{components}{current} }) {
	    $self->{components}{current}[$_] =~ s/\/.*//;
	}

	# from perl cookbook 4.8.2.3
	%seen = ();
	@seen {@{ $self->{components}{all} }} = ( );
	delete @seen {@{ $self->{components}{current} }};

	my @clean_components = ();
	foreach my $component (keys %seen) {
	    opendir(DIR, $component) && push(@clean_components, $component);
	    closedir(DIR);
	}

	my $clean_components = join(" ", @clean_components);
	if (! $clean_components) {
	    print "\nNothing to clean up.\n";
	}
	else {
	    if ($self->{options}{cleanup}) {
		my @not_cleaned_components = ();
		print "\nRemoving unnecessary components:\n$clean_components\n\nPress return to continue...";
		my $foo = <STDIN>;

		COMPONENT: foreach my $clean_component (@clean_components) {
		    my $cvsfh;
		    print "Verifying status of $clean_component: ";

		    # similar to cvs_operation but do not log!
		    # run cvs update quietly and without any disk modifying operations (-n)
		    # to verify cleanliness of checkout
		    if (not open $cvsfh, "cvs -n -z3 -Q update -dP $clean_component 2>&1 |") {
			print "failed\n";
			push(@not_cleaned_components, $clean_component);
			next COMPONENT;
		    }

		    # parse CVS output (checking for added, removed, modified or conflicting
		    # entries, which indicate a unclean component
		    while (<$cvsfh>) {
			if ($_ =~ /^[ARMC\?] /) {
			    # component was somehow modified, request user interaction
			    close $cvsfh;
			    print "modified\n";
			    push(@not_cleaned_components, $clean_component);
			    next COMPONENT;
			} 
		    }
		    close $cvsfh;

		    rmtree($clean_component);
		    print "ok... removing...\n";
		}

		my $not_cleaned_components = join(" ", @not_cleaned_components);
		if ($not_cleaned_components) {
		    print "\nPlease check an manually remove not yet cleaned components:\n$not_cleaned_components\n";
		}
	    }
	    else {
		print "\nSuggest to clean unnecessary components:\n$clean_components\n\nCall ppsetview -S -c to clean up.\n";
	    }
	}

	# stop here?
	print "\n";
	exit 0;
    }

    # do we want all components?
    if ( $self->{options}{checkout_all} ) {
	@{ $self->{components}{current} } = @{ $self->{components}{all} };
    }
    elsif ($self->{options}{checkout_pkg} ) {
	@{ $self->{components}{current} } = @{ $self->{components}{package} };
    }

    # cut of any subtrees (always checkout full component)
    for (0 .. $#{ $self->{components}{current} }) {
	$self->{components}{current}[$_] =~ s/\/.*//;
    }

    %seen = ();
    # uniq
    @{ $self->{components}{current} } = grep !$seen{$_}++, 
					    @{ $self->{components}{current} };

    # don't run again over already checkout base components
    $self->{components}{processed}{build_sys}++;
    $self->{components}{processed}{OEM}++;

    # run this loop at least once
    my $components_to_process = scalar(@{ $self->{components}{current} });
    my $iteration = 1;
    while ($components_to_process > 0) {
	print "Starting checkout iteration $iteration, " .
	      "$components_to_process components to process\n";
	$iteration++;

	foreach my $component (@{ $self->{components}{current} }) {
	    # skip already processed components
	    $self->{components}{processed}{$component}++ and next; 
	    $self->checkout_component(mirror_allowed => 1,
		    component => $component);
	    if ($self->{vcs}{mirror} ne '') {
		# after checkin out from the mirror, modify all CVS/Root files to point to the real
		# repo again instead of the mirror
		`find $component -name Root -exec sed -i s,$self->{vcs}{mirror},$self->{vcs}{root}, {} \\;`;
		# do a CVS update after checking out from the mirror
		$self->cvs_operation(component => $component,
			    message => "Updating directory '$component'",
			    operation => "cvs -z3 update -dP $component");
	    }
	}
	# non resolvable dependencies will be ignored (as possibly fixed in CVS)
	$skip_missing_dirs = 0;
	get_components( $self->{components}{all},
			$self->{components}{current},
			$skip_missing_dirs,
			$self->{target_config}{fw_type},
			$self->{target_config}{board},
			$self->{target_config}{subboard},
			$self->{target_config}{product},
		       "none")
	    or die "\nError during determining the required components\n";

	# cut of any subtrees (always checkout full component)
	for (0 .. $#{ $self->{components}{current} }) {
	    $self->{components}{current}[$_] =~ s/\/.*//;
	}

	%seen = ();
	# uniq
	@{ $self->{components}{current} } = grep !$seen{$_}++, 
						@{ $self->{components}{current} };

	$components_to_process = 0;
	foreach(@{ $self->{components}{current} }) {
		$self->{components}{processed}{$_}
		    or $components_to_process++;
	}
    }
    print "\n";

    $self->setview_endcheck();
}

sub add {
    my $self = shift;

    # checkout fwcomponents root and check if the requested component already exists
    if (not $self->cvs_operation(component => "",
	    message => "Checking out fwcomponents root",
	    operation => "cvs -z3 -f -d \"$self->{vcs}{root}\" co -l -d .fwcomponents fwcomponents",)) 
	{
	exit 1;
    }

    # try checking out the new component to check if it already exists (must fail!)
    if ($self->cvs_operation(component => $self->{options}{new_component},
	    message => "Checking out new component '$self->{options}{new_component}' which must fail",
	    operation => "cd .fwcomponents && cvs -z3 -f -d \"$self->{vcs}{root}\" "
			. "co -l -d \"$self->{options}{new_component}\" "
			. "\"fwcomponents/$self->{options}{new_component}\"")) {
	$self->remove_hidden_fwcomponents_root();
	die "\nThe component '$self->{options}{new_component}' already exists.\n\n";
    }
    delete $self->{vcserror_by_component}{ $self->{options}{new_component} };

    # handle the case when the new component directory exists already on the local side
    # and if not, create the directory
    $self->check_new_component_local();

    # create the new component directory in case it doesn't exist yet
    if (not -d ".fwcomponents/$self->{options}{new_component}" 
	and not mkdir ".fwcomponents/$self->{options}{new_component}")
    {
	$self->remove_hidden_fwcomponents_root();
	die "\nCannot create .fwcomponents/$self->{options}{new_component} ($!).\n\n";
    }

    # add root directory of the new component to cvs, which will be committed
    # implicitly, thus we are able to check it out later visibly
    if (not $self->cvs_operation(component => $self->{options}{new_component},
	    message => "Adding new component '$self->{options}{new_component}'",
	    operation => "cvs -z3 -d \"$self->{vcs}{root}\" add "
			."\".fwcomponents/$self->{options}{new_component}\"")) 
    {
	$self->remove_hidden_fwcomponents_root();
	die "\nFailed to add the new component '$self->{options}{new_component}' to CVS.\n\n";
    }

    # remove the hidden fwcomponents root directory as we don't need it anymore
    $self->remove_hidden_fwcomponents_root();

    # optionally create framework
    if ($self->{options}{create_framework}) {
	# determine framework to use
	my $framework = $self->{options}{new_component} =~ /^libpp_(.*)/ 
			? "libpp_example" : "example";
	my $unprefixed_name = $self->{options}{new_component} =~ /^libpp_(.*)/ 
			? $1 : $self->{options}{new_component};
	my $unprefixed_name_upcase = $unprefixed_name;
	$unprefixed_name_upcase =~ tr/a-z/A-Z/;

	# export framework
	my $branch_spec = $self->{vcs}{tag} ne "" ? "-r $self->{vcs}{tag}" : "-r HEAD";
	if (not $self->cvs_operation(component => $self->{options}{new_component},
		message => "Exporting framework for '$self->{options}{new_component}'",
		operation => "cvs -z3 -d \"$self->{vcs}{root}\" export "
			. "-d \"$self->{options}{new_component}\" $branch_spec "
			. "\"fwcomponents/$framework\"")) 
	{
	    $self->remove_hidden_fwcomponents_root();
	    die "\nFailed to checkout the framework component.\n\n";
	}
	# patch framework
	print "Patching framework: ";
	my $patch_cmd = "find ./$self->{options}{new_component} -type f -exec "
		. "sed -i -e 's,example,$unprefixed_name,' "
		. "-e 's,EXAMPLE,$unprefixed_name_upcase,' {} \\; "
		. "&& if [ -f $self->{options}{new_component}/src/include/pp/example.h ]; "
		. "then mv $self->{options}{new_component}/src/include/pp/example.h "
		. "$self->{options}{new_component}/src/include/pp/$unprefixed_name.h; fi";
	system($patch_cmd);
	if ($? == 0) {
	    print "done\n";
	} else {
	    print "failed (ignored)\n";
	}
    } else {
	system("mkdir '$self->{options}{new_component}' "
	    . "&& touch '$self->{options}{new_component}/.ppsetview.tmp'");
    }

    # checkout root directory of the new component
    my $branch_spec = $self->{vcs}{tag} ne "" ? "-r $self->{vcs}{tag}" : "-A";
    if (not $self->cvs_operation(component => $self->{options}{new_component},
	    message => "Checking out new component '$self->{options}{new_component}'",
	    operation => "cvs -z3 -f -d \"$self->{vcs}{root}\" co "
		. "-d \"$self->{options}{new_component}\" $branch_spec "
		. "\"fwcomponents/$self->{options}{new_component}\"")) 
    {
	$self->remove_hidden_fwcomponents_root();
	system("rm -f '$self->{options}{new_component}/.ppsetview.tmp'");
	die "\nFailed to checkout the new component '$self->{options}{new_component}'.\n\n";
    }
    unlink("$self->{options}{new_component}/.ppsetview.tmp");

}

sub tag {
    my $self = shift;

    $self->check_components_before_tagging();
    my $branch_tag = ($self->{options}{branch_tag}) ? ' -b' : '';
    my $ref_tag = ($self->{options}{ref_tag}) 
		    ? "-r $self->{options}{ref_tag}" : '';
    # tag the base directory (NOTE: subdirectories must be specified here manually!)
    my @base_subdirs = ( "build_sys" );
    $self->cvs_operation(component => "<base>",
	message => "Tagging base directory",
	operation => "cvs -z3 tag -c -l $branch_tag "
		    ."$ref_tag $self->{options}{tag}");
    foreach my $base_subdir (@base_subdirs) {
        $self->cvs_operation(component => "<base>",
	    message => "Tagging directory '$base_subdir'",
	    operation => "cvs -z3 tag -c $branch_tag "
		."$ref_tag $self->{options}{tag} $base_subdir");
    }

    # tag components
    foreach my $component ( @{ $self->{components}{current} } ) {
	$self->cvs_operation(component => $component,
		message => "Tagging '$component'",
		operation => "cvs -z3 tag -c $branch_tag "
		    ."$ref_tag $self->{options}{tag} $component");
    }

    # tag remaining components from the all view if requested
    if ($self->{options}{tag_all}) {
	# use the current branch head if no reference tag is specified
	if ($self->{options}{ref_tag} eq "" and $self->{vcs}{tag} ne "") {
	    $ref_tag = "-r $self->{vcs}{tag}";
	}
	foreach my $component ( @{ $self->{components}{remaining} } ) {
	    $self->cvs_operation(component => $component,
		message => "Tagging '$component'",
		operation => "cvs -z3 rtag $branch_tag "
		. "$ref_tag $self->{options}{tag} fwcomponents/$component");
	}
    }

    print "\n";

    $self->tagging_endcheck();
}

sub merge {
    my $self = shift;

    if (!defined( $self->{options}{fw_type} )) {
	# merge the base directory (NOTE: subdirectories must be specified here manually!)
	my @base_subdirs = ( "build_sys" );
	my $start_tag_spec = $self->{options}{start_tag} ne "" 
			? "-j $self->{options}{start_tag}" : "";
	my $end_tag_spec = $self->{options}{end_tag} ne "" 
			? "-j $self->{options}{end_tag}" : "";

	$self->cvs_operation(component => "<base>",
		message => "Merging base directory",
		operation => "cvs -z3 update -l -P $start_tag_spec $end_tag_spec");

	foreach my $base_subdir (@base_subdirs) {
	    $self->cvs_operation(component => "<base>",
		message => "Merging directory '$base_subdir'",
		operation => "cvs -z3 update -dP $start_tag_spec $end_tag_spec $base_subdir");
	}
    }
    else {
        # get component list
	my $skip_missing_dirs = 0;
	get_components($self->{components}{all},
		    $self->{components}{current},
		    $skip_missing_dirs,
		    $self->{options}{fw_type},
		    "", "", "",
		    "none")
	or die "\nError during determining the components to merge\n";

	$self->filter_and_uniquify_components('all');

	foreach my $component ( @{ $self->{components}{all} } ) {
	    $self->merge_component($component);
	}
    }
    print "\n";

    $self->setview_endcheck();
}

1;
