package PP_VCS_SVN;

use strict;
use Carp;
use Cwd;
use Data::Dumper;
use File::Path;
use File::Find;
use pp_build_lib;
use pp_hw_data;
use base qw(PP_VCS);

sub determine_repo_information {
    my $self = shift;
    my $vc_info;
    if (not open $vc_info, "LC_ALL=C svn info 2>&1 |") {
	die "\nCannot determine repository information from your checkout.\n\n";
    }

    while (<$vc_info>) {
	chomp;
	my @info = split (': ', $_, 2);
	if (defined($info[0]) && $info[0] =~ /URL/) {
	    $self->{vcs}{url} = $info[1];
	}
	elsif (defined($info[0]) && $info[0] =~ /Repository Root/) {
	    $self->{vcs}{root} = $info[1];
	}
	elsif (defined($info[0]) && $info[0] =~ /Revision/) {
	    $self->{vcs}{rev} = $info[1];
	}
    }
    close $vc_info;

    # the "module" is the last part of the url,
    # without the slash (+1)
    $self->{vcs}{module} = substr($self->{vcs}{url},
		rindex($self->{vcs}{url},'/')+1);

    # the path should be so, that repo urls like
    # $root/$path/fwcomponents/$component can be constructed

    # remove repo root from path
    $self->{vcs}{path} = substr($self->{vcs}{url},
		length($self->{vcs}{root}));
    # remove module from path and the leading and trailing slash
    $self->{vcs}{path} = substr($self->{vcs}{path},
	    1, index($self->{vcs}{path},$self->{vcs}{module})-2);

    # check for r/w master server
    $self->{vcs}{master} = '';
    if (open my $master_fh, "$ENV{HOME}/.ppsetview/svnmaster") {
	$self->{vcs}{master} = <$master_fh>;
	$self->{vcs}{master} =~ s/\s*(\S*)\s*/$1/;
	if ($self->{vcs}{master} !~ m!//!) {
	    print "\nRepository Master is not a valid Subversion URL\n"
		. "You won't be able to commit anything!\n";
	    $self->{vcs}{master} = '';
	}
    }

    # assume current stable working copy format
    $self->{vcs}{format} = '1.4';
    if (open my $fh, ".svn/format") {
	my $format = <$fh>;
	$self->{vcs}{format} = '1.5' if $format > 8;
    }

    if ($self->{vcs}{master} ne '') {
	# just try to list repo root,
	# if we don't have read access this will fail
	system("svn ls $self->{vcs}{master} > /dev/null");
	# svn will ask a couple of time by itself before giving up
	if ($? != 0) {
	    # if we don't have access, unset master
	    $self->{vcs}{master} = '';
	}
	if ($self->{vcs}{master} eq $self->{vcs}{root}) {
	    # if we happen to checkout from master, we don't have to run
	    # updates again
	    $self->{vcs}{master} = '';
	}
    }

}


sub svn_operation {
    my $self = shift;
    my %options = @_;
    my $component = $options{component} || 'dummy';
    my $message = $options{message} || '';
    my $svn_operation = $options{operation} or croak "No operation given!";

    print "$message: ";
    $self->do_log($DEBUG{INFO}, "---- SVN: $message ...\n");
    $self->do_log($DEBUG{DEBUG}, "---- Executing: $svn_operation\n");

    my $svnfh;
    # start SVN operation
    if (not open $svnfh, "$svn_operation 2>&1 |") {
	print "failed\n";
	$self->do_log($DEBUG{WARN},
	    "!!!! SVN: Executing svn command ($svn_operation) failed ($!).\n\n");
	$self->{vcsserror_by_component}{$component} = 1;
	return 0;
    }

    # first column: item modifications
    # second column: item property modification
    while (<$svnfh>) {
	if ($_ =~ /^C[^h]/ or $_ =~ /^E/) {
	    $self->{conflicts_by_component}{$component} = 1;
	} elsif ($_ =~ /^M/ or /^[ADM ]M/) {
		$self->{uncommitted_by_component}{$component} = 1;
	} elsif ($_ =~ /^U/ or $_ =~ /^ U/ or $_ =~ /^G/) {
	    $self->{updated_by_component}{$component} = 1;
	} elsif ($_ =~ /^A[M ] / or $_ =~ /^D /) {
	    if ($svn_operation =~ / status/) {
		$self->{uncommitted_by_component}{$component} = 1;
	    } else {
		$self->{updated_by_component}{$component} = 1;
	    }
	}
	$self->do_log($DEBUG{INFO}, "$_");
    }
    close $svnfh;

    # evaluate exit code
    if ($? == 0) {
	print "done\n";
	$self->do_log($DEBUG{INFO}, "---- SVN: $message finished.\n\n");
	return 1;
    } else {
	print "failed\n";
	$self->do_log($DEBUG{WARN}, "!!!! SVN: $message failed.\n\n");
	$self->{vcserror_by_component}{$component} = 1;
	return 0;
    }
}


sub update_base {
    my $self = shift;

    # before doing an update, create md5sum of our precious self
    $self->create_checksum();

    # update the base directory (NOTE: subdirectories must be specified here manually!)
    my @base_subdirs = ( "build_sys" );
    if (not $self->svn_operation(component => "<base>", message => "Updating base directory", 
		operation => "svn update -N") ) {
	die "\nCould not update base\n\n";
    }

    foreach my $base_subdir (@base_subdirs) {
	if (not $self->svn_operation(component => "<base>",
	    message => "Updating directory '$base_subdir'",
	    operation => "svn update $base_subdir")) {
		die "\nCould not update '$base_subdir'\n";
	}
    }

    # check if some bad guy modified us and restart in this case
    if ($self->verify_checksum() != 0) {
	if ( defined($self->{conflicts_by_component}{"<base>"}) ) {
	    die "\nWe were modified and a conflict occured. Check cvs.log! Aborting ...\n\n";
	}
	elsif ( defined($self->{vcserror_by_component}{"<base>"}) ) {
	    die "\nWe were modified and a VCS error occured. Check cvs.log! Aborting ...\n\n";
	}
	else {
	    # We were modified. Someone should restart us
	    return 1;
	}
    }

    return 0;
}


sub checkout_component {
    my $self		    = shift;
    my %options		    = @_;
    my $component	    = $options{component};

    if ( $self->{options}{checkout_new_only} ) {
	if ( -d $component ) {
	    push @{ $self->{components}{existing} }, $component;
	    return 1;
	}
    }

    if (not $self->svn_operation(component => $component,
	    message => "Checking out '$component'",
	    operation => "svn update \"$component\" ") )
    {
	return 0;
    }

    return 1;
}


sub checkout_external {
    my $self		= shift;
    my %options		= @_;
    my $component	= $options{component};
    my $ext_name	= determine_ext_name($component);
    my $current_path	= $self->{vcs}{path};
    my($ext_tag, $tag, $fw_type, $board, $subboard, $product, $OEM,
       $light_OEM, $build_type) = split ",", $component;
    my $ext_base_path = "externals";
    my $ext_path = "$ext_base_path/$ext_name";

    # if fw is a branch
    if (($current_path eq 'trunk') or 
	($current_path =~ /^branches\/.+/)) {
	#empty tag means MAIN/trunk, so this a branch for sure
	if ($tag ne '') { 
	    my $vcfh;
	    if (not open $vcfh, "svn ls $self->{vcs}{root}/branches 2>&1 |") {
		print "failed\n";
		die "cannot verify branchness of component for external checkout";
	    }
	    
	    my $ext_is_branch = 0;
	    while (<$vcfh>) {
		chomp;
		if ($_ =~ /^$tag$/) {
		    $ext_is_branch = 1;
		    last;
		}
	    }
	    close $vcfh;
	    $current_path = $ext_is_branch ? "branches/$tag" : "tags/$tag";
	}
    }
    else {
	# the firmware has been checked out at a tag
	# so we just need to figure out where to get the externals
	my $ext_is_tag = 0;
	my $fw_ext_path = $current_path . '_EXT_' . $ext_name;
	my $fw_ext_path_found = 0;

	# check if the tag given by externals is a real tag (empty tag
	# is MAIN/trunk)
	if ($tag ne '') {
	    system ("svn ls $self->{vcs}{root}/tags/$tag &> /dev/null");
	    if ($? == 0) {
		    $ext_is_tag = 1;
	    }
	}
	# check if the expected tag of the external exists
	system ("svn ls $self->{vcs}{root}/$fw_ext_path &> /dev/null");
	if ($? == 0) {
		$fw_ext_path_found = 1;
	}

	if ($ext_is_tag) {
	    $current_path = "tags/$tag";
	}
	elsif ($fw_ext_path_found) {
	    $current_path = $fw_ext_path;
	}
	else {
	    die "\nCannot checkout external '$ext_name', because it "
	    . "has not been tagged correctly!\"n"
	}
    }

    # checkout external base system if not already there
    if(! -d $ext_path) {
	# create externals path if necessary
	! -d $ext_base_path and mkpath($ext_base_path);

	# store current work directory
	my $cwd = cwd();
	chdir($ext_base_path);

	if (not $self->svn_operation(component => $ext_path,
		message => "Checking out external '$ext_name'",
		operation => "svn checkout \"$self->{vcs}{root}/$current_path/fwbase\" "
		    . "\"$ext_name\"") )
	{
	    $self->{vcserror_by_component}{$ext_path} = 1;
	    chdir($cwd);
	    return 0;
	}
	chdir($cwd);
    }
    else {
	# external path already existing - make sure to update to the correct branch

	# store current work directory
	my $cwd = cwd();
	chdir($ext_path);

	# update fwbase (non-recursive)
	if (not $self->svn_operation(component => "<base>",
		    message => "Updating base directory", 
		    operation => "svn switch -N "
			. "\"$self->{vcs}{root}/$current_path/fwbase\"") )
	{
	    $self->{vcserror_by_component}{$ext_path} = 1;
	    chdir($cwd);
	    return 0;
	}

	# update build_sys
	my @base_subdirs = ( "build_sys" );
	foreach my $base_subdir (@base_subdirs) {
	    if (not $self->svn_operation(component => "<base>",
		    message => "Updating directory '$base_subdir'",
		    operation => "svn switch \"$self->{vcs}{root}/$current_path/"
		    . "fwbase/$base_subdir\" $base_subdir") )
	    {
		$self->{vcserror_by_component}{$ext_path} = 1;
		chdir($cwd);
		return 0;
	    }
	}
	chdir($cwd);
    }
    # set the external view
    $self->setview_external(fw_type => $fw_type, 
			    board => $board,
			    subboard => $subboard,
			    product => $product,
			    oem => $OEM,
			    lightoem => (defined $light_OEM ? $light_OEM : ''),
			    build_type => ($build_type 
					    ? $build_type 
					    : $self->{target_config}{build_type}),
			    ext_name => $ext_name,
			    ext_path => $ext_path);

    return 1;
}

# has to be made vcs specific because we need to call checkout_component
sub choose_target_oem {
    my $self = shift;

    my %oems;
    my $find_oems = sub {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)$/
		and $self->{target_config}{"board"} eq
		    $self->{target_config}{"product"}
		and $1 eq $self->{target_config}{"board"}) {

		$oems{$2} = 1;
	    } elsif ($_ =~ /^([^_]+)_([^_]+)_([^_]+)/
		     and $1 eq $self->{target_config}{"board"}
		     and $2 eq $self->{target_config}{"product"}) {

		$oems{$3} = 1;
	    }
	}
    };

    my %lightoems;
    my $find_lightoems = sub {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)_([^_]+)_([^_]+)$/
		and $1 eq $self->{target_config}{"board"}
		and $2 eq $self->{target_config}{"product"}
		and $3 eq $self->{target_config}{"oem"}) {

		$lightoems{$4} = 1;
	    }
	}
    };

    # check out OEM. this is currently needed for all firmware types.
    # find magic to decide once we have a non OEM firmware
    print "\n";
    if ($self->checkout_component(component => "OEM")) {
	# scan OEM directory
	find({ wanted => $find_oems, no_chdir => 0 }, "OEM");
	my @oems = sort keys %oems;
	# ask for OEMs
	$self->{target_config}{"oem"} = $self->chooser( list => \@oems,
	    list_hidden => [], default => $self->{target_config}{"oem"},
	    prompt => "Choose OEM>");

	# scan OEM directory
	find({ wanted => $find_lightoems, no_chdir => 0 }, "OEM");
	my @lightoems = sort keys %lightoems;
	if (scalar(@lightoems) > 0) {
	    unshift @lightoems, "";
	}
	# ask for light-OEMs
	$self->{target_config}{"lightoem"} = $self->chooser( list => \@lightoems,
	    list_hidden=> [], default => $self->{target_config}{"lightoem"},
	    prompt => "Choose light-OEM>");
    }

}


sub merge_component {
    my $self = shift;
    my $component = shift;
    my $postpone = ($self->{vcs}{format} eq '1.5') ? '--accept postpone' : '';

    if (not -d $component) {
	$self->checkout_component(component=>$component);
    }

    my $start_tag_spec = "$self->{vcs}{root}/$self->{options}{start_tag}/"
			."fwcomponents/$component";
    my $end_tag_spec = "$self->{vcs}{root}/$self->{options}{end_tag}"
			."/fwcomponents/$component";

    if (not $self->svn_operation(component => $component,
		message => "Merging '$component'",
		operation => "svn merge $postpone "
		    ."$start_tag_spec $end_tag_spec "
		    ."$component")) {
	return 0;
    }

    return 1;
}


sub setview {
    my $self = shift;
    my %seen;

    if (not $self->{options}{batch_mode}) {
	$self->read_target_config();
    }

    # ask for firmware type
    $self->choose_target_fw_type();
    # ask for board
    $self->choose_target_board();
    # ask for subboard
    $self->choose_target_subboard();
    # ask for product
    $self->choose_target_product();
    # ask for oem
    $self->choose_target_oem();
    # ask for the build type
    $self->choose_target_build_type();

    # write target configuration to file
    $self->write_target_config();

    $self->print_target_config();

    # we need Config.[sh|mk], generated by build system. generate fake if it isnt there
    $self->generate_config_sh_mk();

    # do we need to go further?
    if($self->{options}{dont_checkout}) {
	print "don't checkout selected\n";
	return;
    }

    # determine required components (do *not* skip missing directories)
    # non resolvable dependencies will be ignored (as possibly fixed in
    # repo)
    my $skip_missing_dirs = 0;
    get_components( $self->{components}{all},
		    $self->{components}{current},
		    $skip_missing_dirs,
		    $self->{target_config}{fw_type},
		    $self->{target_config}{board},
		    $self->{target_config}{subboard},
		    $self->{target_config}{product},
		   "none")
	or die "\nError during determining the required components\n";

    # list or cleanup mode
    if ( ($self->{options}{list} || $self->{options}{cleanup})
	&& ! $self->{options}{checkout_all}
	&& ! $self->{options}{checkout_pkg})
    {
	# cut of any subtrees from components
	# (always checkout full component)
	for (0 .. $#{ $self->{components}{all} }) {
	    $self->{components}{all}[$_] =~ s/\/.*//;
	}
	for (0 .. $#{ $self->{components}{current} }) {
	    $self->{components}{current}[$_] =~ s/\/.*//;
	}

	# from perl cookbook 4.8.2.3
	%seen = ();
	@seen {@{ $self->{components}{all} }} = ( );
	delete @seen {@{ $self->{components}{current} }};

	my @clean_components = ();
	foreach my $component (keys %seen) {
	    opendir(DIR, $component) && push(@clean_components, $component);
	    closedir(DIR);
	}

	my $clean_components = join(" ", @clean_components);
	if (! $clean_components) {
	    print "\nNothing to clean up.\n";
	}
	else {
	    if ($self->{options}{cleanup}) {
		my @not_cleaned_components = ();
		print "\nRemoving unnecessary components:\n$clean_components\n\nPress return to continue...";
		my $foo = <STDIN>;

		COMPONENT: foreach my $clean_component (@clean_components) {
		    my $vcfh;
		    print "Verifying status of $clean_component: ";

		    # run svn status to verify cleanliness of checkout
		    if (not open $vcfh, "svn status -q $clean_component 2>&1 |") {
			print "failed\n";
			push(@not_cleaned_components, $clean_component);
			next COMPONENT;
		    }

		    # any output indicates an unclean component
		    while (<$vcfh>) {
			    # component was somehow modified, request user interaction
			    close $vcfh;
			    print "modified\n";
			    push(@not_cleaned_components, $clean_component);
			    next COMPONENT;
		    }
		    close $vcfh;

		    rmtree($clean_component);
		    print "ok... removing...\n";
		}

		my $not_cleaned_components = join(" ", @not_cleaned_components);
		if ($not_cleaned_components) {
		    print "\nPlease check an manually remove not yet cleaned components:\n$not_cleaned_components\n";
		}
	    }
	    else {
		print "\nSuggest to clean unnecessary components:\n$clean_components\n\nCall ppsetview -S -c to clean up.\n";
	    }
	}

	# stop here?
	print "\n";
	return 0;
    }

    # do we want all components?
    if ( $self->{options}{checkout_all} ) {
	@{ $self->{components}{current} } = @{ $self->{components}{all} };
    }
    elsif ($self->{options}{checkout_pkg} ) {
	@{ $self->{components}{current} } = @{ $self->{components}{package} };
    }

    # cut of any subtrees (always checkout full component)
    for (0 .. $#{ $self->{components}{current} }) {
	$self->{components}{current}[$_] =~ s/\/.*//;
    }

    %seen = ();
    # uniq
    @{ $self->{components}{current} } = grep !$seen{$_}++, 
					    @{ $self->{components}{current} };

    # don't run again over already checkout base components
    $self->{components}{processed}{build_sys}++;
    $self->{components}{processed}{OEM}++;

    # run this loop at least once
    my $components_to_process = scalar(@{ $self->{components}{current} });
    my $iteration = 1;
    while ($components_to_process > 0) {
	print "Starting checkout iteration $iteration, " .
	      "$components_to_process components to process\n";
	$iteration++;

	foreach my $component (@{ $self->{components}{current} }) {
	    # skip already processed components
	    $self->{components}{processed}{$component}++ and next; 
	    $self->checkout_component(component => $component);
	}
	# non resolvable dependencies will be ignored (as possibly fixed in CVS)
	$skip_missing_dirs = 0;
	get_components( $self->{components}{all},
			$self->{components}{current},
			$skip_missing_dirs,
			$self->{target_config}{fw_type},
			$self->{target_config}{board},
			$self->{target_config}{subboard},
			$self->{target_config}{product},
		       "none")
	    or die "\nError during determining the required components\n";

	# cut of any subtrees (always checkout full component)
	for (0 .. $#{ $self->{components}{current} }) {
	    $self->{components}{current}[$_] =~ s/\/.*//;
	}

	%seen = ();
	# uniq
	@{ $self->{components}{current} } = grep !$seen{$_}++, 
						@{ $self->{components}{current} };

	$components_to_process = 0;
	foreach(@{ $self->{components}{current} }) {
		$self->{components}{processed}{$_}
		    or $components_to_process++;
	}
    }
    print "\n";

    if ($self->{vcs}{master} ne '') {
	# after checkin out from the mirror, switch to the
	# real repo again
	qx(svn switch --relocate $self->{vcs}{root} $self->{vcs}{master});
	# do an update after checking out from the mirror
	$self->svn_operation(component => '<base>',
		    message => "Updating checkout",
		    operation => "svn update");
    }

    $self->setview_endcheck();
}

sub add {
    my $self = shift;

#FIXME: this function needs some rethinking. Not sure how we do it, yet
    die  "This function is currently handled by RP.IT.\n"
	."Please contact RP.IT\@raritan.com to request a new component.\n";

}

sub tag {
    my $self = shift;

    # if branching, just copy the current rev to the branch name

    if ($self->{options}{branch_tag}) {
	if ($self->{options}{ref_tag}) {
	    # ref_tag has to be a real tag, no branch
	    $self->svn_operation(component => "<base>",
		message => "Creating branch tag '$self->{options}{tag}'"
			."from reference tag '$self->{options}{ref_tag}'. "
			."Please be patient. There is no output.",
		operation => "svn copy $self->{vcs}{root}/tags/"
			    ."$self->{options}{ref_tag} "
			    ."$self->{vcs}{root}/branches/"
			    ."$self->{options}{tag} "
			    ."-m \"by ppsetview: copied from tags/"
			    ."$self->{options}{ref_tag}\"")
	}
	else {
	    # use rev of wc to copy current path to new branch
	    $self->svn_operation(component => "<base>",
		message => "Creating branch tag '$self->{options}{tag}' "
			  ."Please be patient. There is no output.",
		operation => "svn copy -r $self->{vcs}{rev} "
			    ."$self->{vcs}{root}/"
			    ."$self->{vcs}{path} "
			    ."$self->{vcs}{root}/branches/"
			    ."$self->{options}{tag} "
			    ."-m \"by ppsetview: copied from "
			    ."$self->{vcs}{path} "
			    ."at revision $self->{vcs}{rev}\"")
	}
    }
    else {
	if ($self->{options}{ref_tag}) {
	    $self->svn_operation(component => "<base>",
		message => "Creating tag '$self->{options}{tag}'"
			."from reference tag '$self->{options}{ref_tag}'. "
			."Please be patient. There is no output.",
		operation => "svn copy $self->{vcs}{root}/tags/"
			    ."$self->{options}{ref_tag} "
			    ."$self->{vcs}{root}/tags/"
			    ."$self->{options}{tag} "
			    ."-m \"by ppsetview: copied from tags/"
			    ."$self->{options}{ref_tag}\"")
	}
	else {
	    # copy rev of wc to new tag
	    $self->svn_operation(component => "<base>",
		message => "Creating tag '$self->{options}{tag}' "
			."Please be patient. There is no output.",
		operation => "svn copy -r $self->{vcs}{rev} "
			    ."$self->{vcs}{root}/"
			    ."$self->{vcs}{path} "
			    ."$self->{vcs}{root}/tags/"
			    ."$self->{options}{tag} "
			    ."-m \"by ppsetview: copied "
			    ."$self->{vcs}{path} at rev "
			    ."$self->{vcs}{rev}\"");
	}
    }
    print "\n";

    $self->tagging_endcheck();
}

sub merge {
    my $self = shift;
    my $postpone = ($self->{vcs}{format} eq '1.5') ? '--accept postpone' : '';

    if (!defined( $self->{options}{fw_type} )) {
	# merge the base directory (NOTE: subdirectories must be specified here manually!)
	my @base_subdirs = ( "build_sys" );
	my $start_tag_spec = "$self->{vcs}{root}/$self->{options}{start_tag}/fwcomponents";
	my $end_tag_spec = "$self->{vcs}{root}/$self->{options}{end_tag}/fwcomponents";

	$self->svn_operation(component => "<base>",
		message => "Merging base directory",
		operation => "svn merge -N $postpone "
			    ."$start_tag_spec $end_tag_spec");

	foreach my $base_subdir (@base_subdirs) {
	    $self->svn_operation(component => "<base>",
		message => "Merging directory '$base_subdir'",
		operation => "svn merge $postpone "
			    ."$start_tag_spec/$base_subdir "
			    ."$end_tag_spec/$base_subdir $base_subdir");
	}
    }
    else {
        # get component list
	my $skip_missing_dirs = 0;
	get_components($self->{components}{all},
		    $self->{components}{current},
		    $skip_missing_dirs,
		    $self->{options}{fw_type},
		    "", "", "",
		    "none")
	or die "\nError during determining the components to merge\n";

	$self->filter_and_uniquify_components('all');

	foreach my $component ( @{ $self->{components}{all} } ) {
	    $self->merge_component($component);
	}
    }
    print "\n";

    $self->setview_endcheck();

}

1;
