#
#!/bin/bash
#
# (C) 2005 Peppercon AG, Zwickau (Sachs)
# Frank Hofmann <fho@peppercon.de>
#
# shell script to create peppercon documentation
# - extract ppdoc comments and save then as docbook file
#
# dtc: peppercon docbook tool chain
#
# call:
# ./c2d.sh [-f outputFormat] projectDir [projectDir]*
#
# - projectDir  = directory to process
# - outputFormat = [html|tex|dvi|ps|pdf|rtf]
#                  if not specified: pdf
#
# version 0.0.1 (2005-05-20)

# copyright
echo "================================================================================"
echo "c2d 0.0.1 (2005-05-20)"
echo "(C) 2005 Peppercon AG, Zwickau (Sachs)"
echo "================================================================================"

#PP_DOC_SYS_DIR should be in environment
if [ ${PP_DOC_SYS_DIR} ]; then
	execRoot=${PP_DOC_SYS_DIR}
else
	execRoot=`dirname $0`
fi
#echo $execRoot

# verify parameters
if [ $# -eq 0 ] ; then
	echo "parameters missing:"
	echo "(1) c2d [-f outputFormat] output_directory input_directory [input_directory]*"
	echo "Exiting (error #1)."
	exit 1
fi

# initialize variables
projectDir=""
projectSourceFile=""
docBookSourceFile="ppdoc.docbook"
outputFormat="pdf"
outputProduct="c2d"
pos=1

if [ $# -ge 3 ] ; then
	case "$1" in
		"-f") 
			outputFormat="$2"
			docbookDir="$3"
			# input dirs begin at position 3
			shift
			shift
			shift
			projectDir="$*"
			;;
	esac
else
	docbookDir="$1"
	shift
	projectDir="$*"
fi


case "$outputFormat" in
	"HTML") outputFormat="html";;
	"html") outputFormat="html";;
	"PDF") outputFormat="pdf";;
	"pdf") outputFormat="pdf";;
	"RTF") outputFormat="rtf";;
	"rtf") outputFormat="rtf";;
	"PS") outputFormat="ps";;
	"ps") outputFormat="ps";;
	"DVI") outputFormat="dvi";;
	"dvi") outputFormat="dvi";;
	"TEX") outputFormat="tex";;
	"tex") outputFormat="tex";;
	*)
		echo "unknown output format $1"
		exit 2
		;;
esac

echo "Output format: $outputFormat"

# dtc settings
currentDir=`pwd`
workingDir=`dirname $currentDir`
#buildDir="/tmp/build"
buildDir="$currentDir/doc"
productDir="$buildDir/$outputProduct"
tmpDir="$productDir/tmp"
outputDir="$productDir/output"
flavourDir="$outputDir/$outputFormat"

echo "================================================================================"
# output settings
echo "dtc settings:"
echo "docbook sourcefile: $docBookSourceFile"
echo "output format: $outputFormat"
echo "build doc for product: $outputProduct"
echo "working dir: $workingDir"
echo "build dir: $buildDir"
echo "product dir: $productDir"
echo "temp dir: $tmpDir"
echo "output dir: $outputDir"
#echo "flavour dir: $flavourDir"
echo "================================================================================"


echo "input directory: $projectDir"

# do the build directories exist?
if [ ! -d "$buildDir" ] ; then
	if mkdir "$buildDir" ; then
		echo ""
		#echo "build dir ($buildDir) successfully created."
	else
		echo "Cannot create build directory."
		echo "... c2d stopped due to previous i/o error (4)."
		exit 4
	fi
fi
#echo "Use build dir ($buildDir)."

if [ ! -d "$productDir" ] ; then
	if mkdir "$productDir" ; then
		echo ""
		#echo "product dir ($productDir) successfully created."
	else
		echo "Cannot create product directory."
		echo "... c2d stopped due to previous i/o error (4)."
		exit 4
	fi
fi
#echo "Use product dir ($productDir)."

# clear temp directory
#echo ""
#echo "clear temporary directory"
if [ -d "$tmpDir" ] ; then
	if rm -rf "$tmpDir" ; then
		echo ""
		#echo " ... done "
	else
		echo " ... failed."
		echo " ... c2d stopped (5)."
		exit 5
	fi
fi	

if [ ! -d "$tmpDir" ] ; then
	if mkdir "$tmpDir" ; then
		echo ""
		#echo "temp dir ($tmpDir) successfully created."
	else
		echo "Cannot create temp directory."
		echo "... c2d stopped due to previous i/o error (4)."
		exit 4
	fi
fi
#echo "Use temp dir ($tmpDir)."

if [ ! -d "$outputDir" ] ; then
	if mkdir "$outputDir" ; then
		echo ""
		#echo "output dir ($outputDir) successfully created."
	else
		echo "Cannot create output directory."
		echo "... c2d stopped due to previous i/o error (4)."
		exit 4
	fi
fi
#echo "Use output dir ($outputDir)."

if [ ! -d "$flavourDir" ] ; then
	if mkdir "$flavourDir" ; then
		echo ""
		#echo "flavour dir ($flavourDir) successfully created."
	else
		echo "Cannot create flavour directory."
		echo "... c2d stopped due to previous i/o error (4)."
		exit 4
	fi
fi
#echo "Use flavour dir ($flavourDir)."

# linking project parts to output directory
# ... should be optimized ...
echo "Linking files ... "
# a fake file for pdf, ps ,...
touch "$tmpDir/companylogo.png"

echo "================================================================================"

# call python script to convert from source file to docbook file
python $execRoot/comment2docbook.py "$tmpDir/$docBookSourceFile" $docbookDir $projectDir 

# check return code of project2docbook
if [ $? -ne 0 ] ; then
	echo "c2d stopped due to previous comment2docbook conversion error (6)."
	exit 6
fi

# verify docbook source file
if [ ! -f "$tmpDir/$docBookSourceFile" ] ; then
	echo "docbook source file ($tmpDir/$docBookSourceFile) not found"
	echo "c2d stopped (2)."
	exit 2
fi
echo "docbook source file ($tmpDir/$docBookSourceFile) found ..."

# verify file type: XML
docBookSourceFileType=`file "$tmpDir/$docBookSourceFile"`
if [ "$docBookSourceFileType" = "$docBookSourceFile: XML *" ] ; then
	echo "docbook source file ($tmpDir/$docBookSourceFile) tends not to be an XML file"
	echo "c2d stopped (3)."
	exit 3
fi
echo "docbook source file ($tmpDir/$docBookSourceFile) is XML file ..."

# check for unresolved xml tags
echo ""
echo "check for unresolved xml tags ..."
#echo "python entity.py $tmpDir/$docBookSourceFile" 

python $execRoot/entity.py "$tmpDir/$docBookSourceFile" "$tmpDir/$docBookSourceFile"
if [ $? -ne 0 ] ; then
	echo "... c2d stopped due to previous entity optimization error (15)."
	exit 15
fi
echo " ... done."

# check for included/referenced media files (images etc.)
# extract mediaobjects file list
optFile="$tmpDir/$docBookSourceFile"
echo "check for picture conversions in optimized docbook file "
pictureList="$tmpDir/images.ascii"
fullPictureList="$tmpDir/imagesFull.ascii"
#echo "(python mediaobjects.py $optFile $optFile $outputFormat $pictureList $tmpDir)"
python $execRoot/mediaobjects.py $optFile $optFile $outputFormat $pictureList $fullPictureList $tmpDir
retVal="$?"
if [ "$retVal" -ne 0 ] ; then
	echo "mediaobjects error: $retVal"
	echo "... c2d stopped due to previous optimization error (11)."
	exit 11
fi

# check file list - maybe we have to do a conversion
echo ""
echo "check for picture conversions ..."

if [ -f $pictureList ] ; then
	lineCount=`sed -n -e '$=' $pictureList`
	currentLine=0

	if [ ! "$lineCount" = "" ] ; then
		while [ "$lineCount" -gt 0 ] ; do
			currentLine=`expr $currentLine + 1`

			# read desired line and save in a temporary file
			currentFile=`head -$currentLine $pictureList | tail -1`
		
			echo "converting $currentFile ..."
			convert $currentFile
		
			lineCount=`expr $lineCount - 1`
		done
	fi
fi

echo "================================================================================"
# check for unresolved references with the use of dtc-ref
#echo ""
#echo "checking for unresolved references ..."
#echo "starting dtc-ref (python dtc-ref.py $tmpDir/$docBookSourceFileShort.opt.xmllint.opt --verify)"
#python dtc-ref.py "$tmpDir/$docBookSourceFileShort.opt.xmllint.opt" --verify
#if [ $? -ne 0 ] ; then
#	echo "...dtc stopped due to unresolved references (20)"
#	exit 20
#fi

# transform into desired output format ("flavour")
# use jw/docbook2*
echo "converting to $outputFormat ... "
#echo "starting jw (jw -o $flavourDir -b $outputFormat $tmpDir/$docBookSourceFile):"
jw -o $tmpDir -b $outputFormat "$tmpDir/$docBookSourceFile" > /dev/null
if [ $? -ne 0 ] ; then
	echo "... c2d stopped due to previous jw conversion error (8)"
	exit 8
fi

# cp the created files in the according subdirectory
echo "copying to destination dir"
echo "cp $tmpDir/*.$outputFormat $flavourDir/."
cp $tmpDir/*.$outputFormat $flavourDir/.
if [ $? -ne 0 ] ; then
	echo " copying failed (17)."
	exit 17
fi

if [ -f $fullPictureList ] ; then
	lineCount=`sed -n -e '$=' $fullPictureList`
	currentLine=0

	if [ ! "$lineCount" = "" ] ; then
		while [ "$lineCount" -gt 0 ] ; do
			currentLine=`expr $currentLine + 1`

			# read desired line and save in a temporary file
			currentFile=`head -$currentLine $fullPictureList | tail -1`
		
			echo "copying $currentFile ..."
			cp $currentFile $flavourDir/.
		
			lineCount=`expr $lineCount - 1`
		done
	fi
fi

# we are finished
echo "... dtc finished."
echo "================================================================================"

exit 0
