// system includes
#include <stdio.h>

// fw includes
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/xdefs.h>

// local includes
#include "cfg_conv.h"

static pp_hash_t* acl_objects = NULL;
static vector_t *users_to_delete = NULL;
static vector_t *users_to_cleanup = NULL;
static pp_hash_t *group_mapping = NULL;

static int acl_old_union(pp_acl_t* me, pp_acl_t* other, int optimistic) {
    const char* ae_key;
    pp_acl_entry_t *ae_me;
    pp_acl_entry_t *ae_other;

    assert(me);
    
    if(!other) {
        return PP_SUC;
    }

    MUTEX_LOCK(&me->objects->mtx);
    // loop over me objects
    for (ae_key = pp_hash_get_first_key(me->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(me->acl)) {
	// if found in other objects
	if(NULL != (ae_other = pp_hash_get_entry(other->acl, ae_key))) {
	    ae_me = pp_hash_get_entry(me->acl, ae_key);
	    // apply rule
	    if(ae_me->is_negative != ae_other->is_negative) {
		ae_me->is_negative = optimistic ? 0 : 1;
	    }
	}
    }
    // loop over other objects
    for (ae_key = pp_hash_get_first_key(other->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(other->acl)) {
	// if not found in me
	if(NULL == (ae_me = pp_hash_get_entry(me->acl, ae_key))) {
	    ae_other = pp_hash_get_entry(other->acl, ae_key);
	    // duplicate acl entry
	    ae_me = malloc(sizeof(pp_acl_entry_t));
	    *ae_me = *ae_other;
	    // add acl entry
	    pp_hash_set_entry(me->acl, ae_key, ae_me, free);
	}
    }
    
    MUTEX_UNLOCK(&me->objects->mtx);
    return PP_SUC;
}

static void acl_object_entry_free(void * _object) {
    pp_acl_object_entry_t * oe = (pp_acl_object_entry_t *)_object;
    if (oe) {
        free(oe->name);
        free(oe->longname);
        pp_hash_delete(oe->perms);
        free(oe->type);
        free(oe);
    }
}

static void acl_perm_entry_free(void* _perm_entry) {
    pp_acl_perm_entry_t* perm_entry = (pp_acl_perm_entry_t *)_perm_entry;
    if (perm_entry) {
	free(perm_entry->name);
	free(perm_entry);
    }
}

static pp_acl_object_entry_t* acl_get_object(const char* object) {
    pp_acl_object_entry_t *oe;
    
    assert(object);
    
    if (!acl_objects) {
        acl_objects = pp_hash_create(32);
    }
    
    if (NULL == (oe = pp_hash_get_entry(acl_objects, object))) {
        /* create that entry */
        oe = malloc(sizeof(pp_acl_object_entry_t));
        memset(oe, 0, sizeof(pp_acl_object_entry_t));
    
        oe->name = strdup(object);
        oe->perms = pp_hash_create(3);
        
        pp_hash_set_entry(acl_objects, object, oe, acl_object_entry_free);
    }
    
    return oe;
}        

static pp_acl_perm_entry_t* acl_get_perm(const pp_acl_object_entry_t *oe, 
                                        const char* perm) {
    pp_acl_perm_entry_t *pe;
   
    assert(oe);
    assert(perm);
    
    if (NULL == (pe = pp_hash_get_entry(oe->perms, perm))) {
        /* create that entry */
        pe = malloc(sizeof(pp_acl_perm_entry_t));
        pe->name = strdup(perm);
        
        pp_hash_set_entry(oe->perms, perm, pe, acl_perm_entry_free);
    }
    
    return pe;
}        

static int acl_add_old_entry(pp_acl_t* acl, const char* object,
	                     const char* perm, const int negative) {
    pp_acl_object_entry_t *oe;
    pp_acl_perm_entry_t *perm_ce;
    pp_acl_entry_t *ae;
    char * s;
    int s_size, ret = PP_ERR;

    assert(acl && object && perm);

    MUTEX_LOCK(&acl->objects->mtx);
    
    if ((oe = acl_get_object(object)) == NULL) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
	goto bail;
    }

    if ((perm_ce = acl_get_perm(oe, perm)) == NULL) {
	/* perm not allowed for object */
	errno = PP_ACL_ERR_OBJECT_PERM_DOESNT_EXIST;
	goto bail;
    }

    ae = malloc(sizeof(pp_acl_entry_t));
    ae->object = oe;
    ae->perm = perm_ce;
    ae->is_negative = negative;

    /* FIXME: use static string */
    s_size = strlen(object) + strlen(perm) + 2;
    s = malloc(s_size);
    snprintf(s, s_size, "%s:%s", object, perm);
    pp_hash_set_entry(acl->acl, s, ae, free);
    free(s);
    ret = PP_SUC;

 bail:
    MUTEX_UNLOCK(&acl->objects->mtx);
    return ret;
}

static pp_acl_t* acl_load_old_from_cfg(const char* uidstr) {
    const char* obj;
    const char* perm;
    pp_acl_t* acl = NULL;
    pp_cfg_iter_t* oiter, *piter;

    if (PP_ERR != pp_cfg_iter_create(&oiter, "user[%s].acl[", uidstr)) {
	while(pp_cfg_iter_next(oiter, &obj, NULL)) {
	    if (PP_ERR == pp_cfg_iter_create2(&piter, oiter, "["))
		continue;
	    while(pp_cfg_iter_next(piter, NULL, &perm)) {
		if (perm[0] == '+' || perm[0] == '-') {
                    if(!acl) {
                        acl = pp_acl_create();
                    }
		    if(0 > acl_add_old_entry(acl, obj, 
                                             &perm[1], perm[0] != '+')) {
			pp_log("Adding ACL entry '%s: %s' failed: %s\n",
			       obj, perm, pp_error_string(errno));
		    }
		} else {
		    pp_log("ACL parse error\n");
		}
	    }
	    pp_cfg_iter_destroy(piter);
	}
	pp_cfg_iter_destroy(oiter);
    }
    
    return acl;
}

static int convert_acl_for_group(const pp_acl_t *acl, const char *groupname) {
    pp_acl_entry_t *ae;
    pp_cfg_iter_t *oiter;
    const char* idx, *type, *object;
    int ret;
    char permtmp1[65], permtmp2[65];
    
    assert(groupname);
    
    if (PP_ERR != (ret = pp_cfg_iter_create_at_layer(&oiter,
                                                     PP_PROFILE_LOCAL_CODE,
				                     "device.perms["))) {
	while (pp_cfg_iter_next(oiter, &idx, NULL)) {
	    if (PP_ERR != pp_cfg_iter_get(oiter, &type, "type") &&
		PP_ERR != pp_cfg_iter_get(oiter, &object, "object")) {
                if(strcmp(type, PP_CD_PERMTYPE_PORT_STR)) {
                    /* user or admin perm */
                    /* set to 'yes' if we had change or allow permission,
                       'no' otherwhise */
                    snprintf(permtmp1, sizeof(permtmp1), "%s:change", object);
                    snprintf(permtmp2, sizeof(permtmp2), "%s:allow", object);
                    if(acl &&
                       ((NULL != (ae = pp_hash_get_entry(acl->acl, permtmp1)) &&
                         !ae->is_negative) ||
                        (NULL != (ae = pp_hash_get_entry(acl->acl, permtmp2)) &&
                         !ae->is_negative))) {
                        pp_um_group_set_permission(groupname, object,
                                                   pp_acl_raasip_yes_str);
                    } else {
                        /* either permission not set or wrong value */
                        pp_um_group_set_permission(groupname, object,
                                                   pp_acl_raasip_no_str);
                    }       
                } else {
                    /* port perm */
                    /* set to 'control' if we had port permission, 
                       'none' otherwhise.
                       the 'view' permission may not be derived from old cfg */
                    snprintf(permtmp2, sizeof(permtmp2), "%s:allow", object);
                    if(acl &&
                       (NULL != (ae = pp_hash_get_entry(acl->acl, permtmp2)) &&
                        !ae->is_negative)) {
                        pp_um_group_set_permission(groupname, object,
                                                   pp_acl_raasip_control_str);
                    } else {
                        /* either permission not set or wrong value */
                        pp_um_group_set_permission(groupname, object,
                                                   pp_acl_raasip_none_str);
                    }       
                }
	    }
	}
    }
    pp_cfg_iter_destroy(oiter);

    return ret;
}

static int convert_all_groups(void) {
    int ret = 0, gid;
    pp_cfg_iter_t* iter;
    const char *groupname, *gidstr;
    char *gtmp = NULL;
        
    if (PP_ERR != pp_cfg_iter_create_at_layer(&iter, PP_PROFILE_LOCAL, 
                                              "user[]")) {
        while (pp_cfg_iter_next(iter, &gidstr, NULL)) {
            int is_group;
            int created = 0;
            char alternate_name[65];

            /* we only want groups at the moment */
            pp_cfg_is_enabled_at_layer(PP_PROFILE_LOCAL, &is_group,
                                       "user[%s].is_group", gidstr);
            
            if(!is_group) {
                continue;
            }
            
            /* check, if 'gidstr' is our group id or group name */
            if((gid = pp_strtol(gidstr, -1, 10, NULL)) < 0) {
                /* gidstr is not a number -> old user vector */
                char *parent = NULL;

                groupname = gidstr;

                if(PP_ERR == pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL,
                                                        &parent, 
                                                        "user[%s].parent",
                                                        gidstr)) {
                    gid = 0;
                } else {
                    gid = -1;
                }
                free(parent);
            } else {
                pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &gtmp,
                                           "user[%s].login", gidstr);
                if(!gtmp) {
                    /* we could not get a groupname, set default one */
                    groupname = "noname";
                } else {
                    /* we got groupname from config, use that */
                    groupname = gtmp;
                }
            }
                
            if(gid != 0) {
                pp_acl_t *acl;
                char old_groupname[65];
                
                strncpy(old_groupname, groupname, sizeof(old_groupname));

                /* create new generic group of specified name */
                while(!created) {
                    if(PP_SUC == pp_um_group_create(groupname, 0)) {
                        created = 1;
                    } else if (errno != PP_UM_ERR_GROUP_ALREADY_EXISTS || 
                               sizeof(alternate_name) < strlen(groupname) + 1) {
                        /* maximum size of groupname reached, break */
                        break;
                    } else {
                        /* group of that name exists, append '_' to groupname
                           and try again */
                        snprintf(alternate_name, sizeof(alternate_name),
                                 "%s_", groupname);
                        groupname = alternate_name;
                    }
                }
                
                if(!created) {
                    /* group was not created, this should not happen */
                    abort();
                    goto iter_done;
                }
                pp_hash_set_entry(group_mapping, gidstr, 
                                  strdup(groupname), free);
                pp_log("created group %s from deprecated group %s\n",
                               groupname, old_groupname);
                
                /* get acl */
                acl = acl_load_old_from_cfg(gidstr);
                
                /* convert acl */
                convert_acl_for_group(acl, groupname);
                
                pp_acl_destroy(acl);
            } else  {
                /* supergroup - register mapping to 'admin' only */
                pp_hash_set_entry(group_mapping, gidstr, strdup("Admin"), free);
            }
            
            /* prepare to delete old user[] entry */
            vector_add(users_to_delete, strdup(gidstr));

 iter_done:            
            free(gtmp);
            gtmp = NULL;
        }
        pp_cfg_iter_destroy(iter);
    }

    return ret;
}

static int convert_all_users(void) {
    int ret = 0;
    pp_cfg_iter_t* iter;
    const char *uidstr, *username;
    int uid;

    if (PP_ERR != pp_cfg_iter_create_at_layer(&iter, PP_PROFILE_LOCAL, 
                                              "user[]")) {
        while (pp_cfg_iter_next(iter, &uidstr, NULL)) {
            int is_group;
            int created = 0;
            char alternate_name[65];
            char *utmp = NULL;

            /* we only want users this time */
            pp_cfg_is_enabled_at_layer(PP_PROFILE_LOCAL, &is_group,
                                       "user[%s].is_group", uidstr);
            
            if(is_group) {
                continue;
            }
            
            /* check, if 'uidstr' is our user id or old user name */
            if((uid = pp_strtol(uidstr, -1, 10, NULL)) < 0) {
                /* uidstr is not a number -> old user vector */
                char *parent = NULL;
                char newuidstr[8];

                username = uidstr;

                if(PP_ERR == pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL,
                                                        &parent, 
                                                        "user[%s].parent",
                                                        uidstr)) {
                    uid = 0;
                } else {
                    uid = -1;
                }
                free(parent);
                
                if(uid != 0) {
                    /* create new user of specified name */
                    
                    while(!created) {
                        if(PP_SUC == pp_um_user_create(username, NULL, 0, 
                                                       "<Unknown>", NULL, 0)) {
                            uid = pp_um_user_get_uid(username);
                            
                            assert(uid != PP_ERR);
                            
                            snprintf(newuidstr, sizeof(newuidstr), "%d", uid);
                            created = 1;
                        } else if (errno != PP_UM_ERR_USER_ALREADY_EXISTS || 
                                   sizeof(alternate_name) < 
                                       strlen(username) + 1) {
                            /* maximum size of groupname reached, break */
                            break;
                        } else {
                            /* user of that name exists, append '_' to username
                               and try again */
                            snprintf(alternate_name, sizeof(alternate_name),
                                     "%s_", username);
                            username = alternate_name;
                        }
                    }
                    
                    if(!created || uid < 0) {
                        /* user was not created, this should not happen */
                        abort();
                        continue;
                    }
                } else {
                    /* overwrite predefined administrator */
                    strcpy(newuidstr, "0");
                }
                    
                pp_cfg_copy_at_layer(PP_PROFILE_LOCAL, newuidstr, 
                                     "user[%s]", uidstr);
                
                /* perhaps we overwrote modified login name... */
                pp_cfg_set_at_layer(PP_PROFILE_LOCAL, username,
                                    "user[%u].login", uid);

                pp_log("found deprecated user %s, created new user %s\n",
                       uidstr, username);
                    
                /* prepare to delete old user[] entry */
                vector_add(users_to_delete, strdup(uidstr));
            } else {
                /* uid is a real ID, check if we are already RAASIP */
                int gid;
                
                if(pp_cfg_get_int_at_layer_nodflt(PP_PROFILE_LOCAL, &gid,
                                                  "user[%s].group_id", uidstr)
                       == PP_SUC) {
                    /* ok... user format is correct and we already have RAASIP
                       compliant GID set... no need for conversion! */
                    
                    continue;
                }
                
                pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &utmp,
                                           "user[%s].login", uidstr);
                if(!utmp) {
                    /* we could not get a groupname, set default one */
                    username = "noname";
                } else {
                    /* we got groupname from config, use that */
                    username = utmp;
                }
                
                pp_log("found user %s\n", username);
            }

            if(uid > 0) {
                pp_acl_t *acl;
                vector_t *grp_acls = vector_new(NULL, 10,
                                                (void*)pp_acl_destroy);
                vector_t *in_group = vector_new(NULL, 10, free);
                int grp_acls_sz;
                const char *groupname;
                const char *gidstr;
                
                /* convert acl for 'normal' users only */
                
                /* get acl */
                acl = acl_load_old_from_cfg(uidstr);
                
                /* get group settings */
                for(gidstr = pp_hash_get_first_key(group_mapping);
                    gidstr; gidstr = pp_hash_get_next_key(group_mapping)) {
                    char *group_members = NULL;
                    pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &group_members,
                                               "user[%s].members", gidstr);
                    if(group_members) {
                        char *uid_token;
                        char *tok_ptr;

                        /* parse members */
                        for(uid_token = strtok_r(group_members, ":", &tok_ptr);
                            uid_token; 
                            uid_token = strtok_r(NULL, ":", &tok_ptr)) {
                            /* check if current user is member */
                            if(!strcmp(uidstr, uid_token)) {
                                /* get acl */
                                vector_add(grp_acls,
                                           acl_load_old_from_cfg(gidstr));
                                vector_add(in_group, strdup(gidstr));
                                break;
                            }
                        }
                        free(group_members);
                    }                                               
                }
                grp_acls_sz = vector_size(grp_acls);
                
                /* if acl == NULL and user
                   - in no group, assign to <Unknown>
                   - in exactly one group, assign to that group
                   else (individual settings or in more than one group)
                   - merge and convert new acl */
                
                if(!acl && grp_acls_sz == 0) {
                    groupname = "<Unknown>";
                    pp_log("added user %s to default group\n", username);
                } else if(!acl && grp_acls_sz == 1) {
                    groupname = pp_hash_get_entry(group_mapping, 
                                                  vector_get(in_group, 0));
                    pp_log("added user %s to group %s\n", username, groupname);
                } else {
                    char new_groupname[65];
                    int i;
		    created = 0;
                    
                    /* merge with group settings */
                    if(!acl) {
                        acl = pp_acl_create();
                    }
                    
                    for(i = 0; i < grp_acls_sz; ++i) {
                        acl_old_union(acl, vector_get(grp_acls, i), 1);
                    }

                    /* generate new individual group */
                    snprintf(new_groupname, sizeof(new_groupname),
                             "%c%s", um_individual_group_prefix, username);
                    groupname = new_groupname;
                    while(!created) {
                        if(PP_SUC == pp_um_group_create(groupname, 1)) {
                            created = 1;
                        } else if (errno == PP_UM_ERR_GROUP_ALREADY_EXISTS) {
                            /* group of that name exists, append '_' to
                               groupname and try again */
                            snprintf(new_groupname, sizeof(new_groupname),
                                     "%s_", groupname);
                        } else {
                            break;
                        }
                    }
                    
                    if(!created) {
                        /* group was not created, this should not happen */
                        abort();
                    } else {
                        /* convert acl */
                        convert_acl_for_group(acl, groupname);
                        
                        pp_log("created individual group %s for user %s\n",
                               groupname, username);
                    }
                }
                
                /* set group for user */
                pp_um_user_set_group(username, groupname);
                   
                pp_acl_destroy(acl);
                vector_delete(grp_acls);
            } else if (uid == 0) {
                /* Admin specific settings */
                
                /* Admin is always member of group Admin */
                pp_cfg_set("0", "user[0].group_id");
            }
            
            /* prepare to clean up user[] entry */
            snprintf(alternate_name, sizeof(alternate_name), "%d", uid);
            vector_add(users_to_cleanup, strdup(alternate_name));

            free(utmp);
        }
        pp_cfg_iter_destroy(iter);
    }

    return ret;
}

int conf_old_um_to_raasip(void) {
    int ret, i;
    pp_cfg_tx_t *tx;

    /* users to delete after conversion */
    users_to_delete = vector_new(NULL, 10, free);
    /* users to clean up after conversion */
    users_to_cleanup = vector_new(NULL, 10, free);
    
    /* mapping old groupname to new groupname */
    group_mapping = pp_hash_create(10);

    /* start transaction, great deeds are to be done! */
    tx = pp_cfg_tx_begin(1);
    
    if(PP_ERR != (ret = pp_um_init())) {
        ret |= convert_all_groups();
        ret |= convert_all_users();
    }
    
    /* cleanup modified users */
    for(i = 0; i < (int)vector_size(users_to_cleanup); ++i) {
        char *uidstr = (char*)vector_get(users_to_cleanup, i);
        const char *keystodel[] = { "parent", "is_group", "acl", NULL };
        int key;
        
        /* last elem must be NULL! */
        assert(keystodel[sizeof(keystodel) / sizeof(char*) - 1] == NULL);
        
        for(key = 0; keystodel[key]; ++key) {
            pp_cfg_remove_at_layer(PP_PROFILE_LOCAL, "user[%s].%s",
                                   uidstr, keystodel[key]);
        }
    }
    
    vector_delete(users_to_cleanup);

    /* delete old users, which have been regenerated */
    for(i = 0; i < (int)vector_size(users_to_delete); ++i) {
        pp_cfg_remove_at_layer(PP_PROFILE_LOCAL, "user[%s]",
                               (char*)vector_get(users_to_delete, i));
    }
    
    vector_delete(users_to_delete);
    
    pp_hash_delete(group_mapping);
    
    if (ret == PP_SUC) {
        ret = pp_cfg_tx_commit(tx);
    } else {
        pp_cfg_tx_rollback(tx);
    }
    
    pp_um_cleanup();

    return ret;
}
