/**
 * hex_dump.c
 *
 * Prints out a hex dump of memory area (byte-, word-, or dword-wise)
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <stdio.h>
#include <ctype.h>
#include "hex_dump.h"

static void hex_dump_b(const unsigned char* p, unsigned int size, int phys_addr)
{
    unsigned int i, j;
    unsigned char b[16];
    for (i = 0; i < size; i += 16) {
        printf("%08x:", phys_addr + i);

        /* pre-cache values to avoid multiple mem reads */
        for (j = 0; j < 16; j++)
            if (i + j < size) b[j] = p[i + j];

        /* hex display */
        for (j = 0; j < 8; j++)
            if (i + j >= size) printf("   ");
            else printf(" %02x", b[j]);
        printf(" ");
        for (j = 8; j < 16; j++)
            if (i + j >= size) printf("   ");
            else printf(" %02x", b[j]);
        printf("  ");

        /* ascii display */
        for (j = 0; j < 8; j++)
            if (i + j >= size) printf(" ");
            else if (!isgraph(b[j])) printf(".");
            else printf("%c", b[j]);
        printf(" ");
        for (j = 8; j < 16; j++)
            if (i + j >= size) printf(" ");
            else if (!isgraph(b[j])) printf(".");
            else printf("%c", b[j]);
        printf("\n");
    }
}

static void hex_dump_w(const unsigned short* p, unsigned int size, int phys_addr)
{
    unsigned int i, j, k;
    unsigned short b[8];
    for (i = 0; i < size; i += 8) {
        printf("%08x:", phys_addr + i * 2);

        /* pre-cache values to avoid multiple mem reads */
        for (j = 0; j < 8; j++)
            if (i + j < size) b[j] = p[i + j];

        /* hex display */
        for (j = 0; j < 4; j++)
            if (i + j >= size) printf("         ");
            else printf(" %04x", b[j]);
        printf(" ");
        for (j = 4; j < 8; j++)
            if (i + j >= size) printf("         ");
            else printf(" %04x", b[j]);
        printf("  ");

        /* ascii display */
        for (j = 0; j < 4; j++) {
            if (i + j >= size) printf("    ");
            else for (k = 0; k < 2; k++) {
                char c = ((char*)(b + j))[k];
                if (!isgraph(c)) printf(".");
                else printf("%c", c);
            }
            printf(" ");
        }
        printf(" ");
        for (j = 4; j < 8; j++) {
            if (i + j >= size) printf(" ");
            else for (k = 0; k < 2; k++) {
                char c = ((char*)(b + j))[k];
                if (!isgraph(c)) printf(".");
                else printf("%c", c);
            }
            printf(" ");
        }
        printf("\n");
    }
}

static void hex_dump_d(const unsigned int* p, unsigned int size, int phys_addr)
{
    unsigned int i, j, k;
    unsigned int b[4];
    for (i = 0; i < size; i += 4) {
        printf("%08x:", phys_addr + i * 4);

        /* pre-cache values to avoid multiple mem reads */
        for (j = 0; j < 4; j++)
            if (i + j < size) b[j] = p[i + j];

        /* hex display */
        for (j = 0; j < 2; j++)
            if (i + j >= size) printf("         ");
            else printf(" %08x", b[j]);
        printf(" ");
        for (j = 2; j < 4; j++)
            if (i + j >= size) printf("         ");
            else printf(" %08x", b[j]);
        printf("  ");

        /* ascii display */
        for (j = 0; j < 2; j++) {
            if (i + j >= size) printf("    ");
            else for (k = 0; k < 4; k++) {
                char c = ((char*)(b + j))[k];
                if (!isgraph(c)) printf(".");
                else printf("%c", c);
            }
            printf(" ");
        }
        printf(" ");
        for (j = 2; j < 4; j++) {
            if (i + j >= size) printf("    ");
            else for (k = 0; k < 4; k++) {
                char c = ((char*)(b + j))[k];
                if (!isgraph(c)) printf(".");
                else printf("%c", c);
            }
            printf(" ");
        }
        printf("\n");
    }
}

void hex_dump(const unsigned char* p, unsigned int size, int base, int phys_addr)
{
    switch (base) {
    case 4:  hex_dump_d((const unsigned int*)p, size / 4, phys_addr); break;
    case 2:  hex_dump_w((const unsigned short*)p, size / 2, phys_addr); break;
    default: hex_dump_b(p, size, phys_addr);
    }
}
