/*
 * Dropbear - a SSH2 server
 * 
 * Copyright (c) 2002,2003 Matt Johnston
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. */

#include "includes.h"
#include "packet.h"
#include "buffer.h"
#include "session.h"
#include "dbutil.h"
#include "channel.h"
#include "chansession.h"
#include "sshpty.h"
#include "termcodes.h"
#include "ssh.h"
#include "random.h"
#include "utmp.h"
#include "x11fwd.h"
#include "agentfwd.h"
#include "runopts.h"

/* Handles sessions (either shells or programs) requested by the client */

static int sessioncommand(struct Channel *channel, struct ChanSess *chansess,
		int iscmd, int issubsys);
static int sessionsignal(struct ChanSess *chansess);
static int noptycommand(struct Channel *channel, struct ChanSess *chansess);
static int sessionwinchange(struct ChanSess *chansess);
static void addchildthread(struct ChanSess *chansess, pthread_t thread_id);
static void closechansess(struct Channel *channel);
static int newchansess(struct Channel *channel);
static void chansessionrequest(struct Channel *channel);

static void send_exitsignalstatus(struct Channel *channel);
static void send_msg_chansess_exitstatus(struct Channel * channel,
		struct ChanSess * chansess);
static int sesscheckclose(struct Channel *channel);
//static void get_termmodes(struct ChanSess *chansess);


/* required to clear environment */
extern char** environ;

static int sesscheckclose(struct Channel *channel) {
	struct ChanSess *chansess = (struct ChanSess*)channel->typedata;
	return chansess->thread_exited;
}

/* send the exit status or the signal causing termination for a session */
/* XXX server */
static void send_exitsignalstatus(struct Channel *channel) {

	struct ChanSess *chansess = (struct ChanSess*)channel->typedata;

	if (chansess->thread_exited) {
		send_msg_chansess_exitstatus(channel, chansess);
	}
}

/* send the exitstatus to the client */
static void send_msg_chansess_exitstatus(struct Channel * channel,
		struct ChanSess * chansess) {

	GLOBALS(g);
	assert(chansess->thread_exited);

	CHECKCLEARTOWRITE();

	buf_putbyte(g->ses.writepayload, SSH_MSG_CHANNEL_REQUEST);
	buf_putint(g->ses.writepayload, channel->remotechan);
	buf_putstring(g->ses.writepayload, "exit-status", 11);
	buf_putbyte(g->ses.writepayload, 0); /* boolean FALSE */
	buf_putint(g->ses.writepayload, 0);

	encrypt_packet();

}

/* set up a session channel */
static int newchansess(struct Channel *channel) {

	struct ChanSess *chansess;

	assert(channel->typedata == NULL);

	chansess = (struct ChanSess*)m_malloc(sizeof(struct ChanSess));
	chansess->cmd = NULL;

	/* pty details */
	chansess->master = -1;
	chansess->slave = -1;
	chansess->tty = NULL;
	chansess->term = NULL;

	chansess->thread_exited = 0;

	channel->typedata = chansess;

#ifndef DISABLE_X11FWD
	chansess->x11listener = NULL;
	chansess->x11authprot = NULL;
	chansess->x11authcookie = NULL;
#endif

#ifndef DISABLE_AGENTFWD
	chansess->agentlistener = NULL;
	chansess->agentfile = NULL;
	chansess->agentdir = NULL;
#endif

	return 0;

}

/* clean a session channel */
static void closechansess(struct Channel *channel) {

	struct ChanSess *chansess;
	unsigned int i;
	GLOBALS(g);

	chansess = (struct ChanSess*)channel->typedata;

	send_exitsignalstatus(channel);

	TRACE(("enter closechansess"))
	if (chansess == NULL) {
		TRACE(("leave closechansess: chansess == NULL"))
		return;
	}

	m_free(chansess->cmd);
	m_free(chansess->term);

	if (chansess->tty) {
		/* write the utmp/wtmp login record */
#if 0
		struct logininfo *li;
		li = login_alloc_entry(chansess->pid, g->ses.authstate.username,
				g->ses.remotehost, chansess->tty);
		login_logout(li);
		login_free_entry(li);
#endif

		pty_release(chansess->tty);
		m_free(chansess->tty);
	}

#ifndef DISABLE_X11FWD
	x11cleanup(chansess);
#endif

#ifndef DISABLE_AGENTFWD
	agentcleanup(chansess);
#endif

	/* clear child thread entries */
	for (i = 0; i < g->svr_ses.childthreadsize; i++) {
		if (g->svr_ses.childthreads[i].chansess == chansess) {
			assert(g->svr_ses.childthreads[i].used);
			g->svr_ses.childthreads[i].used = 0;
			g->svr_ses.childthreads[i].chansess = NULL;
		}
	}
				
	m_free(chansess);

	TRACE(("leave closechansess"))
}

/* Handle requests for a channel. These can be execution requests,
 * or x11/authagent forwarding. These are passed to appropriate handlers */
static void chansessionrequest(struct Channel *channel) {

	unsigned char * type = NULL;
	unsigned int typelen;
	unsigned char wantreply;
	int ret = 1;
	struct ChanSess *chansess;
	GLOBALS(g);

	TRACE(("enter chansessionrequest"))

	type = buf_getstring(g->ses.payload, &typelen);
	wantreply = buf_getbyte(g->ses.payload);

	if (typelen > MAX_NAME_LEN) {
		TRACE(("leave chansessionrequest: type too long")) /* XXX send error?*/
		goto out;
	}

	chansess = (struct ChanSess*)channel->typedata;
	assert(chansess != NULL);
	TRACE(("type is %s", type))

	if (strcmp(type, "window-change") == 0) {
		ret = sessionwinchange(chansess);
	} else if (strcmp(type, "shell") == 0) {
		ret = sessioncommand(channel, chansess, 0, 0);
	} else if (strcmp(type, "pty-req") == 0) {
		ret = DROPBEAR_SUCCESS;
	} else if (strcmp(type, "exec") == 0) {
		ret = sessioncommand(channel, chansess, 1, 0);
	} else if (strcmp(type, "subsystem") == 0) {
		ret = sessioncommand(channel, chansess, 1, 1);
#ifndef DISABLE_X11FWD
	} else if (strcmp(type, "x11-req") == 0) {
		//ret = x11req(chansess);
		ret = DROPBEAR_FAILURE;
#endif
#ifndef DISABLE_AGENTFWD
	} else if (strcmp(type, "auth-agent-req@openssh.com") == 0) {
		ret = agentreq(chansess);
#endif
	} else if (strcmp(type, "signal") == 0) {
		ret = sessionsignal(chansess);
	} else {
		/* etc, todo "env", "subsystem" */
	}

out:

	if (wantreply) {
		if (ret == DROPBEAR_SUCCESS) {
			send_msg_channel_success(channel);
		} else {
			send_msg_channel_failure(channel);
		}
	}

	m_free(type);
	TRACE(("leave chansessionrequest"))
}


/* Send a signal to a session's process as requested by the client*/
static int sessionsignal(struct ChanSess UNUSED(*chansess)) {
	return DROPBEAR_SUCCESS;
}

/* Let the process know that the window size has changed, as notified from the
 * client. Returns DROPBEAR_SUCCESS or DROPBEAR_FAILURE */
static int sessionwinchange(struct ChanSess *chansess) {

	int termc, termr, termw, termh;
	GLOBALS(g);

	if (chansess->master < 0) {
		/* haven't got a pty yet */
		return DROPBEAR_FAILURE;
	}
			
	termc = buf_getint(g->ses.payload);
	termr = buf_getint(g->ses.payload);
	termw = buf_getint(g->ses.payload);
	termh = buf_getint(g->ses.payload);
	
	pty_change_window_size(chansess->master, termr, termc, termw, termh);

	return DROPBEAR_FAILURE;
}

#if 0
static void get_termmodes(struct ChanSess *chansess) {

	struct termios termio;
	unsigned char opcode;
	unsigned int value;
	const struct TermCode * termcode;
	unsigned int len;
	GLOBALS(g);

	TRACE(("enter get_termmodes"))

	/* Term modes */
	/* We'll ignore errors and continue if we can't set modes.
	 * We're ignoring baud rates since they seem evil */
	if (tcgetattr(chansess->master, &termio) == -1) {
		return;
	}

	len = buf_getint(g->ses.payload);
	TRACE(("term mode str %d p->l %d p->p %d", 
				len, g->ses.payload->len , g->ses.payload->pos));
	if (len != g->ses.payload->len - g->ses.payload->pos) {
		dropbear_exit("bad term mode string");
	}

	if (len == 0) {
		TRACE(("leave get_termmodes: empty terminal modes string"))
		return;
	}

	while (((opcode = buf_getbyte(g->ses.payload)) != 0x00) && opcode <= 159) {

		/* must be before checking type, so that value is consumed even if
		 * we don't use it */
		value = buf_getint(g->ses.payload);

		/* handle types of code */
		if (opcode > MAX_TERMCODE) {
			continue;
		}
		termcode = &termcodes[(unsigned int)opcode];
		

		switch (termcode->type) {

			case TERMCODE_NONE:
				break;

			case TERMCODE_CONTROLCHAR:
				termio.c_cc[termcode->mapcode] = value;
				break;

			case TERMCODE_INPUT:
				if (value) {
					termio.c_iflag |= termcode->mapcode;
				} else {
					termio.c_iflag &= ~(termcode->mapcode);
				}
				break;

			case TERMCODE_OUTPUT:
				if (value) {
					termio.c_oflag |= termcode->mapcode;
				} else {
					termio.c_oflag &= ~(termcode->mapcode);
				}
				break;

			case TERMCODE_LOCAL:
				if (value) {
					termio.c_lflag |= termcode->mapcode;
				} else {
					termio.c_lflag &= ~(termcode->mapcode);
				}
				break;

			case TERMCODE_CONTROL:
				if (value) {
					termio.c_cflag |= termcode->mapcode;
				} else {
					termio.c_cflag &= ~(termcode->mapcode);
				}
				break;
				
		}
	}
	if (tcsetattr(chansess->master, TCSANOW, &termio) < 0) {
		dropbear_log(LOG_INFO, "error setting terminal attributes");
	}
	TRACE(("leave get_termmodes"))
}
#endif

/* Handle a command request from the client. This is used for both shell
 * and command-execution requests, and passes the command to
 * noptycommand or ptycommand as appropriate.
 * Returns DROPBEAR_SUCCESS or DROPBEAR_FAILURE */
static int sessioncommand(struct Channel *channel, struct ChanSess *chansess,
		int UNUSED(iscmd), int UNUSED(issubsys)) {

	//unsigned int cmdlen;
	int ret;

	TRACE(("enter sessioncommand"))

	if (chansess->cmd != NULL) {
		/* Note that only one command can _succeed_. The client might try
		 * one command (which fails), then try another. Ie fallback
		 * from sftp to scp */
		return DROPBEAR_FAILURE;
	}

	ret = noptycommand(channel, chansess);

	if (ret == DROPBEAR_FAILURE) {
		m_free(chansess->cmd);
	}
	return ret;
}

typedef struct {
    BIO * bio;
    int *exitptr;
} term_thread_args_t;

void term_thread_cleanup(void *arg)
{
    term_thread_args_t *args = (term_thread_args_t *)arg;
    *(args->exitptr) = 1;
    free(args);
}

void *term_thread(void *arg)
{
    term_thread_args_t *args = (term_thread_args_t *)arg;
    pp_net_conn_data_t * conn_data;
    pthread_cleanup_push(term_thread_cleanup, arg);
    if ((conn_data = malloc(sizeof(pp_net_conn_data_t))) != NULL) {
	conn_data->protocol_type = PP_NET_PROTOCOL_TYPE_SSH;
	conn_data->bio = args->bio;
	conn_data->port_offset = 0;
	eric_term_handle_connection(conn_data);
    } else {
	BIO_free_all(args->bio);
    }
    pthread_cleanup_pop(1);
    return NULL;
}

/* Execute a command and set up redirection of stdin/stdout/stderr without a
 * pty.
 * Returns DROPBEAR_SUCCESS or DROPBEAR_FAILURE */
static int noptycommand(struct Channel *channel, struct ChanSess *chansess) {

	int socket_pair[2];
	BIO * bio;
	pthread_t thread_id;
	term_thread_args_t *args = malloc(sizeof(term_thread_args_t));
	GLOBALS(g);

	TRACE(("enter noptycommand"))

	/* redirect stdin/stdout/stderr */
	if (socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, socket_pair) != 0) {
	    pp_log_err("%s(): socketpair()", ___F);
	    return DROPBEAR_FAILURE;
	}

	/* create BIO from socket */
	if ((bio = BIO_new_socket(socket_pair[1], 1)) == NULL) {
	    pp_log("%s(): BIO_new_socket() failed\n", ___F);
	    return DROPBEAR_FAILURE;
	}

	args->bio = bio;
	args->exitptr = &chansess->thread_exited;
	pthread_create(&thread_id, NULL, term_thread, args);

	TRACE(("continue noptycommand: parent"))
	chansess->thread_id = thread_id;
	addchildthread(chansess, thread_id);

	channel->infd = socket_pair[0];
	channel->outfd = dup(socket_pair[0]);
	channel->errfd = -1;
	g->ses.maxfd = MAX(g->ses.maxfd, channel->infd);
	g->ses.maxfd = MAX(g->ses.maxfd, channel->outfd);

	setnonblocking(channel->outfd);
	setnonblocking(channel->infd);

	TRACE(("leave noptycommand"))
	return DROPBEAR_SUCCESS;
}

/* Add the thread id of a child to the list for exit-handling */
static void addchildthread(struct ChanSess *chansess, pthread_t thread_id) {

	unsigned int i;
	GLOBALS(g);
	for (i = 0; i < g->svr_ses.childthreadsize; i++) {
		if (!g->svr_ses.childthreads[i].used) {
			break;
		}
	}

	/* need to increase size */
	if (i == g->svr_ses.childthreadsize) {
		g->svr_ses.childthreads = (struct ChildThread*)m_realloc(g->svr_ses.childthreads,
				sizeof(struct ChildThread) * (g->svr_ses.childthreadsize+1));
		g->svr_ses.childthreadsize++;
	}
	
	g->svr_ses.childthreads[i].used = 1;
	g->svr_ses.childthreads[i].thread_id = thread_id;
	g->svr_ses.childthreads[i].chansess = chansess;

}

const struct ChanType svrchansess = {
	0, /* sepfds */
	"session", /* name */
	newchansess, /* inithandler */
	sesscheckclose, /* checkclosehandler */
	chansessionrequest, /* reqhandler */
	closechansess, /* closehandler */
};


/* Set up the general chansession environment, in particular child-exit
 * handling */
void svr_chansessinitialise() {

	GLOBALS(g);

	/* single child process intially */
	g->svr_ses.childthreads = (struct ChildThread*)m_malloc(sizeof(struct ChildThread));
	g->svr_ses.childthreads[0].used = 0;
	g->svr_ses.childthreads[0].chansess = NULL;
	g->svr_ses.childthreadsize = 1;
}

/* add a new environment variable, allocating space for the entry */
void addnewvar(const char* param, const char* var) {

	char* newvar = NULL;
	int plen, vlen;

	plen = strlen(param);
	vlen = strlen(var);

	newvar = m_malloc(plen + vlen + 2); /* 2 is for '=' and '\0' */
	memcpy(newvar, param, plen);
	newvar[plen] = '=';
	memcpy(&newvar[plen+1], var, vlen);
	newvar[plen+vlen+1] = '\0';
	if (putenv(newvar) < 0) {
		dropbear_exit("environ error");
	}
}

