package nn.pp.rc;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.zip.*;

/*
 * The standard renderer is a 8 bpp renderer
 * 
 * @author Thomas Breitfeld, Peppercon AG
 */
public abstract class ByteColorRFBRenderer
    extends RFBRenderer {

    private Color[]            colors;
    private int[]              intcolors;
    private int[]              tightLUT1BitFullColorBlackWhite;
    private int[]              tightLUT2BitFullColorGrayscale;
    private int[]              tightLUT4BitFullColorGrayscale;
    private int[]              tightLUT4BitFullColor16Colors;
    private int[]              tightLUT8BitFullColor256Colors;
    
    final static int           sh = 16;      // size of sprite
    final static int           sw = 16;
    private Image              simg;
    private MemoryImageSource  simgsrc;      
    private int[]              simgmem;
    private byte[]             simgmemb;
    
    final static int           tightZlibBufferSize = 512;
    Inflater[]                 tightInflaters;
    Image                      timg = null;   // tight decompression img
    MemoryImageSource          timgsrc;
    int[]                      timgmemi = null;
    byte[]                     timgtempb = null;
    byte[]                     timgtempbcache = null;
    
    private byte[]             rimgmemb;
    private byte[]             rimgtempb;

    public TightCacheTile[] tightCacheTiles;

    abstract public void setInterpol(boolean ip);
    abstract public void paint(Graphics g, boolean scale, Dimension scalesize);
    abstract public void repaint();
    abstract public void sendUpdateMsg() throws IOException;
    
    public ByteColorRFBRenderer(Component comp) {
	super(comp);
	DirectColorModel colormodel = new DirectColorModel(8, 7, (7 << 3), (3 << 6));
	// preallocate our 8 bit colors
        colors = new Color[256];
	intcolors = new int[256];
        for (int i = 0; i < 256; i++) {
	    int c = colormodel.getRGB(i);
            colors[i] = new Color(c);
	    intcolors[i] = c;
        }
	simgmem = new int[sh * sw];
	simgmemb = new byte[sh * sw];
	simgsrc = new MemoryImageSource(sw, sh, simgmem, 0, sw);
	simgsrc.setAnimated(true);
	simgsrc.setFullBufferUpdates(false);
	simg = comp.createImage(simgsrc);
	
	initTightLUTXBitFullColor();
    }
    
    public void sendPixelMsg() throws IOException {
	    rfb.writeSetPixelFormat(8, 8, false, true, 7, 7, 3, 0, 3, 6);
    }

    public void setRenderSize(Dimension d) {
	super.setRenderSize(d);
	mysize = d;
	//if(rfb.profile.encodings[1] == RFBproto.EncodingTight)
	createTImg(d);
    }
    
    void checkRawBufSize(int newSize) {
    	if(rimgmemb == null || rimgmemb.length < newSize) {
    	    rimgmemb = new byte[newSize];
    	}
    }

    void checkRawTmpBufSize(int newSize) {
    	if(rimgtempb == null || rimgtempb.length < newSize) {
    	    rimgtempb = new byte[newSize];
    	}
    }

    final public void drawRawRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int c = w * h;
	checkRawBufSize(c);
	rfb.is_rect.readFully(rimgmemb, 0, c);
	colorDecode(timgmemi, rimgmemb, y * mysize.width + x, w, h, mysize.width);
	timgsrc.newPixels(x, y, w, h);
	gvimg.setClip(x, y, w, h);
	gvimg.drawImage(timg, 0, 0, null);
	gvimg.setClip(0, 0, mysize.width, mysize.height);
    }

    final public void drawRawVSCRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int TILE_WIDTH = 16;
	int TILE_HEIGHT = 16;
	
	int flags = rfb.is_rect.readByte();
	
	if((flags & RFBproto.RawVscIsVsc) == 0) {
	    drawRawRect(x, y, w, h, bigEndian);
	    return;
	}
	
	// read the pixel data from the network
	int c = w * h;
	checkRawTmpBufSize(c);
	rfb.is_rect.readFully(rimgtempb, 0, c);
	
	// convert from VSC to linear framebuffer
	w = Math.min(w, mysize.width - x);
	h = Math.min(h, mysize.height - y);
	c = w * h;
	checkRawBufSize(c);
	
	int indexsrc = 0;
	int indexdst = 0;
	int indexsrcstart = 0;
	int lines = 0;
	int rows = 0;
	
	for(int i = 0; i < h; i++) {
	    int cols = 0;
	    for(int j = 0; j < w; j++) {
	    	rimgmemb[indexdst] = rimgtempb[indexsrc];
	    	indexdst++;
	    	indexsrc++;
	    	cols++;
	    	if(cols == TILE_WIDTH) {
	    	    indexsrc += (TILE_HEIGHT - 1) * TILE_WIDTH;
	    	    cols = 0;
	    	}
	    }
	    lines++;
	    indexsrc = indexsrcstart + lines * TILE_WIDTH;
	    if(lines == TILE_HEIGHT) {
	    	rows++;
	    	indexsrcstart = rows * TILE_HEIGHT * w;
	    	indexsrc = indexsrcstart;
	    	lines = 0;
	    }
	}
	
    	// actually draw the data
	colorDecode(timgmemi, rimgmemb, y * mysize.width + x, w, h, mysize.width);
	timgsrc.newPixels(x, y, w, h);
	gvimg.setClip(x, y, w, h);
	gvimg.drawImage(timg, 0, 0, null);
	gvimg.setClip(0, 0, mysize.width, mysize.height);
    }

    final public void drawHextileRawRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int c = w * h;
	
        // write back the buffered data before reading unbuffered from rfb.is_rect
        finishBufferedReading(rfb.is_rect);
        
	rfb.is_rect.readFully(simgmemb, 0, c);
	colorDecode(simgmem, simgmemb, 0, w, h, sw);
	simgsrc.newPixels(0, 0, w, h);
	if(h < sh || w < sw) {
	    gvimg.setClip(x, y, w, h);
	    gvimg.drawImage(simg, x, y, x+w, y+h, 0, 0, w, h, null);
	    gvimg.setClip(0, 0, mysize.width, mysize.height);
	} else {
	    gvimg.drawImage(simg, x, y, null);
	}
    }

    final public void drawCopyRect(int srcX, int srcY, int x, int y, int w, int h)
	throws IOException {
	//System.out.println("CopyRect scrX = " + srcX + ", srcY = " + srcY + ", x = " + x + ", y = " + y + ", w = " + w + ", h = " + h);
	gvimg.copyArea(srcX, srcY, w, h, x - srcX, y - srcY);
    }

    final public void drawHextileRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int bg = 0, fg = 0, ux, uy, uw, uh;
	int tx, ty, tw, th;
	int subencoding, nSubrects, b1, b2, j, c;
	int counter = 0;
	RFBproto rfb = this.rfb;
	Graphics gvimg = this.gvimg;
	for (ty = y; ty < y + h; ty += sh) {
	    for (tx = x; tx < x + w; tx += sw) {
		tw = sw;
		th = sh;
		if (x + w - tx < 16) tw = x + w - tx;
		if (y + h - ty < 16) th = y + h - ty;
		subencoding = readBufferedByte(rfb.is_rect);
		counter++;

		if ((subencoding & rfb.HextileRaw) != 0) {
		    drawHextileRawRect(tx, ty, tw, th, bigEndian);
		    continue;
		}

		if ((subencoding & rfb.HextileBackgroundSpecified) != 0){
		    bg = readBufferedByte(rfb.is_rect);
		    counter++;
		}

		gvimg.setColor(colors[bg]);
		gvimg.fillRect(tx, ty, tw, th);

		if ((subencoding & rfb.HextileForegroundSpecified) != 0) {
		    fg = readBufferedByte(rfb.is_rect);
		    counter++;
		}
		
		if ((subencoding & rfb.HextileAnySubrects) == 0)
		    continue;

		nSubrects = readBufferedByte(rfb.is_rect);
		counter++;
		gvimg.translate(tx, ty);
		
		if ((subencoding & rfb.HextileSubrectsColoured) != 0) {
		    for (j = 0; j < nSubrects; j++) {
			fg = readBufferedByte(rfb.is_rect);
			b1 = readBufferedByte(rfb.is_rect);
			b2 = readBufferedByte(rfb.is_rect);
			ux = b1 >> 4;
			uy = b1 & 0xf;
			uw = (b2 >> 4) + 1;
			uh = (b2 & 0xf) + 1;
			gvimg.setColor(colors[fg]);
			gvimg.fillRect(ux, uy, uw, uh);
		    }
		    counter+=(nSubrects * 3);
		} else {
		    gvimg.setColor(colors[fg]);
		    for (j = 0; j < nSubrects; j++) {
			b1 = readBufferedByte(rfb.is_rect);
			b2 = readBufferedByte(rfb.is_rect);
			ux = b1 >> 4;
			uy = b1 & 0xf;
			uw = (b2 >> 4) + 1;
			uh = (b2 & 0xf) + 1;
			gvimg.fillRect(ux, uy, uw, uh);
		    }
		    counter+=(nSubrects * 2);
		}
		gvimg.translate(-tx, -ty);
	    }
	}
	
	finishBufferedReading(rfb.is_rect);

	// increase the monitoring counters manually
	// (see also MonitoringDataInputStream)
	rfb.min.increaseCounters(counter);
    }

    final public void drawTightRect(int x, int y, int w, int h)
	throws IOException {
	drawTightRect0(x, y, w, h, null, 0, 0, 0);
    }

    final public void drawTightCachedRect(int x, int y, int w, int h)
	throws IOException {
	int subencoding = rfb.is_rect.read();
	int stream_id = (subencoding >> 4) & rfb.TightStreamId;
	int cache_tag = (subencoding >> 4) & rfb.TightCacheTag;
	int tightXBitFullColor = subencoding & 0xF;
	int tile_dim = rfb.TightTileDimension;
	int mc_t_size;
	byte [] memcpy_tag;
	int bitsPixel, pix_red_fac;
	int y_s, h_e;
	int comp_ctl;

	switch (tightXBitFullColor) {
	    case RFBproto.Tight1BitBlackWhite:
	        pix_red_fac = 8;
	        comp_ctl = RFBproto.Tight1BitFullColorBlackWhite;
	        break;
    	    case RFBproto.Tight2BitGrayscale:
    	        pix_red_fac = 4;
    	        comp_ctl = RFBproto.Tight2BitFullColorGrayscale;
    	        break;
    	    case RFBproto.Tight4BitGrayscale:
    	        pix_red_fac = 2;
    	        comp_ctl = RFBproto.Tight4BitFullColorGrayscale;
    	        break;
    	    case RFBproto.Tight4Bit16Colors:
    	        pix_red_fac = 2;
    	        comp_ctl = RFBproto.Tight4BitFullColor16Colors;
    	        break;
    	    case RFBproto.Tight8Bit256Colors:
    	        pix_red_fac = 1;
    	        comp_ctl = 0;
    	        break;
    	    default:
    	        return;			/* Should never happen. */
	}

	if ((y % tile_dim) != 0) {
	    y_s = ((y / tile_dim + 1) * tile_dim) - y;
	} else {
	    y_s = 0;
	}
	if (((y+h) % tile_dim) != 0) {
	    h_e = ((y+h) / tile_dim) * tile_dim - y;
	} else {
	    h_e = h;
	}
	mc_t_size = (w/tile_dim) * (h_e/tile_dim);
	memcpy_tag = new byte[mc_t_size];

	if (mc_t_size < rfb.TightMinToCompress) {
	    rfb.is_rect.readFully(memcpy_tag, 0, mc_t_size);
        } else {
            int zlibDataLen = rfb.readCompactLen();
            byte[] zlibData = new byte[zlibDataLen];
            rfb.is_rect.readFully(zlibData, 0, zlibDataLen);
            if (tightInflaters[stream_id] == null) {
                tightInflaters[stream_id] = new Inflater();
            }
            Inflater myInflater = tightInflaters[stream_id];
            myInflater.setInput(zlibData, 0, zlibDataLen);
            try {
		myInflater.inflate(memcpy_tag, 0, mc_t_size);
            }
            catch(DataFormatException dfe) {
                throw new IOException(dfe.toString());
            }
        }

	if (cache_tag == rfb.TightCacheAll) { // all tiles are cached
	    readCacheMetaRect(x, y, w, h, memcpy_tag, cache_tag, tile_dim, pix_red_fac);
	    drawXBitFullColor(x, y, w, h, comp_ctl);
	    timgsrc.newPixels(x, y, w, h);
	    gvimg.setClip(x, y, w, h);
	    gvimg.drawImage(timg, 0, 0, null);
	    gvimg.setClip(0, 0, mysize.width, mysize.height);
	} else {
	    drawTightRect0(x, y, w, h, memcpy_tag, cache_tag, tile_dim, pix_red_fac);
	}
    }

    final public void drawTightRect0(int x, int y, int w, int h,
                                     byte [] memcpy_tag, int cache_tag, 
                                     int tile_dim, int pix_red_fac)
	throws IOException {
        RFBproto rfb = this.rfb;
        int comp_ctl = rfb.is_rect.read();
        int counter = 1;
        // Flush zlib streams if we are told by the server to do so.
        for (int stream_id = 0; stream_id < 4; stream_id++) {
            if ((comp_ctl & 1) != 0 &&
                tightInflaters[stream_id] != null) {
                tightInflaters[stream_id] = null;
            }
            comp_ctl >>= 1;
        }
        // Check correctness of subencoding value.
        if (comp_ctl > rfb.TightMaxSubencoding) {
            throw new IOException(T._("Incorrect tight subencoding:") +
                                  " " + comp_ctl);
        }
        // Handle solid rectangles.
        if ((comp_ctl == rfb.TightFill) || (comp_ctl == rfb.TightFillXBit)) {
            if (comp_ctl == rfb.TightFill) {
                gvimg.setColor(colors[rfb.is_rect.read()]); 
                counter++;
            }
            else { // TightFillXBit
                int bg = 0;
                switch (rfb.is_rect.read()) {
                    case RFBproto.Tight1BitBlackWhite:
                        bg = tightLUT1BitFullColorBlackWhite[rfb.is_rect.read()];
                        break;
                    case RFBproto.Tight2BitGrayscale:
                        bg = tightLUT2BitFullColorGrayscale[rfb.is_rect.read()];
                        break;
                    case RFBproto.Tight4BitGrayscale:
                        bg = tightLUT4BitFullColorGrayscale[rfb.is_rect.read()];
                        break;
                    case RFBproto.Tight4Bit16Colors:
                        bg = tightLUT4BitFullColor16Colors[rfb.is_rect.read()];
                        break;
                }
                gvimg.setColor(new Color(bg));
                counter+=2;
            }
            gvimg.fillRect(x, y, w, h);
	    // increase the monitoring counters manually
	    // (see also MonitoringDataInputStream)
            rfb.min.increaseCounters(counter);
            return;
        }
        // Read filter id and parameters.
        int numColors = 0, rowSize = w;
        int paletteXBit[] = new int[2];
        if ((comp_ctl | rfb.TightStreamId) == rfb.TightExplicitFilter) {
            int b = rfb.is_rect.read();
            counter++;
            int filter_id = b & 0xF;
            int useXBit = (b >> 4) & 0xF;
            if (filter_id == rfb.TightFilterPalette) {
                numColors = rfb.is_rect.read() + 1; // Must be 2.
                counter++;
                if (numColors != 2) {
                    throw new IOException(T._("Incorrect tight palette size:") +
                                          " " + numColors);
                }
                switch (useXBit) {
		  case RFBproto.Tight1BitBlackWhite:
		      b = rfb.is_rect.read();
		      counter++;
		      paletteXBit[0] = tightLUT1BitFullColorBlackWhite[b >> 1];
		      paletteXBit[1] = tightLUT1BitFullColorBlackWhite[b & 0x1];
		      break;
		  case RFBproto.Tight2BitGrayscale:
		      b = rfb.is_rect.read();
		      counter++;
		      paletteXBit[0] = tightLUT2BitFullColorGrayscale[b >> 2];
		      paletteXBit[1] = tightLUT2BitFullColorGrayscale[b & 0x3];
		      break;
		  case RFBproto.Tight4BitGrayscale:
		      b = rfb.is_rect.read();
		      counter++;
		      paletteXBit[0] = tightLUT4BitFullColorGrayscale[b >> 4];
		      paletteXBit[1] = tightLUT4BitFullColorGrayscale[b & 0xF];
		      break;
		  case RFBproto.Tight4Bit16Colors:
		      b = rfb.is_rect.read();
		      counter++;
		      paletteXBit[0] = tightLUT4BitFullColor16Colors[b >> 4];
		      paletteXBit[1] = tightLUT4BitFullColor16Colors[b & 0xF];
		      break;
		  default:
		      paletteXBit[0] = tightLUT8BitFullColor256Colors
			  [rfb.is_rect.read()];
		      paletteXBit[1] = tightLUT8BitFullColor256Colors
			  [rfb.is_rect.read()];
		      counter+=2;
                }
                rowSize = (w + 7) / 8;
            } else if (filter_id != rfb.TightFilterCopy) {
                throw new IOException(T._("Incorrect tight filter id:") +
                                      " " + filter_id);
            }
        }
        else {
            switch (comp_ctl) {
                case RFBproto.Tight1BitFullColorBlackWhite:
                    rowSize = (w + 7) / 8;
                    break;
                case RFBproto.Tight2BitFullColorGrayscale:
                    rowSize = (w + 3) / 4;
                    break;
                case RFBproto.Tight4BitFullColorGrayscale:
                case RFBproto.Tight4BitFullColor16Colors:
                    rowSize = (w + 1) / 2;
                    break;
            }
        }
        // Read, optionally uncompress and decode data.
        int dataSize = h * rowSize;
        if (dataSize < rfb.TightMinToCompress) {
            if (numColors == 2) {
                byte[] monoData = new byte[dataSize];
                rfb.is_rect.readFully(monoData, 0, dataSize);
                drawMonoData(x, y, w, h, monoData, paletteXBit);
            } else {
		rfb.is_rect.readFully(timgtempb, 0, dataSize);
		drawXBitFullColor(x, y, w, h, comp_ctl);
            }
        } else {
            int zlibDataLen = rfb.readCompactLen();
            byte[] zlibData = new byte[zlibDataLen];
            rfb.is_rect.readFully(zlibData, 0, zlibDataLen);
            int stream_id;
            if ((comp_ctl & rfb.TightCheckForXBitFullColor) != 0) stream_id = 0;
            else stream_id = comp_ctl & rfb.TightStreamId;
            if (tightInflaters[stream_id] == null) {
                tightInflaters[stream_id] = new Inflater();
            }
            Inflater myInflater = tightInflaters[stream_id];
            myInflater.setInput(zlibData, 0, zlibDataLen);
            try {
                if (numColors == 2) {
                    byte[] monoData = new byte[dataSize];
                    myInflater.inflate(monoData, 0, dataSize);
                    drawMonoData(x, y, w, h, monoData, paletteXBit);
                } else {
		    myInflater.inflate(timgtempb, 0, dataSize);
	    	    if (memcpy_tag != null) {
			readCacheMetaRect(x, y, w, h, memcpy_tag, cache_tag, tile_dim, pix_red_fac);
	    	    }
		    drawXBitFullColor(x, y, w, h, comp_ctl);
                }
            }
            catch(DataFormatException dfe) {
                throw new IOException(dfe.toString());
            }

        }
	timgsrc.newPixels(x, y, w, h);
	gvimg.setClip(x, y, w, h);
	gvimg.drawImage(timg, 0, 0, null);
	gvimg.setClip(0, 0, mysize.width, mysize.height);
	// increase the monitoring counters manually
	// (see also MonitoringDataInputStream)
	rfb.min.increaseCounters(counter);
    }

    public void readCacheMetaRect(int x, int y, int w, int h, 
                                  byte [] memcpy_tag, int cache_tag, 
                                  int tile_dim, int pix_red_fac) {
	int i, j, k;
	int mc_t;
	int mc_pos = 0;
	int client_index;
	int y_s, h_e;
	byte src[], dst[];
	
	if ((y % tile_dim) != 0) {
	    y_s = ((y / tile_dim + 1) * tile_dim) - y;
	} else {
	    y_s = 0;
	}
	if (((y+h) % tile_dim) != 0) {
	    h_e = ((y+h) / tile_dim) * tile_dim - y;
	} else {
	    h_e = h;
	}

	if (cache_tag == rfb.TightCacheMix) { // some tiles are cached, some not
	    src = timgtempb;
	    dst = timgtempbcache;
	    if (y_s != 0) {
		System.arraycopy(src, 0, dst, 0, (y_s*w)/pix_red_fac);
		mc_pos += (y_s*w)/pix_red_fac;
	    }
	    for (j = y_s; j < h_e; j+=tile_dim) {
		for (k = 0; k < tile_dim; k++) {
		    for (i = 0; i < w; i+=tile_dim) {
			mc_t = memcpy_tag[((j/tile_dim)*(w/tile_dim)) + (i/tile_dim)];
			client_index = mc_t & rfb.TightCacheClientIndex;
			if ((mc_t & rfb.TightTileCached) == 0) {
			    tightCacheTiles[(((y+j)/tile_dim)*(mysize.width/tile_dim)) + ((x+i)/tile_dim)].saveDataStream(client_index, k * (tile_dim/pix_red_fac), tile_dim/pix_red_fac, src, mc_pos);
			    mc_pos+=tile_dim/pix_red_fac;
			}
			tightCacheTiles[(((y+j)/tile_dim)*(mysize.width/tile_dim)) + ((x+i)/tile_dim)].restoreDataStream(client_index, k * (tile_dim/pix_red_fac), tile_dim/pix_red_fac, dst, (((j+k)*w) + i)/pix_red_fac);
		    }
		}
	    }
	    if (h_e!=h) {
		System.arraycopy(src, mc_pos, dst, (j*w)/pix_red_fac, ((h-h_e)*w)/pix_red_fac);
		mc_pos += ((h-h_e)*w)/pix_red_fac;
	    }
	    System.arraycopy(dst, 0, src, 0, w*h/pix_red_fac);
	} 
	else if  (cache_tag == rfb.TightCacheNone) { // no tile is cached
	    src = timgtempb;
	    for (j = y_s; j < h_e; j+=tile_dim) {
		for (i = 0; i < w; i+=tile_dim) {
		    client_index = memcpy_tag[((j/tile_dim)*(w/tile_dim)) + (i/tile_dim)] & rfb.TightCacheClientIndex;
		    for (k = 0; k < tile_dim; k++) {
			tightCacheTiles[(((y+j)/tile_dim)*(mysize.width/tile_dim)) + ((x+i)/tile_dim)].saveDataStream(client_index, k * (tile_dim/pix_red_fac), tile_dim/pix_red_fac, src, (((j+k)*w) + i)/pix_red_fac);
		    }
		}
	    }
	}
	else { // rfb.TightCacheAll: all tiles are cached
	    dst = timgtempb;
	    for (j = 0; j < h; j+=tile_dim) {
		for (i = 0; i < w; i+=tile_dim) {
		    client_index = memcpy_tag[((j/tile_dim)*(w/tile_dim)) + (i/tile_dim)] & rfb.TightCacheClientIndex;
		    for (k = 0; k < tile_dim; k++) {
			tightCacheTiles[(((y+j)/tile_dim)*(mysize.width/tile_dim)) + ((x+i)/tile_dim)].restoreDataStream(client_index, k * (tile_dim/pix_red_fac), tile_dim/pix_red_fac, dst, (((j+k)*w) + i)/pix_red_fac);
		    }
		}
	    }
	}
    }

    public void dispose() {
	if(timg != null) timg.flush();
	simg.flush();
	if(vimg != null) vimg.flush();
	if(gvimg != null) gvimg.dispose();
    }
	
    final private void drawMonoData(int x, int y, int w, int h, byte[] src,
				    int[] paletteXBit)
	throws IOException {
	int dx, dy, n, b;
	int[] dst = timgmemi;
	int i = y * rfb.framebufferWidth + x, index2 = 0;
	int rowBytes = (w + 7) / 8;

	for (dy = 0; dy < h; dy++) {
	    index2 = dy * rowBytes;
	    for (dx = 0; dx < w / 8; dx++) {
	        b = src[index2 + dx] & 0xFF;
	        //System.out.println("" + (dy * rowBytes + dx) + " " + (index2++));
	        for (n = 7; n >= 0; n--) {
	            dst[i++] = paletteXBit[b >> n & 1];
	        }
	    }
	    for (n = 7; n >= 8 - w % 8; n--) {
	        b = src[index2 + dx] & 0xFF;
	        //System.out.println("" + (dy * rowBytes + dx) + " " + (index2++));
	        dst[i++] = paletteXBit[b >> n & 1];
	    }
	    i += (rfb.framebufferWidth - w);
	}
    }
    
    final private void drawXBitFullColor(int x, int y, int w, int h,
					 int comp_ctl)
	throws IOException {
	int fb_width = rfb.framebufferWidth;
	int index = y * fb_width + x;
	int index2 = 0;
	byte[] src = timgtempb;
	int[] dst = timgmemi;
	int[] lut;
	int i, j, s, wDiv, wMod;
	byte b;
	switch (comp_ctl) {
	    case RFBproto.Tight1BitFullColorBlackWhite:
		lut = tightLUT1BitFullColorBlackWhite;
		wDiv = w / 8;
		wMod = w % 8;
		for (j = 0; j < h; j++) {
		    for (i = 0; i < wDiv; i++) {
			b = src[index2++];
			for (s = 7; s >= 0; s--) {
			    dst[index++] = lut[(b >> s) & 0x1];
			}
		    }
		    if (wMod != 0) {
			b = src[index2++];
			for (s = wMod; s > 0; s--) {
			    dst[index++] = lut[(b >> s) & 0x1];
			}
		    }
		    index+=fb_width - w;
		}
		break;
	    case RFBproto.Tight2BitFullColorGrayscale:
		lut = tightLUT2BitFullColorGrayscale;
		wDiv = w / 4;
		wMod = w % 4;
		for (j = 0; j < h; j++) {
		    for (i = 0; i < wDiv; i++) {
			b = src[index2++];
			for (s = 6; s >= 0; s-=2) {
			    dst[index++] = lut[(b >> s) & 0x3];
			}
		    }
		    if (wMod != 0) {
			b = src[index2++];
			for (s = wMod; s > 0; s--) {
			    dst[index++] = lut[(b >> 2*s) & 0x3];
			}
		    }
		    index+=fb_width - w;
		}
		break;
	    case RFBproto.Tight4BitFullColorGrayscale:
	    case RFBproto.Tight4BitFullColor16Colors:
		if (comp_ctl == RFBproto.Tight4BitFullColor16Colors) {
		    lut = tightLUT4BitFullColor16Colors;
		}
		else {
		    lut = tightLUT4BitFullColorGrayscale;
		}
		wDiv = w / 2;
		wMod = w % 2;
		for (j = 0; j < h; j++) {
		    for (i = 0; i < wDiv; i++) {
			b = src[index2++];
			dst[index++] = lut[(b >> 4) & 0xF];
			dst[index++] = lut[b & 0xF];
		    }
		    if (wMod != 0) {
			dst[index++] = lut[(src[index2++] >> 4) & 0xF];
		    }
		    index+=fb_width - w;
		}
		break;
	    default:
		lut = tightLUT8BitFullColor256Colors;
		for (j = 0; j < h; j++) {
		    for (i = 0; i < w; i++) {
			dst[index++] = lut[src[index2++] & 0xFF];
		    }
		    index+=fb_width - w;
		}
	}
    }

    private void createTImg(Dimension d) {
	Dimension padsize = new Dimension(d);	
	/* we reserve more memory than visible for padding */
	if (padsize.width % sw != 0) padsize.width = (padsize.width / sw + 1) *  sw;
	if (padsize.height % sh != 0) padsize.height = (padsize.height / sh + 1) * sh;
	
	if(tightInflaters == null) {
	    tightInflaters = new Inflater[4];
	    timgtempb = new byte[RFBproto.TightMaxRectSize];
	    timgtempbcache = new byte[RFBproto.TightMaxRectSize];
	}
	if(timg != null) timg.flush();
	timgmemi = new int[padsize.width * padsize.height];
	timgsrc = new MemoryImageSource(mysize.width, mysize.height, timgmemi,
					0, mysize.width);
	jimgsrc = timgsrc;
	timgsrc.setAnimated(true);
	timgsrc.setFullBufferUpdates(false);
	timg = comp.createImage(timgsrc);
    }

    final private void colorDecode(int[] px32, byte[] px8, int off,
				   int w, int h, int scanline) {
        int index_src = 0;
        int index_dst = off;
        for(int i = 0; i < h; ++i) {
	    for(int j = 0; j < w; j++) {
	    	int px = intcolors[0xff & px8[index_src++]];
	    	px32[index_dst++] = px;
	    }
	    index_dst += scanline - w;
        }
    }

    final private void initTightLUTXBitFullColor() {
	// preallocate our LUT
        tightLUT1BitFullColorBlackWhite = new int[2];
        tightLUT2BitFullColorGrayscale = new int[4];
        tightLUT4BitFullColorGrayscale = new int[16];
        tightLUT4BitFullColor16Colors = new int[16];
        tightLUT8BitFullColor256Colors = intcolors;

	tightLUT1BitFullColorBlackWhite[0] = new Color(0, 0, 0).getRGB();
	tightLUT1BitFullColorBlackWhite[1] = new Color(255, 255, 255).getRGB();

	tightLUT2BitFullColorGrayscale[0] = new Color(0, 0, 0).getRGB();
	tightLUT2BitFullColorGrayscale[1] = new Color(128, 128, 128).getRGB();
	tightLUT2BitFullColorGrayscale[2] = new Color(192, 192, 192).getRGB();
	tightLUT2BitFullColorGrayscale[3] = new Color(255, 255, 255).getRGB();

	tightLUT4BitFullColorGrayscale[0] = new Color(0, 0, 0).getRGB();
	tightLUT4BitFullColorGrayscale[1] = new Color(33, 33, 33).getRGB();
	tightLUT4BitFullColorGrayscale[2] = new Color(50, 50, 50).getRGB();
	tightLUT4BitFullColorGrayscale[3] = new Color(67, 67, 67).getRGB();
	tightLUT4BitFullColorGrayscale[4] = new Color(92, 92, 92).getRGB();
	tightLUT4BitFullColorGrayscale[5] = new Color(105, 105, 105).getRGB();
	tightLUT4BitFullColorGrayscale[6] = new Color(117, 117, 117).getRGB();
	tightLUT4BitFullColorGrayscale[7] = new Color(134, 134, 134).getRGB();
	tightLUT4BitFullColorGrayscale[8] = new Color(151, 151, 151).getRGB();
	tightLUT4BitFullColorGrayscale[9] = new Color(163, 163, 163).getRGB();
	tightLUT4BitFullColorGrayscale[10] = new Color(178, 178, 178).getRGB();
	tightLUT4BitFullColorGrayscale[11] = new Color(193, 193, 193).getRGB();
	tightLUT4BitFullColorGrayscale[12] = new Color(209, 209, 209).getRGB();
	tightLUT4BitFullColorGrayscale[13] = new Color(226, 226, 226).getRGB();
	tightLUT4BitFullColorGrayscale[14] = new Color(79, 79, 79).getRGB();
	tightLUT4BitFullColorGrayscale[15] = new Color(255, 255, 255).getRGB();

	tightLUT4BitFullColor16Colors[0] = new Color(0, 0, 0).getRGB();
	tightLUT4BitFullColor16Colors[1] = new Color(128, 0, 0).getRGB();
	tightLUT4BitFullColor16Colors[2] = new Color(255, 0, 0).getRGB();
	tightLUT4BitFullColor16Colors[3] = new Color(0, 128, 0).getRGB();
	tightLUT4BitFullColor16Colors[4] = new Color(128, 128, 0).getRGB();
	tightLUT4BitFullColor16Colors[5] = new Color(255, 255, 0).getRGB();
	tightLUT4BitFullColor16Colors[6] = new Color(0, 255, 0).getRGB();
	tightLUT4BitFullColor16Colors[7] = new Color(0, 0, 128).getRGB();
	tightLUT4BitFullColor16Colors[8] = new Color(128, 0, 128).getRGB();
	tightLUT4BitFullColor16Colors[9] = new Color(0, 128, 128).getRGB();
	tightLUT4BitFullColor16Colors[10] = new Color(128, 128, 128).getRGB();
	tightLUT4BitFullColor16Colors[11] = new Color(192, 192, 192).getRGB();
	tightLUT4BitFullColor16Colors[12] = new Color(255, 0, 255).getRGB();
	tightLUT4BitFullColor16Colors[13] = new Color(0, 255, 255).getRGB();
	tightLUT4BitFullColor16Colors[14] = new Color(255, 255, 255).getRGB();
	tightLUT4BitFullColor16Colors[15] = new Color(0, 0, 255).getRGB();
    }

    public void setNewTightCacheSize() {
	if (handler.params.encoding.isTightCacheUsed()) {
	    if (tightCacheTiles != null) {
	    	deinitTightCache();
	    	System.gc();
	    }
	    initTightCache();
	}
    }

    public void enableTightCache(boolean enable) {
    	if (enable) {
    	    if (tightCacheTiles == null) {
    	    	initTightCache();
    	    }
    	}
    	else {
    	    if (tightCacheTiles != null) {
    	    	deinitTightCache();
    	    }
	}
    }

    public void initTightCache() {
	int tightCacheTilesSize = (rfb.framebufferWidthPadded/rfb.TightTileDimension) * (rfb.framebufferHeightPadded/rfb.TightTileDimension);
	tightCacheTiles = new TightCacheTile[tightCacheTilesSize];
	for (int i = 0; i < tightCacheTilesSize; i++) {
	    tightCacheTiles[i] = new TightCacheTile(rfb.TightCacheBufferDepth, rfb.TightTileDimension * rfb.TightTileDimension);
	}
    }

    public void deinitTightCache() {
	tightCacheTiles = null;
    }

    public void drawLRLERect(long encoding, int x, int y, int w, int h) throws IOException
    {
	throw new IOException(T._("Byte color LRLE encoding not implemented"));        
    }
}
