/*
 * @(#)Timer.java	1.42 01/12/03
 *
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
 
package nn.pp.rc;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.io.Serializable;
import javax.swing.event.EventListenerList;


/**
 * Fires one or more action events after a specified delay.  
 * For example, an animation object can use a <code>Timer</code>
 * as the trigger for drawing its frames.
 *
 *<p>
 *
 * Setting up a timer
 * involves creating a <code>Timer</code> object,
 * registering one or more action listeners on it,
 * and starting the timer using
 * the <code>start</code> method.
 * For example, 
 * the following code creates and starts a timer
 * that fires an action event once per second
 * (as specified by the first argument to the <code>Timer</code> constructor).
 * The second argument to the <code>Timer</code> constructor
 * specifies a listener to receive the timer's action events.
 *
 *<pre>
 *  int delay = 1000; //milliseconds
 *  ActionListener taskPerformer = new ActionListener() {
 *      public void actionPerformed(ActionEvent evt) {
 *          <em>//...Perform a task...</em>
 *      }
 *  };
 *  new Timer(delay, taskPerformer).start();</pre>
 *
 * <p>
 * Each <code>Timer</code>
 * has one or more action listeners
 * and a <em>delay</em>
 * (the time between action events).  
 * When
 * <em>delay</em> milliseconds have passed, the <code>Timer</code>
 * fires an action event to its listeners.  
 * By default, this cycle repeats until
 * the <code>stop</code> method is called.
 * If you want the timer to fire only once,
 * invoke <code>setRepeats(false)</code> on the timer.
 *
 * <p>
 * Although all <code>Timer</code>s perform their waiting
 * using a single, shared thread 
 * (created by the first <code>Timer</code> object that executes),
 * the action event handlers for <code>Timer</code>s
 * execute on another thread -- the event-dispatching thread.
 * This means that the action handlers for <code>Timer</code>s
 * can safely perform operations on Swing components.
 * However, it also means that the handlers must execute quickly
 * to keep the GUI responsive.
 *
 * <p>
 * In v 1.3, another <code>Timer</code> class was added
 * to the Java platform: <code>java.util.Timer</code>.
 * Both it and <code>javax.swing.Timer</code>
 * provide the same basic functionality,
 * but <code>java.util.Timer</code> 
 * is more general and has more features.
 * The <code>javax.swing.Timer</code> has two features
 * that can make it a little easier to use with GUIs.
 * First, its event handling metaphor is familiar to GUI programmers
 * and can make dealing with the event-dispatching thread
 * a bit simpler.
 * Second, its
 * automatic thread sharing means that you don't have to 
 * take special steps to avoid spawning
 * too many threads.
 * Instead, your timer uses the same thread
 * used to make cursors blink,
 * tool tips appear, 
 * and so on.
 *
 * <p>
 * You can find further documentation
 * and several examples of using timers by visiting
 * <a href="http://java.sun.com/docs/books/tutorial/uiswing/misc/timer.html"
 * target = "_top">How to Use Timers</a>,
 * a section in <em>The Java Tutorial.</em>
 * For more examples and help in choosing between
 * this <code>Timer</code> class and 
 * <code>java.util.Timer</code>, 
 * see 
 * <a href="http://java.sun.com/products/jfc/tsc/articles/timer/"
 * target="_top">Using Timers in Swing Applications</a>,
 * an article in <em>The Swing Connection.</em>
 * <p>
 * <strong>Warning:</strong>
 * Serialized objects of this class will not be compatible with
 * future Swing releases. The current serialization support is
 * appropriate for short term storage or RMI between applications running
 * the same version of Swing.  As of 1.4, support for long term storage
 * of all JavaBeans<sup><font size="-2">TM</font></sup>
 * has been added to the <code>java.beans</code> package.
 * Please see {@link java.beans.XMLEncoder}.
 *
 * @see java.util.Timer <code>java.util.Timer</code>
 *
 *
 * @version 1.42 12/03/01
 * @author Dave Moore
 */
public class GenericTimer implements Serializable
{
    protected ActionListener actionListener;		// action listener

    int     delay;
    boolean repeats = true;

    // These fields are maintained by TimerQueue.
    // eventQueued can also be reset by the TimerQueue, but will only ever
    // happen in applet case when TimerQueues thread is destroyed.
    long    expirationTime;
    GenericTimer   nextTimer;
    boolean running;
    boolean notify;


    /**
     * Creates a <code>Timer</code> that will notify its listeners every
     * <code>delay</code> milliseconds. If <code>delay</code> is less than
     * or equal to zero the timer will fire as soon as it
     * is started. If <code>listener</code> is not <code>null</code>,
     * it's registered as an action listener on the timer.
     *
     * @param delay the number of milliseconds between action events
     * @param listener an initial listener; can be <code>null</code>
     *
     * @see #addActionListener
     * @see #setRepeats
     */
    public GenericTimer(int delay, ActionListener listener) {
        super();
        this.delay = delay;

	    actionListener = listener;
    }
    
    /**
     * Returns the timer queue.
     */
    TimerQueue timerQueue() {
        return TimerQueue.sharedInstance();
    }

    /**
     * Sets the <code>Timer</code>'s delay, the number of milliseconds
     * between successive action events.
     *
     * @param delay the delay in milliseconds
     */
    public void setDelay(int delay) {
        if (delay < 0) {
            throw new IllegalArgumentException(T._("Invalid delay:") + " " + delay);
        }
        else {
            this.delay = delay;
        }
    }
    
    public int getDelay() {
    	return this.delay;
   	}

    /**
     * If <code>flag</code> is <code>false</code>,
     * instructs the <code>Timer</code> to send only one
     * action event to its listeners.
     *
     * @param flag specify <code>false</code> to make the timer
     *             stop after sending its first action event
     */
    public void setRepeats(boolean flag) {
        repeats = flag;
    }

    /**
     * Returns <code>true</code> (the default)
     * if the <code>Timer</code> will send
     * an action event 
     * to its listeners multiple times.
     *
     * @see #setRepeats
     */
    public boolean isRepeats() {
        return repeats;
    }

    /**
     * Returns <code>true</code> if the <code>Timer</code> is running.
     */
    public boolean isRunning() {
        return timerQueue().containsTimer(this);
    }

	/**
     * Starts the <code>Timer</code>,
     * causing it to start sending action events
     * to its listeners
     */
    public void start() {
    	notify = true;
        timerQueue().addTimer(this,
                              System.currentTimeMillis() + getDelay());
    }

    /**
     * Stops the <code>Timer</code>,
     * causing it to stop sending action events
     * to its listeners.
     */
    public void stop() {
        timerQueue().removeTimer(this);
        repeats = false;
        notify = false;
    }
    
    synchronized void post() {
    	if ( notify )
      		actionListener.actionPerformed(new ActionEvent(this, 0, ""));
    }
}
