package nn.pp.rc;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.io.*;
import java.awt.geom.*;
import java.text.*;
import java.util.*;
import nn.pp.rckbd.*;

/**
 * The actual representation of the remote frame buffer.
 * The constructor wont try to make a connection to a RFB.
 * The canvas will initialy assume a size and display a black
 * window (we should better take a LOGO... ;-). 
 * The console can be connected to an RFB by setting a connected
 * RFBproto. Ones this is done the console will exchange some 
 * of its specific requirements with the server and display the 
 * remote Buffer
 *
 * @author Thomas Breitfeld, Peppercon AG
 */
public class RCCanvasPanel
    extends Panel implements ContentScalingInfo {

    ServerConsolePanelBase scp;
    PrintStream	logger;
    RCHandler rchandler;
    KeyEventHandler keyevth;
    KeyTranslator keytrans;
    MouseHandler mhandler;
    Locale locale;
    
    Hashtable hashMouseSync;
    boolean bMouseSyncCombo;
    Hashtable hashFullScreen;
    boolean bFullScreenCombo;
    int syncState;
    String[] synckeycodes;
    int currentKvmPort = 0;

    Dimension fbsize;                            // framebuffer pixel dimensions
    Dimension scaled_fb_size;
    double    scaleX     = 1.0, scaleY = 1.0;
    boolean   is100pc    = true;
    boolean   scaleToFit = false;
    boolean   interpol   = false;
    boolean   debug      = false;
    boolean   monitorMode;
    
    RCRenderer rdr;
    
    public RCCanvasPanel(ServerConsolePanelBase scp, Locale locale,
			  PrintStream logger, int initWidth, int initHeight) {
	
	this.logger = logger;
	this.scp = scp;
        this.locale = locale;

    	try {
    	    enableInputMethods(false);
    	} catch (Throwable ignore) {
    	}

	fbsize = new Dimension(initWidth, initHeight);
	scaled_fb_size = fbsize;
    }

    public void setParameters(boolean monitorMode, String synckeycodes, String fskeycodes, KeyEventHandler keyevth) {
	this.monitorMode = monitorMode;
	this.keyevth = keyevth;
	
	keytrans = KbdFactory.getInstance().
	    getKeyTranslator(keyevth.getActualLayout().getName(),
			     locale/*Locale.GERMANY*/);

	try {
	    initMouseSync(synckeycodes);
	} catch (ParseException e) {
	    logger.println(T._("Console: Mousesync hotkey error:") + " " + e);
	}
	try {
	    initFullScreen(fskeycodes);
	} catch (ParseException e) {
	    logger.println(T._("Console: Fullscreen hotkey error:") + " " + e);
	}

        long eventMask = AWTEvent.MOUSE_EVENT_MASK |
	                 AWTEvent.MOUSE_MOTION_EVENT_MASK |
	                 AWTEvent.MOUSE_WHEEL_EVENT_MASK |
		         AWTEvent.KEY_EVENT_MASK |
 	                 AWTEvent.FOCUS_EVENT_MASK;

	enableEvents(eventMask);
    }

    public void setRCHandler(RCHandler rchandler) {
	if (this.rchandler != null)
	    keyevth.removeKeyboardListener(this.rchandler);
	this.rchandler = rchandler;
	if (keyevth != null) keyevth.addKeyboardListener(this.rchandler);
	rdr = rchandler.getRenderer();
	if (rdr != null) rdr.setInterpol(interpol);
    }
    
    public void setRCRenderer(RCRenderer rdr) {
    	this.rdr = rdr;
    }

    public void setMouseHandler(MouseHandler mhandler) {
	this.mhandler = mhandler;
    }

    public void update(Graphics g) {
	paint(g);
    }
    
    public void paint(Graphics g) {
	if(rdr != null)
	    rdr.paint(g, !is100pc, scaled_fb_size);
    }

    public String toString() {
	if (rchandler == null) {
	    return T._("Console not connected");
	}
	return MessageFormat.format(T._("Console({0}): Desktop size is {1} x {2}"),
	                            new Object[] { rchandler, new Integer(fbsize.width),
	                                           new Integer(fbsize.height) });
    }

    public Dimension getPreferredSize() {
	return scaled_fb_size;
    }

    public Dimension getMinimumSize()  {
	return scaled_fb_size;
    }

    public void setKbdLocale(Locale locale) {
        keyevth.getActualLayout();
        keyevth.getActualLayout().getName();
	keytrans = KbdFactory.getInstance().
	    getKeyTranslator(keyevth.getActualLayout().getName(), locale);
    }

    public void setMonitorMode(boolean mode) {
	monitorMode = mode;
	if (rchandler != null) {
	    rchandler.proto.setMonitorMode(mode);
	}
    }

    public void adjustSize(Dimension newsize, boolean force) {
	if (force || newsize.width != fbsize.width
	    || newsize.height != fbsize.height) {
	    fbsize = newsize;
	    if(rdr != null) rdr.setRenderSize(fbsize);
            if(scaleToFit) {
                setScaleToFit(scp.getBlankSize());
            } else {
                Dimension sz = setScaledDimension(newsize.width,
						  newsize.height);
                scp.canvasSizeChanged(sz);
            }
	    logger.println(this);
	}
    }
    
    public void forceAdjustSize() {
    	adjustSize(fbsize, true);
    }

    public void setInterpol(boolean interpol) {
	this.interpol = interpol;
	rdr.setInterpol(interpol);
    }
    
    public void setScale(double sx, double sy) {
        //System.out.println("Canvas: setScales: "+sx+"x"+sy);
        if(sx > 0.95 && sx < 1.05 && sy > 0.95 && sy < 1.05) {
            if(is100pc) return;
            is100pc = true;
            scaleX = scaleY = 1.0;
        } else {
            is100pc = false;
            this.scaleX = sx;
            this.scaleY = sy;
        }
        Dimension newd = setScaledDimension(fbsize.width, fbsize.height);
	if(!scaleToFit) scp.canvasSizeChanged(newd);
    }

    public Point2D.Double getScale() {
	return new Point2D.Double(this.scaleX, this.scaleY);
    }

    public void setScaleToFit(boolean stf) {
        if(stf == scaleToFit) return;
        scaleToFit = stf;
        if(scaleToFit) {
            setScaleToFit(scp.getBlankSize());
        } else {
            setScale(1.0, 1.0);
        }
    }

    private void setScaleToFit(Dimension scrsize) {
        setScale((double)scrsize.width/(double)fbsize.width,
                 (double)scrsize.height/(double)fbsize.height);
    }

    private Dimension setScaledDimension(int fbwidth, int fbheight){
        scaled_fb_size = new Dimension((int)(scaleX * fbwidth),
                                       (int)(scaleY * fbheight));
        setSize(scaled_fb_size);
        return scaled_fb_size;
    }

    int buttonmask = 0;

    public void processEvent(AWTEvent e) {
	MouseEvent me;
	KeyEvent ke;
	int eID;
	boolean virtualAltGr = false;

	if (rchandler == null) return;
	if (debug) if(e instanceof KeyEvent) logger.println(e);

	try {
	    eID = e.getID();

	    if(e instanceof ComponentEvent && eID == ComponentEvent.COMPONENT_RESIZED) {
		if(scaleToFit) {
		    ServerConsolePanelBase scp =
			(ServerConsolePanelBase)e.getSource();
		    setScaleToFit(scp.getBlankSize());
		}
	    }

	    if (e instanceof MouseEvent) {
		me = (MouseEvent)e;
		mhandler.handleMouseEvent(me);
	    }

	    else if (e instanceof KeyEvent) {
		ke = (KeyEvent)e;

		  /*
		   * we try to translate every KeyEvent, no matter
		   * what type it is. most of the time this will
		   * fail for KEY_TYPED events as our translation
		   * is based on keycodes. however this is not always
		   * true. every successfully translated event is
		   * passed forward to our own KeyEventHandler that
		   * tries to do its java key event reconstruction
		   * magic, in case some events our missing...
		   */

		  Short keynr = keytrans.translateKeyEvent(ke);
		  if(null == keynr) {
		      if( KeyEvent.KEY_TYPED != eID)
			  System.out.println("ERROR: no key number found for " +e);
		      return;
		  } 
		  if(debug) System.out.println("Key Translator " +
					       keytrans.getLocale() +
					       " found key number: " + keynr);

		  if (ke.isAltGraphDown() && (keynr.intValue() != 57) && (keyevth.altgrFlag == false) && (eID == KeyEvent.KEY_PRESSED)) {
		      /* System.out.println(" !!! Using virtual AltGr-Press !!!"); */
		      keyevth.pressed(new Short((short)57), keyevth.KEYBOARD);
		      virtualAltGr = true;
		  }

		  handleMouseSyncKey(keynr, eID);
		  handleFullScreenKey(keynr, eID);

		  if ((keytrans instanceof KeyTranslator_ja) &&
		      (keynr.intValue() >= 108 ||
		       keynr.intValue() == 0)
		      ) {
		      
		      if (eID == KeyEvent.KEY_PRESSED) {
		          keyevth.handleHardKbdEvent(keynr, KeyEvent.KEY_TYPED);
		      }
		  } else {
		      keyevth.handleHardKbdEvent(keynr, eID);
		  }

		  if (virtualAltGr) {
		      /* System.out.println(" !!! Release virtual AltGr-Press !!!"); */
		      keyevth.released(new Short((short)57), keyevth.KEYBOARD);
		      virtualAltGr = false;
		  }

		  ke.consume();
	    }
	    
	    else if (e instanceof FocusEvent) {
		  keyevth.releaseAllKeys();
	    }
	} catch (Exception f) {
	    // FIXME
	    f.printStackTrace();
	}
    }

    /**
     * check if user pressed the sync hotkey combo      
     */
    private boolean[] handleKey(Hashtable hash, Short keynr, int eID, boolean combo) {
    	boolean ret[] = new boolean[2];
    	ret[0] = false;
    	ret[1] = combo;
    	
	if ((hash == null) || hash.isEmpty()) return ret;
	switch (eID) {
	  case KeyEvent.KEY_PRESSED:
	      if (hash.containsKey(keynr)) {
		  hash.put(keynr, new Boolean(true));
	      }                   
	      break;
	  case KeyEvent.KEY_RELEASED:
	      if (hash.containsKey(keynr)) {
		  hash.put(keynr, new Boolean(false));
	      }                   
	      break;     
	}

	// now we look if our sync combination is currently pressed
	if (hash.contains(new Boolean(false)) == false) {
	    if (debug) System.out.println("Sync: all keys pressed");
	    if (combo) {
		ret[0] = true;
		ret[1] = false;
	    }
	}
	
	if (hash.contains(new Boolean(true)) == false) {
	    ret[1] = true;
	}
	
	return ret;
    }
    
    public void handleMouseSyncKey(Short keynr, int eID) {
	boolean b[] = handleKey(hashMouseSync, keynr, eID, bMouseSyncCombo);
	if (b[0]) {
	    mhandler.handleMouseSyncKey();
	}
	bMouseSyncCombo = b[1];
    }
	
    public void handleFullScreenKey(Short keynr, int eID) {
	boolean b[] = handleKey(hashFullScreen, keynr, eID, bFullScreenCombo);
	if (b[0]) {
	    scp.changeWindowMode();
	}
	bFullScreenCombo = b[1];
    }
	
    /**
     * init vars and parse hotkey for manual mouse syncing
     */
    private void initKeys(Hashtable hash, String kcodes) throws ParseException {
	if (kcodes == null) return;
	
	StringTokenizer st = new StringTokenizer(kcodes);
	while(st.hasMoreTokens()) {
	    try {
		int c = Integer.parseInt(st.nextToken(), 16);
		if (debug) System.out.println("Sync: parsed keyNo: " + c);
		hash.put(new Short((short)c), new Boolean(false));
	    } catch(NumberFormatException e) {
		throw new ParseException(kcodes, 0);
	    }
	}
    }
    
    public void initMouseSync(String kcodes) throws ParseException {
	bMouseSyncCombo = true;
	hashMouseSync   = new Hashtable();

	initKeys(hashMouseSync, kcodes);
    }    

    public void initFullScreen(String kcodes) throws ParseException {
	bFullScreenCombo = true;
	hashFullScreen   = new Hashtable();

	initKeys(hashFullScreen, kcodes);
    }    

    /**
     * kvm port events are a special case that we currently receive
     * from the Button Panel only, however, you never know... so
     * we are going to implement sending the events here
     */
    public void switchKvmPort(short newport) {
	if (rchandler == null) return;
	try { 
	    rchandler.writeKvmSwitchEvent(newport);
	    currentKvmPort = newport;
	} catch(IOException e) {
	    logger.println(T._("Console: Kvm port switch error:") + " "
			   + e.getMessage());
	}
    }
}
