package nn.pp.rc;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.awt.datatransfer.*;
import java.io.*;

// This class is used to hold an image while on the clipboard.
class ImageSelection implements Transferable {
    private Image image;

    public ImageSelection(Image image) {
        this.image = image;
    }

    // Returns supported flavors
    public DataFlavor[] getTransferDataFlavors() {
        return new DataFlavor[]{DataFlavor.imageFlavor};
    }

    // Returns true if flavor is supported
    public boolean isDataFlavorSupported(DataFlavor flavor) {
        return DataFlavor.imageFlavor.equals(flavor);
    }

    // Returns image
    public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
        if (!DataFlavor.imageFlavor.equals(flavor)) {
            throw new UnsupportedFlavorException(flavor);
        }
        return image;
    }
}

/*
 * some really basic stuff, all our RFB renderers will need
 * 
 * @author Thomas Rendelmann, Peppercon AG
 */
public abstract class RFBRenderer
    implements RCRenderer, ActionListener, ImageObserver {

    protected RFBproto  rfb;
    protected RFBHandler handler;
    protected Component comp;
    protected Dimension	mysize;
    protected Image     vimg;
    protected Graphics  gvimg;
    protected Image	jimg;
    protected Graphics	gjimg;

    /*
     * Buffered reading in the renderer
     *
     * We use a buffered reading here to avoid the virtual call
     * of the stream's read method. Currently, the following
     * rules must be followed to use the buffered reading:
     *
     * - Reading is only possible bytewise (this could be
     *   changed by omplementing other read methods).
     * - After the first buffered read, no unbuffered
     *   access to the data stream is allowed (because
     *   the previously buffered data would be skipped).
     *   Please pay attention to it when multithreading
     *   is used!
     * - when the buffered reading is finished, 
     *   finishBufferedReading() must be called to hand
     *   the bytes which are buffered, but not used, back
     *   to the data stream
     * - after that, unbuffered access is allowed again
     *
     */
    private int bufsize = 64*1024;
    private byte[] buf = new byte[bufsize];
    private int count = 0;
    private int pos = 0;

    protected final int readBufferedByte(MonitoringDataInputStream is) throws IOException {
    	if((count - pos) <= 0) {
    	    count = is.read(buf);
    	    pos = 0;
    	}
    	return buf[pos++] & 0xff;
    }
    
    protected final void finishBufferedReading(MonitoringDataInputStream is) {
        is.stall(buf, pos, count - pos);
        pos = count = 0;
    }
    
    // OSD ... TODO better in own class, but the size, the siizeee!!1
    private final int default_osd_height = 20;
    private final int default_osd_margin = 20;
    private GenericTimer osd_timer = null;
    private Color osd_bgcolor = new Color(0x20, 0x20, 0x80);
    private Color osd_fgcolor = new Color(0x60, 0x60, 0xF0);
    private Color osd_transcolor = new Color(0, 0, 0, 0);
    private int osd_alpha = 100;
    private int osd_position = ConnectionParameters.OSD_TOP;
    private boolean osd_needs_double_buffering;
    private Font osd_font;
    private int osd_width;
    private int osd_height = 0;
    private FontMetrics osd_metrics;
    private String osd_message;
    private boolean osd_blank  = false;
    private boolean screen_blank = false;
    private Image osd_image = null;
    private boolean osd_show = false;
    private Image osd_offscreen; 
    private Graphics osd_graphics; 
    
    abstract void sendPixelMsg() throws IOException;
    
    abstract void sendUpdateMsg()	throws IOException;
	
    abstract void drawRawRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException;
    
    abstract void drawRawVSCRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException;
    
    abstract void drawHextileRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException;
    
    abstract void drawTightRect(int x, int y, int w, int h)
	throws IOException;

    abstract void drawTightCachedRect(int x, int y, int w, int h)
	throws IOException;

    abstract void drawCopyRect(int srcX, int srcY, int x, int y, int w, int h)
	throws IOException;

    abstract void drawLRLERect(long encoding, int x, int y, int w, int h)
	throws IOException;

    abstract void setNewTightCacheSize();
    
    abstract void enableTightCache(boolean enable);
    
    public abstract void repaint();

    public abstract void setInterpol(boolean ip);
       
    public abstract void dispose();

    public RFBRenderer(Component comp) {
	this.comp = comp;
	osd_font = new Font("Monospaced", Font.BOLD, default_osd_height);
	osd_metrics = Toolkit.getDefaultToolkit().getFontMetrics(osd_font);
    }

    public void setRFBProto(RFBproto rfb) {
	this.rfb = rfb;
    }

    public void setRFBHandler(RFBHandler handler) {
	this.handler = handler;
    }

    public void setRenderSize(Dimension d) {
	mysize = d;
	generateOSDImage();
	// generate a new jpeg image if we already have an old one
	if (jimg != null) {
	    createJImg(d);
	}
	// generate a new back buffer for OSD rendering
	if (osd_needs_double_buffering) {
	    osd_offscreen = comp.createImage(d.width, d.height);
	    osd_graphics = osd_offscreen.getGraphics(); 
	}
    }
    
    private void createJImg(Dimension d) {
    	if (jimg != null) {
    	    jimg.flush();
    	}
    	jimg = comp.createImage(d.width, d.height);
    	gjimg = jimg.getGraphics();
    }

    public void actionPerformed(ActionEvent e) {	
	osd_show = false;
	repaint();
    }

    public void setOSDColorAndPosition(Color bgColor, Color fgColor, int alpha, int position) {
        osd_bgcolor = bgColor;
        osd_fgcolor = fgColor;
        osd_alpha = alpha;
        osd_position = position;
        osd_needs_double_buffering = osd_position == ConnectionParameters.OSD_CENTER || osd_alpha != 100;
    }

    public void setOSDState(boolean blank, String message, int timeout_ms) {
	osd_blank = blank;
	osd_message = message;

	generateOSDImage();
	
	if (osd_timer != null) {
	    osd_timer.stop();
	}
	
	if (timeout_ms > 0 && osd_show == true) {
	    osd_timer = new GenericTimer(timeout_ms, this);
	    osd_timer.start();
	}

	repaint();
    }

    void setBlank(boolean blank) {
    	screen_blank = blank;
    	repaint();
    }

    private void generateOSDImage() {
	if (osd_message != null && osd_message.length() > 0) {
	    Graphics osd_g;
	    int text_width, text_height;
	    
	    text_width = osd_metrics.stringWidth(osd_message);
	    text_height= osd_metrics.getMaxDescent() + osd_metrics.getMaxAscent();

	    if (osd_position == ConnectionParameters.OSD_CENTER) {
	        osd_width = text_width + default_osd_margin;
	        osd_height = default_osd_height + default_osd_margin;
	    } else {
	        osd_width = mysize.width;
	        osd_height = default_osd_height;
	    }
	    
	    osd_image = comp.createImage(osd_width, osd_height);
	    osd_g = osd_image.getGraphics();
            osd_g.setColor(osd_bgcolor);
            osd_g.fillRect(0, 0, osd_width, osd_height);
	    osd_g.setColor(osd_fgcolor);
	    osd_g.setFont(osd_font);
	    osd_g.drawString(osd_message, (osd_width - text_width)/2,
			     osd_height - (osd_height - text_height)/2 - osd_metrics.getMaxDescent());
	    osd_show = true;
	} else {
	    osd_show = false;
	}
    }

    protected void paintContent(Graphics g, boolean scale, Dimension scalesize) {
	Dimension size;

	size = (scale) ? scalesize : mysize;

	if (osd_show) {
	    if (osd_needs_double_buffering) {
	        // we need double buffering if either the OSD is transparent or it is in the center of the screen;
	        // double buffering in other cases won't hurt, but it eats up much performance, so we distinguish here
	        int osd_y = 0;
	        int osd_x = Math.max((mysize.width - osd_width) / 2, 0);

                switch (osd_position) {
                    case ConnectionParameters.OSD_TOP: osd_y = 0; break;
                    case ConnectionParameters.OSD_BOTTOM: osd_y = mysize.height - osd_height; break;
                    case ConnectionParameters.OSD_CENTER: osd_y = (mysize.height - osd_height) / 2; break;
                }

                Graphics2D g2d = (Graphics2D)osd_graphics;

                if (osd_alpha != 100) {
                    AlphaComposite alpha = AlphaComposite.getInstance(AlphaComposite.SRC, (float)1.0);
                    g2d.setComposite(alpha);
                }

                if (osd_blank || screen_blank) {
            	    osd_graphics.setColor(Color.black);
            	    osd_graphics.fillRect(0, 0, mysize.width, mysize.height);
                } else {
            	    osd_graphics.drawImage(vimg, 0, 0, mysize.width, mysize.height, null);
                }

                if (osd_alpha != 100) {
                    AlphaComposite alpha = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, (float)osd_alpha / 100);
                    g2d.setComposite(alpha);
                }
                
                osd_graphics.drawImage(osd_image, osd_x, osd_y, osd_width, osd_height, null);
                
                g.drawImage(osd_offscreen, 0, 0, size.width, size.height, null);
	    } else {
	        int osd_height_draw = scalesize.height*osd_height/mysize.height;
	        int osd_width_draw = scalesize.width*osd_width/mysize.width;

	        int osd_y = 0;
	        int osd_x = Math.max((size.width - osd_width_draw) / 2, 0);

                switch (osd_position) {
                    case ConnectionParameters.OSD_TOP:
                        osd_y = 0;
                        g.setClip(0, osd_height_draw, size.width, size.height-osd_height_draw);
                        break;
                    case ConnectionParameters.OSD_BOTTOM:
                        osd_y = size.height - osd_height_draw;
                        g.setClip(0, 0, size.width, size.height-osd_height_draw);
                        break;
                }
                
                if (osd_blank || screen_blank) {
            	    g.setColor(Color.black);
            	    g.fillRect(0, 0, size.width, size.height);
                } else {
            	    g.drawImage(vimg, 0, 0, size.width, size.height, null);
                }
                	    
                g.setClip(osd_x, osd_y, osd_width_draw, osd_height_draw);
                g.drawImage(osd_image, osd_x, osd_y, osd_width_draw, osd_height_draw, null);
            }
	} else {
	    g.setClip(0, 0, size.width, size.height);
	    if (osd_blank || screen_blank) {
		g.setColor(Color.black);
		g.fillRect(0, 0, size.width, size.height);		
	    } else {
		g.drawImage(vimg, 0, 0, size.width, size.height, null);//ohne size schneller?
	    }
	}
    }

    /* JPEG encoding stuff */
    private Rectangle jpegRect;
    protected MemoryImageSource  jimgsrc;

    public boolean imageUpdate(Image img, int infoflags,
                               int _x, int _y, int _w, int _h) {
    	if ((infoflags & (ALLBITS | ABORT)) == 0) {
    	    return true;		// We need more image data.
    	} else {
    	    // If the whole image is available, draw it now.
    	    if ((infoflags & ALLBITS) != 0) {
    	    	if (jpegRect != null) {
    	    	    synchronized(jpegRect) {
    	    	    	gjimg.drawImage(img, 0, 0, null);
    	    	    	jpegRect.notify();
    	    	    }
    	    	}
    	    }
    	    return false;		// All image data was processed.
    	}
    }

    public void drawJpegRect(int x, int y, int w, int h) throws IOException {
	int size = (int)rfb.is.readUnsignedInt();
	byte jpegData[] = new byte[size];
	rfb.is.readFully(jpegData);
	
	// Create an Image object from the JPEG data.
	Image jpegImage = Toolkit.getDefaultToolkit().createImage(jpegData);
	
	// Remember the rectangle where the image should be drawn.
	jpegRect = new Rectangle(x, y, w, h);
	
	if (jimg == null) {
	    createJImg(mysize);
	}
	
	// Let the imageUpdate() method do the actual drawing, here just
	// wait until the image is fully loaded and drawn.

    	synchronized(jpegRect) {
	    Toolkit.getDefaultToolkit().prepareImage(jpegImage, -1, -1, this);
	    try {
	    	// Wait no longer than three seconds.
	    	jpegRect.wait(3000);
	    } catch (InterruptedException e) {
	    	throw new IOException("Interrupted while decoding JPEG image");
	    }
    	}

    	// Done, jpegRect is not needed any more.
    	jpegRect = null;

        jimgsrc.newPixels(x, y, w, h);
        gvimg.setClip(x, y, w, h);
        gvimg.drawImage(jimg, x, y, null);
        gvimg.setClip(0, 0, mysize.width, mysize.height);
    }
    
    /* screenshot clipboard stuff */
    public void createScreenshot() {
        System.out.println("Creating screenshot");
        
        ImageSelection imgSel = new ImageSelection(vimg);
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents(imgSel, null);
    }
}
