package nn.pp.rcrdp;

import java.io.*;
import nn.pp.rc.T;

public class RDPStream  {
    private boolean debug = false;
    
    // the data in the stream
    private byte packet[];
    
    // some position variables
    private int position = 0;
    private int start = 0;
    private int end = -1;
    
    // the size of the packet
    private int size = 0;
    
    // positions of various headers
    private int isoHdr = -1;
    private int mcsHdr = -1;
    private int secHdr = -1;
    private int rdpHdr = -1;
    
    // constants for the different header types
    public final static int isoHdrID = 1;
    public final static int mcsHdrID = 2;
    public final static int secHdrID = 3;
    public final static int rdpHdrID = 4;
    
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // constructor; gets the number of bytes to allocate
    public RDPStream(int size) {
        this.size = size;
        packet = new byte[size];
    }
    
    // reallocate a (larger) buffer
    public void reallocate(int newSize) {
    	byte newPacket[] = new byte[newSize];
    	System.arraycopy(packet, 0, newPacket, 0, size);
    	size = newSize;
    	packet = newPacket;
    }
    
    // reallocate a (larger) buffer
    public void reallocateNoCopy(int newSize) {
    	packet = new byte[newSize];
    	size = newSize;
    }
    
    // refill the stream with 0's
    public void clear(int len) {
    	if(len == 0) return;
    	packet[0] = 0;
    	for(int i = 1; i < len; i += i) {
    	    System.arraycopy(packet, 0, packet, i, ((len - i) < i) ? (len - i) : i);
    	}
    }
    
    // some set/get functions
    public byte[] getPacket() {
    	return packet;
    }
    
    public int getStart() {
	return start;
    }

    public void setStart(int position) {
	start = position;
    }
    
    public void markEnd() {
	end = position;
    }
    
    public void markEnd(int position) {
	if (position > this.size) {
	    debug("markEnd: End (" + position + ") > size (" + size + ")");
	    throw new ArrayIndexOutOfBoundsException(T._("markEnd: End > size"));
	}
	end = position;
    }

    public int getEnd() {
	return end;
    }
    
    public int getSize() {
	return size;
    }
   
    public int getPosition() {
	return position;
    }

    public void incrementPosition(int length) {
	if (length + position > size || length < 0) {
	    debug("incrementPosition: new pos (" + (length + position) + ") > size (" + size + ")");
	    throw new ArrayIndexOutOfBoundsException(T._("incrementPosition: pos > size"));
	}
	position += length;
    }

    public void decrementPosition(int length) {
    	if ((position - length) < 0) {
	    debug("decrementPosition: new pos (" + (position - length) + ") < 0");
	    throw new ArrayIndexOutOfBoundsException(T._("decrementPosition: pos < 0"));
	}
    	position -= length;
    }

    public void setPosition(int position) {
	if (position > size || position < 0) {
	    debug("setPosition: new pos (" + (position) + ") > size (" + size + ")");
	    throw new ArrayIndexOutOfBoundsException(T._("setPosition: pos > size"));
	}
	this.position=position;
    }
    
    // check operations
    public boolean checkSize() {
    	return position <= end;
    }
    
    public boolean checkRemaining(int n) {
    	return position + n <= end;
    }
    
    public boolean checkEnd() {
    	return position == end;
    }
    
    // data read and write operations
    // 8 Bit operations
    public int inUInt8() {
	return packet[position++] & 0x000000ff;
    }
    
    public void outUInt8(int val) {
	packet[position++] = (byte) (val & 0x000000ff);
    }
    
    public int inSInt8() {
    	int i = inUInt8();
    	// convert from unsigned to signed
    	byte b = (byte)(i & 0xff);
    	return b;
    }
    
    public int nextBE(int val) {
	return (val << 8) + inUInt8();
    }
    
    // 16 Bit operations
    public int inUInt16LE() {
	int val = packet[position++] & 0x000000ff;
	val |= (packet[position++] & 0x000000ff) << 8;
	return val;
    }
    
    public int inUInt16BE() {
	int val = (packet[position++] & 0x000000ff) << 8;
	val |= packet[position++] & 0x000000ff;
	return val;
    }
    
    public int inSInt16LE() {
    	int i = inUInt16LE();
    	// convert to signed
    	short s = (short)(i & 0xffff);
    	return s;
    }
    
    public int inSInt16BE() {
    	int i = inUInt16BE();
    	// convert to signed
    	short s = (short)(i & 0xffff);
    	return s;
    }
    
    public int inUInt16() {
	return inUInt16BE();
    }
    
    public void outUInt16LE(int val) {
	packet[position++] = (byte) (val & 0x000000ff);
	packet[position++] = (byte) ((val >> 8) & 0x000000ff);
    }
    
    public void outUInt16BE(int val) {
	packet[position++] = (byte) ((val >> 8) & 0x000000ff);
	packet[position++] = (byte) (val & 0x000000ff);
    }
    
    public void outUInt16(int val) {
	outUInt16LE(val);
    }
    
    // 32 Bit operations
    public int inUInt32LE() {
	int val = inUInt16LE() & 0x0000ffff;
	val |= (inUInt16LE() & 0x0000ffff) << 16;
	return val;
    }
    
    public int inUInt32BE() {
	int val = (inUInt16BE() & 0x0000ffff) << 16;
	val |= inUInt16BE() & 0x0000ffff;
	return val;
    }
    
    public int inUInt32() {
	return inUInt32BE();
    }
    
    public void outUInt32LE(int val) {
	outUInt16LE(val & 0x0000ffff);
	outUInt16LE((val >> 16) & 0x0000ffff);
    }
    
    public void outUInt32BE(int val) {
	outUInt16BE((val >> 16) & 0x0000ffff);
	outUInt16BE(val & 0x0000ffff);
    }
    
    public void outUInt32(int val) {
	outUInt32BE(val);
    }
    
    // array operations
    public void inUInt8Array(byte array[], int offset, int length) {
	System.arraycopy(packet, position, array, offset, length);
	position += length;
    }
    
    public void outUInt8Array(byte array[], int offset, int length) {
	System.arraycopy(array, offset, packet, position, length);
	position += length;
    }
    
    public void inUInt8ArrayNoCount(byte array[], int offset, int pos, int length) {
	System.arraycopy(packet, pos, array, offset, length);
//	position += length;
    }
    
    public void outUInt8ArrayNoCount(byte array[], int offset, int pos, int length) {
	System.arraycopy(array, offset, packet, pos, length);
//	position += length;
    }
    
    // Unicode/String operations
    public void outUnicodeString(String s, int len) {
    	int i = 0, j = 0;
    	len += 2;	//terminating 0
    	if(s.length() != 0) {
    	    char sc[] = s.toCharArray();
    	    // send the data out
    	    while(i < Math.min(s.length() * 2, len)) {
    	    	outUInt16LE((int)sc[j++]);
    	    	i += 2;
    	    }
	    // set terminating 0 i s is longer than len
	    if(s.length() * 2 > len - 2) {
		decrementPosition(2);
		outUInt16LE(0);
	    }
    	    // fill up with 0
    	    while(i < len) {
    	    	outUInt16LE(0);
    	    	i += 2;
    	    }
	}
	else {
	    while(i < len) {
	    	outUInt16LE(0);
	    	i += 2;
	    }
	}
    }
    
    public void outByteString(String s, int len) {
    	int i = 0, j = 0;
    	len++;	//terminating 0
    	if(s.length() != 0) {
    	    byte sc[] = null;
    	    try {
    	    	sc = s.getBytes("US-ASCII");
    	    } catch(UnsupportedEncodingException e) {
    	    	return;
    	    }
    	    // send the data out
    	    while(i < Math.min(s.length(), len)) {
    	    	outUInt8((int)sc[j++]);
    	    	i++;
    	    }
	    // set terminating 0 i s is longer than len
	    if(s.length() > len - 1) {
		decrementPosition(1);
		outUInt8(0);
	    }
    	    // fill up with 0
    	    while(i < len) {
    	    	outUInt8(0);
    	    	i++;
    	    }
	}
	else {
	    while(i < len) {
	    	outUInt8(0);
	    	i++;
	    }
	}
    }
    
    public void outByteString(String s) {
    	outByteString(s, s.length());
    }

    public void outByteStringNoTerm(String s, int len) {
    	int i = 0, j = 0;
    	if(s.length() != 0) {
    	    byte sc[] = null;
    	    try {
    	    	sc = s.getBytes("US-ASCII");
    	    } catch(UnsupportedEncodingException e) {
    	    	return;
    	    }
    	    // send the data out
    	    while(i < Math.min(s.length(), len)) {
    	    	outUInt8((int)sc[j++]);
    	    	i++;
    	    }
    	    // fill up with 0 if space left
    	    while(i < len) {
    	    	outUInt8(0);
    	    	i++;
    	    }
	}
	else {
	    while(i < len) {
	    	outUInt8(0);
	    	i++;
	    }
	}
    }
    
    public void outByteStringNoTerm(String s) {
    	outByteStringNoTerm(s, s.length());
    }

    // layer push/pop operations
    public void pushLayer(int layer, int size) {
	switch(layer) {
	    case RDPStream.isoHdrID:
		isoHdr = getPosition();
		incrementPosition(size);
		break;
	    case RDPStream.mcsHdrID:
		mcsHdr = getPosition();
		incrementPosition(size);
		break;
	    case RDPStream.secHdrID:
		secHdr = getPosition();
		incrementPosition(size);
		break;
	    case RDPStream.rdpHdrID:
		rdpHdr = getPosition();
		incrementPosition(size);
		break;
        }
    }

    public void popLayer(int layer) {
        switch(layer) {
	    case RDPStream.isoHdrID:
		setPosition(isoHdr);
		break;
	    case RDPStream.mcsHdrID:
		setPosition(mcsHdr);
		break;
	    case RDPStream.secHdrID:
		setPosition(secHdr);
		break;
	    case RDPStream.rdpHdrID:
		setPosition(rdpHdr);
		break;
        }
    }
}
