package nn.pp.rcrdp;

import java.io.*;
import java.security.*;
import java.math.*;

import nn.pp.rc.T;
import nn.pp.rcrdp.crypto.*;

/**
 * class SecureLayer
 *
 * the Secure-Layer of the MS RDP protocol
 */
 
public class SecureLayer {
    private boolean debug = false;
    
    private MCSLayer mcsLayer;
    
    private byte cryptedRandom[];
    
    private int rc4KeyLen;
    
    private byte signKey[];
    private byte decryptKey[];
    private byte encryptKey[];
    private byte decryptUpdateKey[];
    private byte encryptUpdateKey[];
    
    private byte[] modulus;
    private byte[] exponent;
    private byte[] serverRandom;
    
    private RDPProfile profile;
    private RDPProto rdpProto;
    private Licence licence;
    
    private SecureRandom rand;
    
    private RC4 rc4Decrypt;
    private RC4 rc4Encrypt;
    
    private int encCounter;
    private int decCounter;
    
    private String hostname;
    private String username;
    private int serverRDPVersion;
    
    private int kbdLayout;
    
    private static final byte[] pad54 = {
	54, 54, 54, 54, 54, 54, 54, 54, 54, 54,
	54, 54, 54, 54, 54, 54, 54, 54, 54, 54,
	54, 54, 54, 54, 54, 54, 54, 54, 54, 54,
	54, 54, 54, 54, 54, 54, 54, 54, 54, 54
    };
    
    private static final byte[] pad92 = {
	92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92,
	92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92,
	92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92,
	92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92
    };
    
    // constructor
    public SecureLayer(RDPProfile profile, RDPProto rdp, int kbdLayout) {
    	mcsLayer = new MCSLayer(this);
    	rc4Encrypt = new RC4();
    	rc4Decrypt = new RC4();
    	
    	rdpProto = rdp;
    	this.profile = profile;
    	
    	this.kbdLayout = kbdLayout;
    	
    	rand = new SecureRandom();
    	
    	signKey = new byte[16];
    	decryptKey = new byte[16];
    	encryptKey = new byte[16];
    	decryptUpdateKey = new byte[16];
    	encryptUpdateKey = new byte[16];
    	
    	encCounter = decCounter = 0;
    	
    	// generate random numbers
    	cryptedRandom = new byte[RDPConstants.secModulusSize];
    	generateRandom(cryptedRandom);
    }
    
    public void setKbdKayout(int kbdLayout) {
    	this.kbdLayout = kbdLayout;
    }

    // internal functions
    
    // debug print
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // generate random numbers
    private void generateRandom(byte[] data) {
    	// FIXME
    	// debug variante
    	/*
    	for(int i = 0; i < data.length; i++) {
    	    data[i] = (byte)i;
    	}
    	*/
    	// normal variante
    	
    	rand.nextBytes(data);
    	
    }
    
    /* Reduce key entropy from 64 to 40 bits */
    private void make40Bit(byte[] key) {
	key[0] = (byte)0xd1;
	key[1] = (byte)0x26;
	key[2] = (byte)0x9e;
    }
    
    /* Generate a session key and RC4 keys, given client and server randoms */
    private void generateKeys(byte[] clientKey, byte[] serverKey, int rc4KeySize) throws CryptoException {
    	byte sessionKey[];
    	byte tempHash[];
    	byte input[] = new byte[48];
    	
    	byte temp[] = new byte[16];
    	
	// Construct input data to hash
	System.arraycopy(clientKey, 0, input, 0, 24);
	System.arraycopy(serverKey, 0, input, 24, 24);
	
	if(debug) {
	    int i;
    	    System.out.println("RDP SecureLayer rsaEncrypt: input");
    	    for(i = 0; i < input.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(input[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
	}
	
	// Generate session key - two rounds of sec_hash_48
	tempHash = hash48(input, clientKey, serverKey, 65);
	
	if(debug) {
	    int i;
    	    System.out.println("RDP SecureLayer rsaEncrypt: tempHash");
    	    for(i = 0; i < tempHash.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(tempHash[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
	}
	
	sessionKey = hash48(tempHash, clientKey, serverKey, 88);
	
	// Store first 16 bytes of session key, for generating signatures
	System.arraycopy(sessionKey, 0, signKey, 0, 16);
	
	// Generate RC4 keys
	System.arraycopy(sessionKey, 16, temp, 0, 16);
	decryptKey = hash16(temp, clientKey, serverKey);
	System.arraycopy(sessionKey, 32, temp, 0, 16);
	encryptKey = hash16(temp, clientKey, serverKey);
	
	if(rc4KeySize == 1) {
	    make40Bit(signKey);
	    make40Bit(decryptKey);
	    make40Bit(encryptKey);
	    rc4KeyLen = 8;
	}
	else {
	    rc4KeyLen = 16;
	}
	
	// Save initial RC4 keys as update keys
	System.arraycopy(decryptKey, 0, decryptUpdateKey, 0, 16);
	System.arraycopy(encryptKey, 0, encryptUpdateKey, 0, 16);
	
	if(debug) {
	    int i;
    	    System.out.println("RDP SecureLayer rsaEncrypt: decryptKey");
    	    for(i = 0; i < decryptKey.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(decryptKey[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	    System.out.println("RDP SecureLayer rsaEncrypt: encryptKey");
    	    for(i = 0; i < encryptKey.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(encryptKey[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
	}
	
	rc4Decrypt.engineInitDecrypt(decryptKey, rc4KeyLen);
	rc4Encrypt.engineInitEncrypt(encryptKey, rc4KeyLen);
    }
    
    // Encrypt data using RC4
    private void encrypt(byte[] data, int pos, int len, byte[] out, int outPos) throws CryptoException {
    	if(encCounter == 4096) {
    	    encryptKey = update(encryptKey, encryptUpdateKey);
    	    rc4Encrypt.engineInitEncrypt(encryptKey, rc4KeyLen);
    	    encCounter = 0;
	}
	
	rc4Encrypt.crypt(data, pos, len, out, outPos);
	encCounter++;
    }
    
    // Decrypt data using RC4
    private void decrypt(byte[] data, int pos, int len, byte[] out, int outPos) throws CryptoException {
    	if(decCounter == 4096) {
    	    decryptKey = update(decryptKey, decryptUpdateKey);
    	    rc4Decrypt.engineInitDecrypt(decryptKey, rc4KeyLen);
    	    decCounter = 0;
    	}
    	
    	rc4Decrypt.crypt(data, pos, len, out, outPos);
    	decCounter++;
    }
    
    // reverses a byte array
    private void reverse(byte[] data) {
    	byte temp;
    	int i, j;
    	for(i = 0, j = data.length - 1; i < j; i++, j--) {
    	    temp = data[i];
    	    data[i] = data[j];
    	    data[j] = temp;
    	}
    }
    
    private void reverse(byte[] data, int len) {
    	byte temp;
    	int i, j;
    	for(i = 0, j = len - 1; i < j; i++, j--) {
    	    temp = data[i];
    	    data[i] = data[j];
    	    data[j] = temp;
    	}
    }
    
    // Perform an RSA public key encryption operation
    private byte[] rsaEncrypt(byte[] in, int len) {
        int i;

    	if(debug) {
    	    System.out.println("RDP SecureLayer rsaEncrypt: Exponent");
    	    for(i = 0; i < exponent.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(exponent[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	    System.out.println("RDP SecureLayer rsaEncrypt: Modulus");
    	    for(i = 0; i < modulus.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(modulus[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	    System.out.println("RDP SecureLayer rsaEncrypt: in");
    	    for(i = 0; i < in.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(in[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	}
    	
        byte[] out = new byte[RDPConstants.secModulusSize];
        byte[] inr = new byte[len];
        
        byte[] temp;
        
        BigInteger mod, exp, x, y;
        
        reverse(modulus);
        reverse(exponent);        
        System.arraycopy(in, 0, inr, 0, len);
        reverse(inr, len);
        
    	if(debug) {
    	    System.out.println("RDP SecureLayer rsaEncrypt: inr");
    	    for(i = 0; i < inr.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(inr[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	}

        //BN_bin2bn(modulus, SEC_MODULUS_SIZE, mod)
        if ((modulus[0] & 0x80) != 0) {
            temp = new byte [modulus.length+1];
            System.arraycopy(modulus, 0, temp, 1, modulus.length);
            temp[0]=0;
            mod = new BigInteger(temp);
        } else {
            mod = new BigInteger(modulus);
        }

        if ((exponent[0] & 0x80) != 0) {
            temp = new byte [exponent.length+1];
            System.arraycopy(exponent, 0, temp, 1, exponent.length);
            temp[0]=0;
            exp = new BigInteger(temp);
        } else {
            exp = new BigInteger(exponent);
        }
        
        if ((inr[0] & 0x80) != 0) {
            temp = new byte [inr.length+1];
            System.arraycopy(inr, 0, temp, 1, inr.length);
            temp[0]=0;
            x = new BigInteger(temp);
        } else {
            x = new BigInteger(inr);
        }

        y = x.modPow(exp, mod);
        
        out = y.toByteArray();
        
        reverse(out);
        
    	if(debug) {
    	    System.out.println("RDP SecureLayer rsaEncrypt: outLen = " + out.length);
    	    System.out.println("RDP SecureLayer rsaEncrypt: out");
    	    for(i = 0; i < out.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(out[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	}

        return out;
    }
    
    // Transfer the client random to the server
    private void establishKey() throws IOException {
    	int len = RDPConstants.secModulusSize + RDPConstants.secPaddingSize;
    	int flags = RDPConstants.secClientRandom;
    	RDPStream s;
    	
    	s = init(flags, RDPConstants.secModulusSize + RDPConstants.secPaddingSize + 4);
    	
    	s.outUInt32LE(len);
    	s.outUInt8Array(cryptedRandom, 0, RDPConstants.secModulusSize);
    	s.incrementPosition(RDPConstants.secPaddingSize);
    	//for(int i = 0; i < RDPConstants.secPaddingSize; i++) s.outUInt8(0);
    	
    	s.markEnd();
    	send(s, flags);
    }
    
    // Output connect initial data blob
    private void outMCSData(RDPStream s) {
    	int hostlen = hostname.length() * 2;
    	int len = 158 + 76 + 12 + 4 + 20;	// ?
    	
    	if(hostlen > 30) hostlen = 30;
    	
    	s.outUInt16BE(5);	// unknown
    	s.outUInt16BE(0x14);
    	s.outUInt8(0x7c);
    	s.outUInt16BE(1);
    	
    	s.outUInt16BE(len | 0x8000);	// remaining length
    	
    	s.outUInt16BE(8);	// length ???
    	s.outUInt16BE(16);
    	s.outUInt8(0);
    	s.outUInt16LE(0xc001);
    	s.outUInt8(0);
    	
    	s.outUInt32LE(0x61637544);	// "Duca" ???
    	s.outUInt16BE((len - 14) | 0x8000);	// remaining length
    	
    	// Client information
    	s.outUInt16LE(RDPConstants.secTagCliInfo);
    	s.outUInt16LE(212);		// length
    	s.outUInt16LE(1);		// RDP version. 1 == RDP4, 4 == RDP5
    	s.outUInt16LE(8);
    	s.outUInt16LE(rdpProto.getWidth());
    	s.outUInt16LE(rdpProto.getHeight());
    	s.outUInt16LE(0xca01);
    	s.outUInt16LE(0xaa03);
    	s.outUInt32LE(kbdLayout);	// keyboard layout
    	s.outUInt32LE(2600);	// Client build. We are now 2600 compatible :-)
    	
    	// Unicode name of client, padded to 32 bytes
    	s.outUnicodeString(hostname, hostlen);
    	s.incrementPosition(30 - hostlen);
    	
    	s.outUInt32LE(4);
    	s.outUInt32(0);
    	s.outUInt32LE(12);
    	s.incrementPosition(64);	// reserved? 4 + 12 doublewords
    	
    	switch(profile.hostBpp) {
    	    case 8:
    	    	s.outUInt16LE(0xca01);
    	    	break;
    	    case 15:
    	    	s.outUInt16LE(0xca02);
    	    	break;
    	    case 16:
    	    	s.outUInt16LE(0xca03);
    	    	break;
    	    case 24:
    	    	s.outUInt16LE(0xca04);
    	    	break;
    	}
    	s.outUInt16LE(1);
    	
    	s.outUInt32(0);
    	s.outUInt8(profile.hostBpp);
    	s.outUInt16LE(0x0700);
    	s.outUInt8(0);
    	s.outUInt32LE(1);
    	s.incrementPosition(64);	// End of client info
    	
    	s.outUInt16LE(RDPConstants.secTagCli4);
    	s.outUInt16LE(12);
    	s.outUInt32LE(9);
    	s.outUInt32(0);
    	
    	// Client encryption settings
    	s.outUInt16LE(RDPConstants.secTagCliCrypt);
    	s.outUInt16LE(12);	// length
    	s.outUInt32LE(profile.encryption ? 0x3 : 0);	// encryption supported, 128-bit supported
    	s.outUInt32(0);		// unknown
    	
    	s.outUInt16LE(RDPConstants.secTagCliChannels);
    	s.outUInt16LE(20);	// length
    	s.outUInt32LE(1);	// number of virtual channels
    	s.outByteStringNoTerm("cliprdr", 8);	// name padded to 8(?)
    	s.outUInt16(0);
    	s.outUInt16LE(0xc0a0);	// Flags
    	
    	s.markEnd();
    }
    
    // Parse a public key structure
    private boolean parsePublicKey(RDPStream s) throws IOException {
    	int magic, modulusLen;
    	
    	magic = s.inUInt32LE();
    	if(magic != RDPConstants.secRsaMagic) {
    	    throw new IOException(T._("RDP: RSA magic wrong"));
    	}
    	
    	modulusLen = s.inUInt32LE();
    	if(modulusLen != RDPConstants.secModulusSize + RDPConstants.secPaddingSize) {
    	    throw new IOException(T._("RDP: RSA modulus length wrong"));
    	}
    	
    	s.incrementPosition(8);		// modulus_bits, unknown
    	
    	exponent = new byte[RDPConstants.secExponentSize];
    	s.inUInt8Array(exponent, 0, RDPConstants.secExponentSize);
    	modulus = new byte[RDPConstants.secModulusSize];
    	s.inUInt8Array(modulus, 0, RDPConstants.secModulusSize);
    	s.incrementPosition(RDPConstants.secPaddingSize);
    	
    	return s.checkSize();
    }
    
    // Parse a crypto information structure
    private int parseCryptInfo(RDPStream s) throws IOException {
    	int rc4KeyLen = 0, cryptLevel, randomLen, rsaInfoLen;
    	int end, tag, len, nextTag;
    	int flags;
    	
    	rc4KeyLen = s.inUInt32LE();	// 1 = 40-Bit 2 = 128 Bit
    	cryptLevel = s.inUInt32LE();	// 1 = low, 2 = medium, 3 = high
    	
    	if(cryptLevel == 0) {		// no encryption
    	    return 0;
    	}
    	
    	randomLen = s.inUInt32LE();
    	rsaInfoLen = s.inUInt32LE();
    	
    	if(randomLen != RDPConstants.secRandomSize) {
    	    throw new IOException(T._("RDP: wrong random length received"));
    	}
    	
    	serverRandom = new byte[randomLen];
    	s.inUInt8Array(serverRandom, 0, randomLen);
    	
    	end = s.getPosition() + rsaInfoLen;
    	if(end > s.getEnd()) {
    	    return 0;
    	}
    	
    	flags = s.inUInt32LE();
    	if((flags & 1) != 0) {
    	    // RDP 4 style encryption
    	    s.incrementPosition(8);	// unknown
    	    while(s.getPosition() < end) {
    	    	tag = s.inUInt16LE();
    	    	len = s.inUInt16LE();
    	    	
    	    	nextTag = s.getPosition() + len;
    	    	
    	    	switch(tag) {
    	    	    case RDPConstants.secTagPubkey:
    	    	    	if(!parsePublicKey(s)) {
    	    	    	    return 0;
    	    	    	}
    	    	    	break;
    	    	    case RDPConstants.secTagKeysig:
			/* Is this a Microsoft key that we just got? */
			/* Care factor: zero! */
			/* Actually, it would probably be a good idea to check if the public key is signed with this key, and then store this 
			   key as a known key of the hostname. This would prevent some MITM-attacks. */
			break;
		    default:
		        throw new IOException(T._("RDP: parseCryptInfo: received unknown tag"));
    	    	}
    	    	
    	    	s.setPosition(nextTag);
    	    }
    	}
    	else {
    	    // rdp 5 style encryption; not supported
    	    throw new IOException(T._("RDP: no RDP4 encryption received"));
    	}
    	    	
    	return rc4KeyLen;
    }
    
    // Process crypto information blob
    private void processCryptInfo(RDPStream s) throws IOException {
    	byte clientRandom[] = new byte[RDPConstants.secRandomSize];
    	int rc4KeyLen = parseCryptInfo(s);
    	
    	if(rc4KeyLen == 0) {
    	    System.out.println("failed to parse crypt info");
    	    return;
    	}
    	
    	generateRandom(clientRandom);
    	
    	cryptedRandom = rsaEncrypt(clientRandom, RDPConstants.secRandomSize);
    	
    	// FIXME
    	//generateRandom(serverRandom);	// only for debugging!!
    	try {
    	    generateKeys(clientRandom, serverRandom, rc4KeyLen);
    	}
    	catch (CryptoException e) {
    	    throw new IOException(T._("RDP: error during crypto operation"));
    	}
    }
    
    // Process SRV_INFO, find RDP version supported by server
    private void processSrvInfo(RDPStream s) {
    	serverRDPVersion = s.inUInt16LE();
    }
    
    
    // exported functions

    
    // get the particular layers
    public MCSLayer getMCSLayer() {
    	return mcsLayer;
    }
    
    public ISOLayer getISOLayer() {
    	return mcsLayer.getISOLayer();
    }
    
    public TCPLayer getTCPLayer() {
    	return mcsLayer.getTCPLayer();
    }
    
    /*
     * General purpose 48-byte transformation, using two 32-byte salts (generally,
     * a client and server salt) and a global salt value used for padding.
     * Both SHA1 and MD5 algorithms are used.
     */
    public byte[] hash48(byte[] in, byte[] salt1, byte[] salt2, int salt) throws CryptoException {
    	byte shasig[];
    	byte pad[] = new byte[4];
    	byte out[] = new byte[48];
    	byte temp[];
    	SHA1 sha = new SHA1();
    	MD5 md5 = new MD5();
    	
    	for(int i = 0; i < 3; i++) {
    	    //memset(pad, salt + i, i + 1);
    	    for(int j = 0; j <= i; j++) {
    	    	pad[j] = (byte) (salt + i);
    	    }
    	    
    	    sha.engineUpdate(pad, i + 1);
    	    sha.engineUpdate(in, 48);
    	    sha.engineUpdate(salt1, 32);
    	    sha.engineUpdate(salt2, 32);
    	    shasig = sha.engineDigest();
    	    
    	    md5.engineUpdate(in, 48);
    	    md5.engineUpdate(shasig, 20);
    	    temp = md5.engineDigest();
    	    System.arraycopy(temp, 0, out, i*16, 16);
    	}
    	
    	return out;
    }
    
    /*
     * Weaker 16-byte transformation, also using two 32-byte salts, but
     * only using a single round of MD5.
     */
    public byte[] hash16(byte[] in, byte[] salt1, byte[] salt2) throws CryptoException {
	MD5 md5 = new MD5();

	md5.engineUpdate(in, 16);
	md5.engineUpdate(salt1, 32);
	md5.engineUpdate(salt2, 32);
	return md5.engineDigest();
    }
    
    public byte[] hash16(byte[] in, byte[] salt1, byte[] salt2, int pos) throws CryptoException {
	MD5 md5 = new MD5();

	md5.engineUpdate(in, pos, 16);
	md5.engineUpdate(salt1, 32);
	md5.engineUpdate(salt2, 32);
	return md5.engineDigest();
    }
    
    
    // Output a uint32 into a buffer (little-endian)
    public void bufOutUInt32(byte[] buffer, int value) {
	buffer[0] = (byte)((value) & 0xff);
	buffer[1] = (byte)((value >>> 8) & 0xff);	// >>> = unsigned shoft operator
	buffer[2] = (byte)((value >>> 16) & 0xff);
	buffer[3] = (byte)((value >>> 24) & 0xff);
    }
    
    // Generate a signature hash, using a combination of SHA1 and MD5
    public byte[] sign(int sigLen, byte[] sessionKey, int keyLen, byte[] data, int dataLen) throws CryptoException {
    	byte shaSig[];
    	byte md5Sig[];
    	byte lenHdr[] = new byte[4];
    	byte signature[] = new byte[sigLen];
    	SHA1 sha = new SHA1();
    	MD5 md5 = new MD5();
    	
    	bufOutUInt32(lenHdr, dataLen);
    	
    	sha.engineUpdate(sessionKey, keyLen);
    	sha.engineUpdate(pad54, 40);
    	sha.engineUpdate(lenHdr, 4);
    	sha.engineUpdate(data, dataLen);
    	shaSig = sha.engineDigest();
    	
    	md5.engineUpdate(sessionKey, keyLen);
    	md5.engineUpdate(pad92, 48);
    	md5.engineUpdate(shaSig, 20);
    	md5Sig = md5.engineDigest();
    	
    	System.arraycopy(md5Sig, 0, signature, 0, sigLen);    	
    	
    	return signature;
    }
    
    // Update an encryption key - similar to the signing process
    public byte[] update(byte[] key, byte[] updateKey) throws CryptoException {
    	byte shaSig[];
    	byte newKey[] = new byte[rc4KeyLen];
    	byte update[] = new byte[rc4KeyLen];
    	SHA1 sha = new SHA1();
    	MD5 md5 = new MD5();
    	RC4 updateEngine = new RC4();
    	
    	sha.engineUpdate(updateKey, rc4KeyLen);
    	sha.engineUpdate(pad54, 40);
    	sha.engineUpdate(key, rc4KeyLen);
    	shaSig = sha.engineDigest();
    	
    	md5.engineUpdate(updateKey, rc4KeyLen);
    	md5.engineUpdate(pad92, 48);
    	md5.engineUpdate(shaSig, 20);
    	key = md5.engineDigest();
    	
    	// copy the new key... just to be sure
    	System.arraycopy(key, 0, newKey, 0, rc4KeyLen);
    	System.arraycopy(key, 0, update, 0, rc4KeyLen);
    	
    	updateEngine.engineInitEncrypt(update);
    	newKey = updateEngine.crypt(update);
    	
    	if(rc4KeyLen == 8) {
    	    make40Bit(newKey);
    	}
    	
    	return newKey;
    }
    
    // Initialise secure transport packet
    public RDPStream init(int flags, int maxLen) {
    	RDPStream s = null;
    	int hdrLen;
    	
    	if(!licence.licenceIssued()) {
    	    hdrLen = ((flags & RDPConstants.secEncrypt) != 0) ? 12 : 4;
	}
	else {
	    hdrLen = ((flags & RDPConstants.secEncrypt) != 0) ? 12 : 0;
	}
	
	s = mcsLayer.init(hdrLen + maxLen);
    	s.pushLayer(RDPStream.secHdrID, hdrLen);
    	
    	return s;
    }
    
    public RDPStream initNew(int flags, int maxLen) {
    	RDPStream s = null;
    	int hdrLen;
    	
    	if(!licence.licenceIssued()) {
    	    hdrLen = ((flags & RDPConstants.secEncrypt) != 0) ? 12 : 4;
	}
	else {
	    hdrLen = ((flags & RDPConstants.secEncrypt) != 0) ? 12 : 0;
	}
	
	s = mcsLayer.initNew(hdrLen + maxLen);
    	s.pushLayer(RDPStream.secHdrID, hdrLen);
    	
    	return s;
    }
    
    byte sendBuffer[] = null;
    private byte[] getSendBuffer(int len) {
    	if(sendBuffer == null || sendBuffer.length < len) {
    	    sendBuffer = new byte[len];
    	}
    	return sendBuffer;
    }
    
    // Transmit secure transport packet
    // this has to be synchronized because:
    //   * 2 Threads can send (Ping Pong Timer and AWT Event Thread)
    //   * RC4 is stream oriented, which means only one Thread should
    //     use it at the same time
    //   * re-use of the buffer
    synchronized public void send(RDPStream s, int flags) throws IOException {
    	int dataLen;
    	
    	s.popLayer(RDPStream.secHdrID);
    	if(!licence.licenceIssued()  || ((flags & RDPConstants.secEncrypt) != 0)) {
    	    s.outUInt32LE(flags);
    	}
    	
    	if((flags & RDPConstants.secEncrypt) != 0) {
    	    flags &= ~RDPConstants.secEncrypt;
    	    dataLen = s.getEnd() - s.getPosition() - 8;
    	    
    	    byte[] data = getSendBuffer(dataLen);
    	    s.inUInt8ArrayNoCount(data, 0, s.getPosition() + 8, dataLen); //this doesn't set the internal stream pointer
    	    byte[] signature = null;

    	    try {
    	    	signature = sign(8, signKey, rc4KeyLen, data, dataLen);
    	    	// encrypt directly into the packet
    	    	encrypt(data, 0, dataLen, s.getPacket(), s.getPosition() + 8);
    	    }
    	    catch (CryptoException e) {
    	    	throw new IOException(T._("RDP SecureLayer send: Error during encryption"));
    	    }
    	    s.outUInt8ArrayNoCount(signature, 0, s.getPosition(), 8);
    	}
    	
    	mcsLayer.send(s);
    }
    
    // Process connect response data blob
    public void processMCSData(RDPStream s) throws IOException {
    	int tag, len, length, nextTag = 0;
    	
    	s.incrementPosition(21);	// header (T.124 stuff, probably)
    	len = s.inUInt8();
    	if((len & 0x80) != 0) {
    	    len = s.inUInt8();
    	}
    	
    	while(s.getPosition() < s.getEnd()) {
    	    tag = s.inUInt16LE();
    	    length = s.inUInt16LE();
    	    
    	    if(length <= 4) {
    	    	return;
    	    }
    	    
    	    nextTag = s.getPosition() + length - 4;
    	    
    	    switch(tag) {
    	    	case RDPConstants.secTagSrvInfo:
    	    	    processSrvInfo(s);
    	    	    break;
    	    	case RDPConstants.secTagSrv3:
    	    	    break;
    	    	case RDPConstants.secTagSrvCrypt:
    	    	    processCryptInfo(s);
    	    	    break;
    	    	default:
    	    	    //throw new IOException("RDP: processMCSData: received unknown tag");
    	    	    System.out.println("RDP: processMCSData: received unknown tag");
    	    }
    	    
    	    s.setPosition(nextTag);
    	}
    }
    
    byte recvBuffer[] = null;
    private byte[] getRecvBuffer(int len) {
    	if(recvBuffer == null || recvBuffer.length < len) {
    	    recvBuffer = new byte[len];
    	}
    	return recvBuffer;
    }

    // Receive secure transport packet
    public RDPStream receive() throws IOException {
    	RDPStream s = null;
    	int secFlags;
    	
    	while((s = mcsLayer.receive()) != null) {
    	    if(profile.encryption && !licence.licenceIssued()) {
    	    	secFlags = s.inUInt32LE();
    	    	
    	    	if((secFlags & RDPConstants.secLicenceNeg) != 0) {
    	    	    licence.process(s);
    	    	    continue;
    	    	}
    	    	
    	    	if((secFlags & RDPConstants.secEncrypt) != 0) {
    	    	    s.incrementPosition(8);	// signature
    	    	    int len = s.getEnd() - s.getPosition();
    	    	    
    	    	    byte data[] = getRecvBuffer(len);
    	    	    
    	    	    s.inUInt8ArrayNoCount(data, 0, s.getPosition(), len);
    	    	    try {
    	    	    	decrypt(data, 0, len, s.getPacket(), s.getPosition());
    	    	    }
    	    	    catch (CryptoException e) {
    	    	    	throw new IOException(T._("RDP: secure receive: decryption error"));
    	    	    }
    	    	}
    	    	
    	    	return s;
    	    }
    	}
    	
    	return s;
    }
    
    // Establish a secure connection
    public void connect(String server, int port, String username, String hostname) throws IOException {
    	RDPStream mcsData = new RDPStream(512);
    	
    	this.hostname = hostname;

    	// generate the licence handler
    	licence = new Licence(this, hostname, username);
    	
    	outMCSData(mcsData);
    	this.username = username;
    	
    	mcsLayer.connect(server, port, mcsData, username);
    	
    	if(profile.encryption) {
    	    establishKey();
    	}
    }
    
    // Disconnect a connection
    public void disconnect() {
    	mcsLayer.disconnect();
    }
    
    public int getMCSUserID() {
    	return mcsLayer.getUserID();
    }
}
	