package nn.pp.sara;

import java.awt.*;
import java.util.*;
import java.text.*;
import java.net.*;
import java.awt.event.*;
import nn.pp.rc.*;

public class InfoPanel extends FramePanel {
    MousePanel mousePanel;
    KeyboardPanel keyboardPanel;
    SessionsPanel sessionsPanel;
    SecurityPanel securityPanel;
    EventLogPanel eventLogPanel;
    SimpleDateFormat dateFormatShort, dateFormatLong;
    boolean localOnly, noKbd;
    Hashtable sessionHash;
    
    Panel visualPanel, panelPart1, panelPart2;
    
    String title = "Forensic Console";
    
    public InfoPanel(URL codeBase, Locale locale, String defkbd,
    		     SasForensicPanel parentPanel, boolean localOnly, boolean noKbd) {
    	super(codeBase, parentPanel);

    	this.localOnly = localOnly;
    	this.noKbd = noKbd;
    	
    	dateFormatShort = new SimpleDateFormat("HH:mm:ss.SSS");
    	dateFormatLong = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
    	
    	sessionHash = new Hashtable();

    	if (!noKbd) {
    	    keyboardPanel = new KeyboardPanel(this, locale, defkbd);
    	}
    	mousePanel = new MousePanel(this);
    	if (!localOnly) {
    	    sessionsPanel = new SessionsPanel(this);
    	    securityPanel = new SecurityPanel(this);
    	}
    	eventLogPanel = new EventLogPanel(this);
    	
    	addLayout();
    }
    
    protected String getFrameName() {
    	return title;
    }

    protected void addVisualPanels(boolean inFrame) {
    	if (inFrame) {
    	    visualPanel.remove(panelPart1);
    	    visualPanel.remove(panelPart2);

    	    if (!localOnly) {
    	    	visualPanel.setLayout(new GridLayout(2, 2));
    	    
    	    	if (sessionsPanel != null) visualPanel.add(sessionsPanel);
    	    	if (securityPanel != null) visualPanel.add(securityPanel);
    	    } else if (keyboardPanel != null) {
    	    	visualPanel.setLayout(new GridLayout(1, 2));
    	    } else {
    	    	visualPanel.setLayout(new GridLayout(1, 1));
    	    }
    	    
    	    if (keyboardPanel != null) visualPanel.add(keyboardPanel);
    	    if (mousePanel != null)    visualPanel.add(mousePanel);
    	} else {
    	    if (sessionsPanel != null) visualPanel.remove(sessionsPanel);
    	    if (securityPanel != null) visualPanel.remove(securityPanel);
    	    if (keyboardPanel != null) visualPanel.remove(keyboardPanel);
    	    if (mousePanel != null)    visualPanel.remove(mousePanel);

    	    visualPanel.setLayout(new BorderLayout());
    	    
    	    if (!localOnly) {
    	    	if (sessionsPanel != null) panelPart1.add(sessionsPanel, BorderLayout.WEST);
    	    	if (securityPanel != null) panelPart1.add(securityPanel, BorderLayout.CENTER);
    	    	
    	    	visualPanel.add(panelPart1, BorderLayout.WEST);
    	    }
    	    
    	    if (keyboardPanel != null) panelPart2.add(keyboardPanel, BorderLayout.CENTER);
    	    if (mousePanel != null)    panelPart2.add(mousePanel, BorderLayout.EAST);

    	    visualPanel.add(panelPart2, BorderLayout.EAST);
    	}
    }
    
    private void addLayout() {
    	eventLogPanel.addLayout();
    	if (mousePanel != null)    mousePanel.addLayout();
    	if (keyboardPanel != null) keyboardPanel.addLayout();
    	if (sessionsPanel != null) sessionsPanel.addLayout();
    	if (securityPanel != null) securityPanel.addLayout();
    	
    	mainPanel = new Panel(new BorderLayout());
    	
    	Panel textClearPanel = new Panel();
    	textClearPanel.setLayout(new BorderLayout());
    	
    	Panel frameButtonPanel = getFrameButtonPanel();
    	
    	textClearPanel.add(frameButtonPanel, BorderLayout.WEST);
    	textClearPanel.add(eventLogPanel, BorderLayout.CENTER);
    	
    	mainPanel.add(textClearPanel, BorderLayout.CENTER);
    	
    	visualPanel = new Panel();
    	panelPart1 = new Panel(new BorderLayout());
    	panelPart2 = new Panel(new BorderLayout());
    	
    	addVisualPanels(inFrame);
    	
    	// we add a dummy label here to CENTER to avoid that our
    	// visual panels are stretched (so the dummy label is stretched)
    	Label dummyLabel = new Label();
    	Panel visualDummyPanel = new Panel();
    	visualDummyPanel.setLayout(new BorderLayout());
    	visualDummyPanel.add(visualPanel, BorderLayout.WEST);
    	visualDummyPanel.add(dummyLabel, BorderLayout.CENTER);

    	mainPanel.add(visualDummyPanel, BorderLayout.SOUTH);
    	
    	setLayout(new BorderLayout());
    	
    	showInitial();
    }
    
    public void setEnabled(boolean b) {
    	eventLogPanel.setEnabled(b);
    	super.setEnabled(b);
    }
    
    synchronized void putSession(Long key, UserSession newEntry) {
    	UserSession e;
    	
    	if ((e = (UserSession)sessionHash.get(key)) != null) {
    	    System.out.println("Already found session 0x" + Long.toHexString(key.longValue()) + ": " + e.toString());
    	    sessionHash.remove(key);
    	}

    	sessionHash.put(key, newEntry);
    }
    
    synchronized UserSession getSession(Long key) {
    	UserSession e = (UserSession)sessionHash.get(key);
    	if (e == null) {
    	    e = new UserSession("(unknown)", "(unknown)", new Date(), false);
    	    putSession(key, e);
    	}
    	
    	return e;
    }
    
    synchronized void removeSession(Long key) {
    	sessionHash.remove(key);
    }
    
    synchronized void removeAllSessions() {
    	sessionHash.clear();
    }
    
    synchronized Vector getSessions() {
    	Vector vec = new Vector();
    	for (Enumeration e = sessionHash.elements(); e.hasMoreElements(); ) {
    	    UserSession entry = (UserSession)e.nextElement();
    	    vec.add(entry.toString());
    	}
    	
    	return vec;
    }
    
    void newEventText(InfoPanelPart source, SasEvent evt, String s) {
    	eventLogPanel.newEvent(s);
    	if (source != null) {
    	    source.logEvent(evt, s);
    	}
    }
    
    void newEvent(InfoPanelPart source, SasEvent evt, String name, String text, boolean showUser) {
    	String s = getEventTime(evt, true) + " - " + name;
    	if (showUser && !localOnly) {
    	    if (evt.session == 0) {
    	    	s += " - local user";
    	    } else {
    	    	s += " - User: " + getEventUser(evt) + ", IP: " + getEventIp(evt);
    	    }
    	}
    	if (text != null) {
    	    s += " - " + text;
    	}
    	newEventText(source, evt, s);
    }

    void newEvent(InfoPanelPart source, SasEvent evt, String name, String text) {
    	newEvent(source, evt, name, text, true);
    }
    
    void newEvent(InfoPanelPart source, SasEvent evt, String name, boolean showUser) {
    	newEvent(source, evt, name, null, showUser);
    }

    void newEvent(InfoPanelPart source, SasEvent evt, String name) {
    	newEvent(source, evt, name, null, true);
    }

    public void newKbdMapping(Locale locale) {
    	if (keyboardPanel != null) keyboardPanel.newKbdMapping(locale);
    }
    
    public void newKbdLayout(String layout) {
    	if (keyboardPanel != null) {
    	    keyboardPanel.newKbdLayout(layout);
    	    if (inFrame) {
    	        frame.pack();
    	    } else {
    	        pack();
    	    }
    	}
    }

    /* handling of the specific SAS events */
    String getEventUser(SasEvent evt) {
    	if (evt.session == (long)0) {
    	    return "(local)";
    	}
    	UserSession s = getSession(new Long(evt.session));
    	return s.user;
    }
    
    String getEventIp(SasEvent evt) {
    	if (evt.session == (long)0) {
    	    return "(local)";
    	}
    	UserSession s = getSession(new Long(evt.session));
    	return s.ip;
    }
    
    synchronized String getEventTime(Date time, boolean full) {
    	SimpleDateFormat dateFormat;
    	if (full) {
    	    dateFormat = dateFormatLong;
    	} else {
    	    dateFormat = dateFormatShort;
    	}
    	return dateFormat.format(time).toString();
    }
    
    String getEventTime(SasEvent evt, boolean full) {
    	return getEventTime(evt.time, full);
    }
    
    // session events
    private void handleExistingSession(SasEvent evt) {
    	if (sessionsPanel == null) return;
    	
    	sessionsPanel.handleSasEvent(evt);
    	newEventText(sessionsPanel, evt, "Session - User: " + getEventUser(evt) + ", IP: " + getEventIp(evt) + ", Login: " + 
    		getEventTime(evt.exSession.loginTime, true));
    }
    
    private void handleExistingKvmSession(SasEvent evt) {
    	if (sessionsPanel == null) return;
    	
    	sessionsPanel.handleSasEvent(evt);
    	newEventText(sessionsPanel, evt, "KVM Session - User: " + getEventUser(evt) + ", IP: " + getEventIp(evt) + ", Login: " + 
    		getEventTime(evt.exKvmSession.loginTime, true));
    }
    
    private void handleSessionEvent(SasEvent evt) {
    	if (sessionsPanel != null) sessionsPanel.handleSasEvent(evt);
    }

    // input events
    private void handleInput(SasEvent evt) {
    	if (evt.inputEvent.type == SasEvent.InputKeyboard) {
    	    if (keyboardPanel != null) keyboardPanel.handleSasEvent(evt);
    	} else {
    	    if (mousePanel != null) mousePanel.handleSasEvent(evt);
    	}
    }

    // other events
    private void handleUserLoginFailure(SasEvent evt) {
    	if (securityPanel != null) securityPanel.handleSasEvent(evt);
    }
    
    private void handleKvmSwitchEvent(SasEvent evt) {
    	newEvent(null, evt, "KVM Switch", "channel: " + evt.kvmSwitchEvent.channel + " unit: " + evt.kvmSwitchEvent.unit + " port: " + evt.kvmSwitchEvent.port);
    }

    public void handleSasEvent(SasEvent evt) {
    	switch (evt.type) {
    	    /* session events */
    	    case SasEvent.ExistingSession:
    	    	if (evt.exSession != null) {
    	    	    handleExistingSession(evt);
    	    	}
    	    	break;
    	    case SasEvent.ExistingKvmSession:
    	    	if (evt.exKvmSession != null) {
    	    	    handleExistingKvmSession(evt);
    	    	}
    	    	break;
    	    case SasEvent.UserSessionOpened:
    	    case SasEvent.UserSessionClosed:
    	    case SasEvent.KvmSessionOpened:
    	    case SasEvent.KvmSessionClosed:
    	    case SasEvent.KvmExclusiveOn:
    	    case SasEvent.KvmExclusiveOff:
    	    case SasEvent.ResetSessions:
    	    	handleSessionEvent(evt);
    	    	break;
    	    
    	    /* input events */
    	    case SasEvent.Input:
    	    	if (evt.inputEvent != null) {
    	    	    handleInput(evt);
    	    	}
    	    	break;

    	    /* other events */
    	    case SasEvent.UserLoginFailure:
    	    	handleUserLoginFailure(evt);
    	    	break;
    	    
    	    case SasEvent.KvmSwitch:
    	    	handleKvmSwitchEvent(evt);
    	    	break;
    	}
    }
    
    void setHost(String hostName, String hostAddress) {
    	title = "Forensic Console - " + hostName + " (" + hostAddress + ")";
    	setTitle(title);
    }
}
