#include <sys/poll.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <liberic_pthread.h>
#include <pp/base.h>
#include <pp_gpio.h>

static const char *usage_msg =
    "\n"
    "Options:\n"
    "     -d <device>                           ..  the i2c device file\n"
    "     -b <pin>                              ..  read one bit (return value: error=-1 low=0 high=1)\n"
    "     -v <0|1|z>                            ..  write one bit (use together with -b)\n"
    "     -r                                    ..  read the current GPIO input\n"
    "     -s <hex value>                        ..  set the <value> GPIO bits\n"
    "     -c <hex value>                        ..  clear the <value> GPIO bits\n"
    "     -z <hex value>                        ..  set the <value> GPIO bits to tristate\n"
    "     -i <hex value>                        ..  subscribe for IRQs on the <value> GPIO bits\n"
    "     -q <4 hex values, comma separated>    ..  change IRQ trigger params\n"
    "                                               1. level, 2. edge, 3. pos edge/active high, 4. neg edge/active low\n"
    "     -a                                    ..  read alternate functions\n"
    "     -A <2 hex value, comma separated>     ..  set/clr alternate functions of these GPIOs\n"
    "\n";

static pthread_t irq_thread;
static void* irq_thread_func(void* arg);
static int fd = -1;

int
main(int argc, char ** argv)
{
 
    int ret = -1;
    int c, errflag = 0;
    u_char do_bit = 0, do_val = 0, do_read = 0, do_set = 0, do_clr = 0, do_tri = 0,
           do_irq = 0, do_irqconf = 0, do_altconf_get = 0, do_altconf_set = 0;
    u_char bit = 0, val = 0;
    u_long set = 0, clr = 0, tri = 0, irq = 0, altconf_set = 0, altconf_clr = 0;
    u_long irqconf_level = 0, irqconf_edge = 0, irqconf_pos = 0, irqconf_neg = 0;
    char* filename = NULL;
    
    while ((c = getopt(argc, argv, "d:rs:c:z:i:q:A:ab:v:")) != -1) {
	switch (c) {
	  case 'd':
	      filename = strdup(optarg);
	      break;
          case 'b':
              do_bit = 1;
              bit = atoi(optarg);
              break;
          case 'v':
              do_val = 1;
              switch (optarg[0]) {
              case 'z': val = 'z'; break;
              case '1': val = 1; break;
              default:  val = 0;
              }
              break;
	  case 'r':
	      do_read = 1;
	      break;
	  case 's':
	      do_set = 1;
	      set = (u_long)strtoll(optarg, NULL, 16);
	      break;
	  case 'c':
	      do_clr = 1;
	      clr = (u_long)strtoll(optarg, NULL, 16);
	  break;
	  case 'z':
	      do_tri = 1;
	      tri = (u_long)strtoll(optarg, NULL, 16);
	      break;
	  case 'i':
	      do_irq = 1;
	      irq = (u_long)strtoll(optarg, NULL, 16);
	      break;
	  case 'q':
	      {
		  char *end;

		  irqconf_level = (u_long)strtoll(optarg, &end, 16);
		  if (*end == '\0') break;
		  end++;
		  irqconf_edge = (u_long)strtoll(end, &end, 16);
		  if (*end == '\0') break;
		  end++;
		  irqconf_pos = (u_long)strtoll(end, &end, 16);
		  if (*end == '\0') break;
		  end++;
		  irqconf_neg = (u_long)strtoll(end, &end, 16);
		  
		  do_irqconf = 1;
		  
		  break;
	      }
	  case 'a':
	      do_altconf_get = 1;
	      break;
	  case 'A':
	      {
		  char *end;
		  
		  altconf_set = (u_long)strtoll(optarg, &end, 16);
		  if (*end == '\0') break;
		  end++;
		  altconf_clr = (u_long)strtoll(end, &end, 16);
		  
		  do_altconf_set = 1;
		  break;
	      }
	  case '?':
	      printf("Unknown option");
	      errflag++;
	      break;
	}
    }

    if (!filename) {
	printf("No device given\n");
	printf(usage_msg);
	goto bail;
    }

    if ((fd = open(filename, O_RDWR)) < 0) {
	printf("Cannot open device\n");
	printf(usage_msg);
	goto bail;
    }

    if (do_bit) {
        // TODO old alt state could be preserved
        if (gpio_ioctl_alt_func(fd, 0, 1 << bit) < 0) { ret = -1; goto bail; }

        if (do_val) {
	    if (val == 'z') {
		if (gpio_ioctl_set_bit_enable(fd, bit, 0) < 0) { ret = -1; goto bail; }
		printf("GPIO bit %d set to tristate\n", bit);
	    } else {
		if (gpio_ioctl_set_bit(fd, bit, val) < 0
		 || gpio_ioctl_set_bit_enable(fd, bit, 1) < 0) { ret = -1; goto bail; }
		printf("GPIO bit %d set to %d\n", bit, val);
	    }
            ret = 0;
        } else {
            int v;

            // TODO old out state could be preserved
            if (gpio_ioctl_tri(fd, 1 << bit) < 0) { ret = -1; goto bail; }

            v = gpio_ioctl_get_bit(fd, bit);
            if (v < 0) { ret = -1; goto bail; }
            ret = v ? 1 : 0;
            printf("GPIO bit %d is %d\n", bit, v);
        }
        goto bail; // preserve ret
    }

    if (do_altconf_get) {
	unsigned long long_val;
	gpio_ioctl_get_alt(fd, &long_val);
	printf("GPIO alternate functions 0x%08lx\n", long_val);	
    }
    
    if (do_altconf_set) {
	if (gpio_ioctl_alt_func(fd, altconf_set, altconf_clr) != 0) {
	    printf("GPIO alt conf failed\n");
	}
    }
    
    if (do_read) {
	unsigned long long_val;
	gpio_ioctl_in(fd, &long_val);
	printf("GPIO read 0x%08lx\n", long_val);
    }

    if (do_set) {
	printf("GPIO set 0x%08lx\n", set);
	if (gpio_ioctl_set(fd, set) != 0) {
	    printf("GPIO set failed\n");	
	}
    }

    if (do_clr) {
	printf("GPIO clr 0x%08lx\n", clr);
	if (gpio_ioctl_clr(fd, clr) != 0) {
	    printf("GPIO clr failed\n");
	}
    }

    if (do_tri) {
	printf("GPIO tri x%08lx\n", tri);
	if (gpio_ioctl_tri(fd, tri) != 0) {
	    printf("GPIO clr failed\n");
	}
    }   

    if (do_irqconf) {	
	printf("GPIO IRQconf, level=0x%08lx, edge=0x%08lx, pos=0x%08lx, neg=0x%08lx\n",
	       irqconf_level, irqconf_edge, irqconf_pos, irqconf_neg);

	if (gpio_ioctl_irqconf(fd, irqconf_level, irqconf_edge, irqconf_pos, irqconf_neg) != 0) {

	    printf("Error setting IRQ config\n");
	}
    }
    
    if (do_irq) {		
	gpio_ioctl_watch(fd, irq);
	    
	if (eric_pthread_create(&irq_thread, 0, 64 * 1024, irq_thread_func, NULL) != 0) {
	    printf("Could not create IRQ thread");
	    goto bail;
	}
    }
   
    if (errflag) {
	printf(usage_msg);
	goto bail;
    }

    if (do_irq) {
	pthread_join(irq_thread, NULL);
    }
    
    ret = 0;
    
 bail:
    if (fd != -1) close(fd);
    return ret;
}

static void*
irq_thread_func(void* arg UNUSED)
{
    struct pollfd fds[1];
    unsigned long val;
    int r;

    fds[0].fd = fd;
    fds[0].events = POLLIN;
    
    while (1) {
	if ((r = poll(fds, 1, 1000)) < 0) {
	    printf("Poll failed (%d)\n", r);
	    return NULL;
	} else if (r == 0) continue;
	if ((r = gpio_ioctl_in(fd, &val)) <0) {
	    printf("GPIO in failed (%d)\n", r);
	}	    
	printf("GPIO in = 0x%08x\n", (unsigned int)val);
    }
    return NULL;
}
