/***************************************************************************
*
*   fpga.c
*
*   program to write the ALTERA FPGA 
*
*   Author: Swen Anderson
*
*   Copyright (c) by Peppercon AG 2000, 2003
*
***************************************************************************/

/***************************************************************************
*
*   Includes
*
***************************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#include <lara.h>
#include <pp/base.h>

#include "fpga.h"


/***************************************************************************
*
*   global variables
*
***************************************************************************/

static int silent = 0;
int device = DEVICE_EP1C4;


static char *usage_msg =
"\n"
"   Functional Parameters (give at least one):\n"
"     -f <filename>            ..  program file into FPGA \n"
"     -h -?                    ..  show this help \n"
"   Optional Parameters:\n"
"     -s                       ..  silent mode \n"
"     -w <hex-number>          ..  write GPIO bits \n"
"        <nstatus:conf_done:nconfig:data:clk\n"
"\n";

/***************************************************************************
*
*   bitwise functions
*
***************************************************************************/

void set_dclk(unsigned char val) {
    pp_gpio_bit_set(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_DCLK, val);
}

void set_data0(unsigned char val) {
    pp_gpio_bit_set(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_DATA0, val);
}

void set_nconfig(unsigned char val) {
    pp_gpio_bit_set(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_NCONFIG, val);
}

int set_conf_done(unsigned char val) {
    pp_gpio_bit_set(PP_GPIP_FPGA_DEV, PP_GPIO_FPGA_CONF_DONE, val);
}

void set_nstatus(unsigned char val) {
    pp_gpio_bit_set(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_NSTATUS, val);
}

int get_conf_done(void) {
    return pp_gpio_bit_get(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_CONF_DONE);
}

int get_nstatus(void) {
    return pp_gpio_bit_get(PP_GPIO_FPGA_DEV, PP_GPIO_FPGA_NSTATUS);
}

/***************************************************************************
*
*   macros for bitwise work
*
***************************************************************************/

#define SET_DCLK  set_dclk(1)
#define CLR_DCLK  set_dclk(0)

#define SET_DATA0 set_data0(1)
#define CLR_DATA0 set_data0(0)

#define SET_NCONFIG set_nconfig(1)
#define CLR_NCONFIG set_nconfig(0)

#define SET_CONF_DONE set_conf_done(1)
#define CLR_CONF_DONE set_conf_done(0)

#define SET_NSTATUS set_nstatus(1)
#define CLR_NSTATUS set_nstatus(0)

#define ATM_PRINT(fmt, args...)  { if(!silent) printf(fmt, ##args); }

/***************************************************************************
*
*   generic functions
*
***************************************************************************/

/***************************************************************************
*
* FUNCTIONS
*	fpga_program(), 
*	programs a file into FPGA 
*
***************************************************************************/

int fpga_program(char * filename) {
    int fd_file;
	
    /* open input file */
    if ((fd_file = open(filename, O_RDONLY)) == -1) {
	printf("fpga.c: Error opening file %s\n", filename);
	return -1;
    }
	
    /* Processing and Programming */
    ProcessFileInput( fd_file );

    close(fd_file);
    return 0;
}


/********************************************************************************/
/* Name:			ProcessFileInput				*/
/*										*/
/* Parameters:		FILE* fd_file	- programming file pointer.		*/
/*										*/
/* Return Value: None.								*/
/*										*/
/* Descriptions: Get programming file size, parse through every single byte	*/
/*		 and dump to parallel port.					*/
/*										*/
/*		 Configuration Hardware is verified before configuration	*/
/*		 starts.							*/
/*										*/
/*		 For every [CHECK_EVERY_X_BYTE] bytes, NSTATUS pin is		*/
/*		 checked for error. When the file size reached, CONF_DONE	*/
/*		 pin is checked for configuration status. Then, another		*/
/*		 [INIT_CYCLE] clock cycles are dumped to initialize		*/
/*		 the device.							*/
/*										*/
/*		 Configuration process is restarted whenever error found.	*/
/*		 The maximum number of auto-reconfiguration is			*/
/*		 [RECONF_COUNT_MAX].						*/
/*										*/
/********************************************************************************/

void ProcessFileInput( int fd_file )
{
	int		program_done = 0;			/* programming process (configuration and initialization) done = 1 */
	long int	file_size = 0;				/* programming file size */
	int		configuration_count = RECONF_COUNT_MAX;	/* # reprogramming after error */
	int		one_byte = 0;				/* the byte to program */
	long int	i = 0;					/* standard counter variable */
	int		j,bit;					
	int		confdone_ok = 1;			/* CONF_DONE pin. '1' - error */
	int		nstatus_ok = 0;				/* NSTATUS pin. '0' - error */
	int		clock_x_cycle = CLOCK_X_CYCLE; /* Clock another 'x' cycles during initialization (optional) */
	unsigned int    max_size;
	struct stat     f_state;
	
	max_size = DEVICE_EP1C4_CONFSIZE;

	/* Get file size */

	if((fstat(fd_file, &f_state)) == -1) {
		printf("fpga.c: Error at fstat\n");
		return;
	}

	if(f_state.st_size > max_size) {
		printf("fpga.c: Error: file too big !!\n");
		return;
	}

	file_size = f_state.st_size;
	printf( "Info: Programming file size: %ld\n", file_size );
	
	/* Start configuration */
	while ( !program_done && (configuration_count>0) )
	{
		/* Reset file pointer and parallel port registers */
		lseek( fd_file, 0, SEEK_SET );

		/* init GPIOs */ 
		SET_CONF_DONE; /* tristate bidi GPIO */
		SET_NSTATUS;   /* tristate bidi GPIO */
		CLR_DCLK;
		CLR_DATA0;


		SET_NCONFIG;
		printf( "\n***** Start configuration process *****\nPlease wait...\n" );

		/* Drive a transition of 0 to 1 to NCONFIG to indicate start of configuration */
		CLR_NCONFIG;
		sleep(1);
		SET_NCONFIG;
		sleep(1);

		/* Loop through every single byte */
		for ( i = 0; i < file_size; i++ )
		{


	                read(fd_file, &one_byte, 1);

               		if(i<44);
               		else
               		{

				/* Program a byte */
				/* write from LSb to MSb */
				for ( j = 0; j < 8; j++ )
				{
					bit = one_byte >> j;
					bit = bit & 0x1;
					
					/* Dump to DATA0 and insert a positive edge pulse at the same time */
			
					CLR_DCLK;
					set_data0(bit);
					SET_DCLK;
					
				}
			}
		
			/* Check for error through NSTATUS for every 10KB programmed and the last byte */
			if ( !(i % CHECK_EVERY_X_BYTE) || (i == file_size - 1) )
			{
				nstatus_ok = get_nstatus();

				if ( !nstatus_ok )
				{
					printf("Error NSTATUS %i\n", nstatus_ok);
					PrintError( configuration_count-1 );
					
					program_done = 0;
					break;
				}
				else 
					program_done = 1;
			}
		}

		printf( "Info: Clock another %d cycles in device initialization process...\n", clock_x_cycle );
		for ( i = 0; i < CLOCK_X_CYCLE; i++ )
		{
			CLR_DCLK;
			SET_DCLK;
		}

		configuration_count--;

		if ( !program_done )
			continue;

		/* Configuration end */
		/* Check CONF_DONE that indicates end of configuration */
		confdone_ok = get_conf_done();

		if ( !confdone_ok )
		{
			printf( "Error: Configuration done but contains error... CONF_DONE is %s\n", (confdone_ok? "HIGH":"LOW") );
			program_done = 0;
			PrintError( configuration_count );			
			if ( configuration_count == 0 )
				break;
		}
	
		/* if contain error during configuration, restart configuration */
		if ( !program_done )
			continue;

		/* program_done = 1; */

		/* Start initialization */
		/* Clock another (INIT_CYCLE) cycles to initialize the device */
		for ( i = 0; i < INIT_CYCLE; i++ )
		{
			CLR_DCLK;
			SET_DCLK;
		}
		/* Initialization end */

		nstatus_ok = get_nstatus();
		confdone_ok = get_conf_done();

		if ( !nstatus_ok || !confdone_ok )
		{
			printf( "Error: Initialization finish but contains error: NSTATUS is %s and CONF_DONE is %s. Exiting...", (nstatus_ok?"HIGH":"LOW"), (confdone_ok?"LOW":"HIGH") );
			program_done = 0; 
			configuration_count = 0; /* No reconfiguration */
		}
	}

	/* add another 'x' clock cycles to make sure the device is initialized (certain cases) */
	if ( clock_x_cycle > 0 )
	{
		printf( "Info: Clock another %d cycles in device initialization process...\n", clock_x_cycle );
		for ( i = 0; i < CLOCK_X_CYCLE; i++ )
		{
			CLR_DCLK;
			SET_DCLK;
		}
	}

	if ( !program_done )
	{
		printf( "\nError: Configuration not successful! Error encountered...\n" );
		return;
	}

	printf( "\nInfo: Configuration successful!\n" );

}


/***************************************************************************
*
* FUNCTIONS
*	write_gpio_bits() / read_gpio_bits()
*	writes / reads the gpio bits
*
***************************************************************************/

int write_gpio_bits(int gpio) {
	unsigned short my_gpio = (unsigned short) gpio;
	ATM_PRINT("setting GPIO bits: 0x%02X\n", my_gpio);

	if (gpio & 0x01) SET_DCLK;     //GPIO4 
	else CLR_DCLK;
	
	if (gpio & 0x02) SET_DATA0;    //GPIO2
	else CLR_DATA0;

	if (gpio & 0x04) SET_NCONFIG;  //GPI20 (auch /IRQ3)
	else CLR_NCONFIG;

	if (gpio & 0x08) SET_CONF_DONE;//GPIO8
	else CLR_CONF_DONE;

	if (gpio & 0x10) SET_NSTATUS;  //GPIO23 (auch /IRQ6)
	else CLR_NSTATUS;

	return 0;
}

int read_gpio_bits(void) {

	int i;
	
	ATM_PRINT("CONF_DONE is %s\n", get_conf_done() ? "1":"0");
	ATM_PRINT("NSTATUS   is %s\n", get_nstatus() ? "1":"0");

	return 0;
}

/********************************************************************************/
/*	Name:			PrintError 													*/
/*																				*/
/*	Parameters:		int configuration_count										*/
/*					- # auto-reconfiguration left  								*/
/*																				*/
/*	Return Value:	None.														*/
/*																				*/
/*	Descriptions:	Print error message to standard error.						*/
/*																				*/
/********************************************************************************/
void PrintError( int configuration_count )
{
	if ( configuration_count == 0 )
		printf( "Error: Error in configuration #%d... \nError: Maximum number of reconfiguration reached. Exiting...\n", (RECONF_COUNT_MAX-configuration_count) );
	else
	{
		printf( "Error: Error in configuration #%d... configuration aborted.\n", (RECONF_COUNT_MAX-configuration_count) );
	}
}


/***************************************************************************
*
* main()
*
***************************************************************************/


int main(int argc, char ** argv) {
	/* local variables */
	int c;
	int errflag = 0;
	
	int flash_write = 0, eeprom_write = 0, del_chip = 0, flash_read = 0, eeprom_read = 0, read_gpio = 0, write_gpio = 0;
	int read_lock = 0, write_lock = 0, reset = 0;
	
	char * flash_write_file = NULL;
	int ret = 0;
	int gpio = 0;
	unsigned char hw_id;
	
	if (pp_base_init("fpga", LOG_NOT_SILENT) == -1) {
	    fprintf(stderr, "Initializing base-library failed.\n");
	    ret = 1;
	    goto fail;
	}

	/* parse command line */
	while ((c = getopt(argc, argv, "f:rw:h?")) != -1) {
		switch(c) {
			case 'f':
				flash_write_file = optarg;
				flash_write++;
				break;
			case 'h':
			case 'r':
				read_gpio++;
				break;
			case 'w':
				write_gpio++;
				gpio = strtol(optarg, NULL, 16);
				break;
			case '?':
				errflag++;
		}
	}
	
	/* if no action is wanted show usage message */
	if(!del_chip && !flash_write && !eeprom_write && !flash_read && !eeprom_read && !read_gpio &&
	   !write_gpio && !read_lock && !write_lock && !reset) {
		errflag++;
	}
	
	ATM_PRINT("\n%s %s\nCopyright (c) Peppercon AG 2002\n", __FILE__, __DATE__);
		
	/* get hardware revision id */
	if (ioctl(eric_fd, ERICIOCGETHARDWAREREVISION, &hw_id)) {
	    printf("fpga.c: Error getting hardware revision\n");
	    ret = 1;
	    goto fail;
	}
	
	ATM_PRINT("Found hardware ID %d\n", hw_id);
	
	if(errflag) {
	    ATM_PRINT(usage_msg);
	    ret = 1;
	    goto fail;
	}

	/* write flash ? */
	if(flash_write) {
		if((fpga_program(flash_write_file)) != 0) {
			ret++;
		}
	}

	/* write gpio_bits? */
	if(write_gpio) {
		write_gpio_bits(gpio);
	}

	/* write gpio_bits? */
	if(read_gpio) {
		read_gpio_bits();
	}

 fail:
	pp_base_cleanup();
	return ret;
}
