/**
 * adc_kira.c
 *
 * Kernel module for the ADC on KIRA100
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <linux/delay.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/rtc.h>
#include <linux/types.h>
#include <asm/io.h>
#include <asm/semaphore.h>
#include <asm/uaccess.h>

#include "common_debug.h"
#include "adc_kira.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define ADC_MAJOR		((uint8_t)251)
#define DRV_NAME		"adc_kira"
#define ADC_IO_BASE		CPE_ADC_BASE
#define ADC_IO_SIZE		0x28
#define ADC_READ_TIMEOUT	HZ
#define ADC_READ_WAIT_US	100

#define ADC_CHANNEL_COUNT	8

#define ADC_REG_CONTROL		0x00
#define ADC_REG_DATA_VALID	0x01
#define ADC_REG_CHANNEL_0	0x02
#define ADC_REG_CHANNEL_1	0x03
#define ADC_REG_CHANNEL_2	0x04
#define ADC_REG_CHANNEL_3	0x05
#define ADC_REG_CHANNEL_4	0x06
#define ADC_REG_CHANNEL_5	0x07
#define ADC_REG_CHANNEL_6	0x08
#define ADC_REG_CHANNEL_7	0x09

#define REG_CONTROL_CHANNEL_0_SEL	(0x01 << 0)
#define REG_CONTROL_CHANNEL_1_SEL       (0x01 << 1)
#define REG_CONTROL_CHANNEL_2_SEL	(0x01 << 2)
#define REG_CONTROL_CHANNEL_3_SEL	(0x01 << 3)
#define REG_CONTROL_CHANNEL_4_SEL	(0x01 << 4)
#define REG_CONTROL_CHANNEL_5_SEL	(0x01 << 5)
#define REG_CONTROL_CHANNEL_6_SEL	(0x01 << 6)
#define REG_CONTROL_CHANNEL_7_SEL	(0x01 << 7)
#define REG_CONTROL_ACTIVATE		(0x01 << 8)

#define REG_DATA_VALID_VALID		(0x01 << 0)

#define SUCCESS	0
#define ADC_PROC_ACCESS
#define PROC_BUF_SIZE		1024

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# define MOD_INC_USE_COUNT  
# define MOD_DEC_USE_COUNT
#endif

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct adc_dev_s {
    volatile uint32_t	*regs;
    struct semaphore    mtx;
    u_long		ref_cnt;
#ifdef ADC_PROC_ACCESS
    char		proc_buf[PROC_BUF_SIZE];
    int			proc_size;
#endif
} adc_dev_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

adc_dev_t dev;

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

int    init_module(void);
void   cleanup_module(void);

static int	adc_init(void);
static void	adc_cleanup(void);

static int	adc_open(struct inode * inode, struct file * file);
static int	adc_release(struct inode *inode, struct file *file);
static int	adc_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg);

static int	adc_io(adc_kira_io_t *io);

#ifdef ADC_PROC_ACCESS
int adc_read_proc(char* buf, char **start, off_t offset, int count,
		  int *eof, void *data);
static int gen_proc_buf(void);
#endif

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Michael Baumann <miba@peppercon.de>");
MODULE_DESCRIPTION("Linux kernel module for ADC on Kira");

/*
 * Initialize the module
 */
int init_module(void)
{
    return adc_init();
}

/*
 * Cleanup module
 */
void cleanup_module(void)
{
    adc_cleanup();
}

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations adc_ops = {
    owner:	THIS_MODULE,
    open:	adc_open,
    release:	adc_release,
    ioctl:	adc_ioctl
};

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

static int
adc_init(void)
{
    int ret = SUCCESS, r;

    memset(&dev, 0, sizeof(adc_dev_t));
    
    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(ADC_MAJOR, DRV_NAME, &adc_ops)) < 0) {
        D(D_ERROR, "failed to register device (err %d)\n", r);
        ret = -ENODEV;
        goto bail;
    }

    if (request_mem_region(ADC_IO_BASE, ADC_IO_SIZE, DRV_NAME) == NULL) {
	D(D_ERROR, "memory region @ 0x%08x (size %d) already in use\n",
	  ADC_IO_BASE, ADC_IO_SIZE);
	goto bail;
    }

    if ((dev.regs = ioremap_nocache(ADC_IO_BASE, ADC_IO_SIZE)) == NULL) {
	D(D_ERROR, "could not remap GPIO regs @ 0x%08x (size %d)\n",
	  ADC_IO_BASE, ADC_IO_SIZE);	
	goto bail;	
    }

    D(D_NOTICE, "Mapped ADC regs @0x%08x to 0x%p (size %d)\n",
      ADC_IO_BASE, dev.regs, ADC_IO_SIZE);
    
    init_MUTEX(&dev.mtx);

#ifdef ADC_PROC_ACCESS
    create_proc_read_entry(DRV_NAME, 0, NULL, adc_read_proc, NULL);
#endif
    
 bail:
    return ret;
}

static void
adc_cleanup(void)
{
    int r;

#ifdef ADC_PROC_ACCESS
    remove_proc_entry(DRV_NAME, NULL);
#endif

    if (dev.regs != NULL) {
	iounmap((void*)dev.regs);
	dev.regs = NULL;
    }

    release_mem_region(ADC_IO_BASE, ADC_IO_SIZE);
    
    if ((r = unregister_chrdev(ADC_MAJOR, DRV_NAME)) < 0) {
        D(D_ERROR, "failed to unregister device (err %d)\n", r);
    }
}

/* ------------------------------------------------------------------------- *
 * the driver operations 
 * ------------------------------------------------------------------------- */

static int
adc_open(struct inode * inode, struct file * file)
{
    int ret = SUCCESS;

    if (down_interruptible(&dev.mtx) < 0) return -EINTR;
	
    if (++dev.ref_cnt == 1) {
	/* nothing to do currently */
    }
    
    MOD_INC_USE_COUNT;

    up(&dev.mtx);
    return ret;
}

static int
adc_release(struct inode *inode, struct file *file)
{
    int ret = SUCCESS;

    if (down_interruptible(&dev.mtx) < 0) return -EINTR;

    if (--dev.ref_cnt == 0) {
	/* nothing to do currently */
    }
	
    MOD_DEC_USE_COUNT;

    up(&dev.mtx);
    return ret;
}

static int
adc_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    int ret = SUCCESS;

    if (down_interruptible(&dev.mtx) < 0) return -EINTR;
    
    switch (cmd) {
      case ADC_KIRA_IOC_GET_READING:
	  {
	      adc_kira_io_t io;

	      if (copy_from_user(&io, (char*) arg, sizeof(adc_kira_io_t)) != 0) {
		  ret = -EFAULT;
		  break;
	      }

	      ret = adc_io(&io);
	      
	      if (copy_to_user((char*) arg, &io, sizeof(adc_kira_io_t)) != 0) {
		  ret = -EFAULT;
		  break;
	      }
	  }
	  break;
      default:
	  ret = -EINVAL;
	  break;
    }

    up(&dev.mtx);
    return ret;
}

#ifdef ADC_PROC_ACCESS
int adc_read_proc(char* buf, char **start, off_t offset, int count,
		  int *eof, void *data)
{
    *start = NULL;
    *eof = 1;
    
    if (down_interruptible(&dev.mtx) < 0) {
	return 0;
    }
  
    if (offset == 0 && gen_proc_buf() != 0) {
	D(D_ERROR, "Could not generate proc buffer\n");
	*eof = 1;
	
	goto error_out;
    }
   
    memcpy(buf, dev.proc_buf, dev.proc_size);
    
    up(&dev.mtx);
    return dev.proc_size;

 error_out:
    return 0;
}
#endif

/* ------------------------------------------------------------------------- *
 * private/helper functions
 * ------------------------------------------------------------------------- */

static int
adc_io(adc_kira_io_t *io)
{
    int ret = -1;
    unsigned long timeout;

    if (io->channel >= ADC_CHANNEL_COUNT) {
	D(D_ERROR, "Channel nr. %d exceeds count (%d)\n",
	  io->channel, ADC_CHANNEL_COUNT);
	goto bail;
    }

    dev.regs[ADC_REG_CONTROL] = REG_CONTROL_ACTIVATE | (0x01 << io->channel);

    udelay(ADC_READ_WAIT_US);
    
    /* wait for valid data */
    timeout = jiffies + ADC_READ_TIMEOUT;
    while ((dev.regs[ADC_REG_DATA_VALID] & REG_DATA_VALID_VALID) == 0) {

	if (time_after(jiffies, timeout)) {
	    D(D_ERROR, "Timeout while waiting for valid data on channel %d\n",
	      io->channel);
	    goto bail;
	}	
    }
   
    io->reading = dev.regs[ADC_REG_CHANNEL_0 + io->channel];
    D(D_BLABLA, "Reading (Channel %d): %d\n", io->channel, io->reading);

    dev.regs[ADC_REG_CONTROL] = 0;
    ret = SUCCESS;

 bail:
    return ret;   
}

#ifdef ADC_PROC_ACCESS
static int
gen_proc_buf()
{
    int i, len = 0, count, ret = -1;
    char* buf = dev.proc_buf;
    
    for (i = 0; i < ADC_CHANNEL_COUNT; i++) {
	adc_kira_io_t io = { .channel = i, .reading = 0 };

	if (adc_io(&io) != 0) {
	    D(D_ERROR, "Error getting value for channel %d\n", i);
	    goto bail;
	}

	count = sprintf(buf, "CH%d=%d\n", io.channel, io.reading);

	len += count;
	buf += count;
    }

    dev.proc_size = (len > PAGE_SIZE) ? PAGE_SIZE : len;

    ret = 0;
    
 bail:

    return ret;
}
#endif
