/**
 * fml_ophir.c
 *
 * FML Advanced Pass Through sideband protocol of
 * Intel 82571 Gigabit Ethernet Controller (Ophir)
 *
 * (c) 2004 Peppercon AG, 2004/12/20, Ralf Guenther <rgue@peppecon.de>
 */

#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <asm/byteorder.h>

#include "fml_ophir.h"

#define SUCCESS 0

#if 1
#define dump_recv_pkt(c, p)
#else
static void dump_recv_pkt(int count, tco_recv_pkt_t *pkt)
{
    int i = 0;
    printk("PKT count=%d op_code=0x%02x data=", count, pkt->op_code);
    for (i = 0; i < count - 1; i++) printk(" %02x", pkt->data[i]);
    printk("\n");

}
#endif

int ophir_xmit_packet(tco_t *tco, uint8_t seq, uint8_t len, uint8_t *data)
{
    return tco->write_block(tco, seq | OPHIR_CMD_XMIT_PKT, len, data);
}

int ophir_write_config(tco_t *tco, uint16_t addr, uint32_t data)
{
    uint8_t b[] = {
        (addr >>  8) & 0xff,
        (addr >>  0) & 0xff,
        (data >> 24) & 0xff,
        (data >> 16) & 0xff,
        (data >>  8) & 0xff,
        (data >>  0) & 0xff
    };

    return tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_WRITE_CFG,
                            sizeof(b), b);
}

int ophir_recv_enable(tco_t *tco, uint8_t ctrl)
{
    return tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_RECV_EN,
                            1, &ctrl);
}

int ophir_recv_enable_ex(tco_t *tco,
                       uint8_t ctrl,
                       uint8_t mac_addr[6],
                       uint8_t ip_addr[4],
                       uint8_t bmc_smbus_addr,
                       uint8_t if_data,
                       uint8_t alert_data)
{
    uint8_t data[] = {
        ctrl,
        mac_addr[0], mac_addr[1], mac_addr[2],
        mac_addr[3], mac_addr[4], mac_addr[5],
        ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3],
        bmc_smbus_addr,
        if_data,
        alert_data
    };

    return tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_RECV_EN,
                            sizeof(data), data);
}

int ophir_update_filter(tco_t *tco, uint8_t number, uint8_t len,
                        uint8_t *param)
{
    uint8_t data[32] = { number };
    memcpy(data + 1, param, len);
    return tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_UPDATE_FILTER,
                            len + 1, data);
}

int ophir_request_status(tco_t *tco)
{
    uint8_t data[] = { 0 };
    return tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_REQ_STAT,
                            sizeof(data), data);
    /* a ophir_recv_status must follow! */
}


int ophir_recv_any(tco_t *tco, uint8_t *count, tco_recv_pkt_t *recv_pkt)
{
    int r;
    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_RECV_ANY,
                             count, (uint8_t*)recv_pkt)) < 0)
        return r;

    return SUCCESS;
}

int ophir_read_filter(tco_t *tco, uint8_t number, uint8_t *len, uint8_t *param)
{
    int r;
    uint8_t data[] = { number };
    tco_recv_pkt_t recv_pkt;
    uint8_t count;

    // send parameter number
    if ((r = tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_UPDATE_FILTER,
                              sizeof(data), data) < 0))
        return r;

    // receive parameter value
    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_FILTER,
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if (recv_pkt.op_code != (OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_FILTER)) return -EIO;
    if (count < 2) return -EIO;
    if (recv_pkt.data[0] != number) return -EIO;
    if (*len < count - 2) return -ENOBUFS;

    *len = count - 2; // without op code and number
    memcpy(param, recv_pkt.data + 1, *len);
    return SUCCESS;
}


int ophir_recv_packet(tco_t *tco, uint8_t *seq, uint8_t *len, uint8_t *data)
{
    /* last frame: data=status */

    int r;
    tco_recv_pkt_t recv_pkt;
    uint8_t count;
    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_RECV_PKT, 
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if ((recv_pkt.op_code & OPHIR_CMD_MASK) != OPHIR_OP_RECV_PKT) return -EIO;

    *seq = recv_pkt.op_code & OPHIR_SEQ_MASK;
    if (*len < count - 1) return -ENOBUFS;
    *len = count - 1;
    memcpy(data, recv_pkt.data, *len);
    return SUCCESS;
}

int ophir_read_config(tco_t *tco, uint16_t addr, uint32_t *data)
{
    int r;
    tco_recv_pkt_t recv_pkt;
    uint8_t count;
    uint8_t b[] = {
        (addr >>  8) & 0xff,
        (addr >>  0) & 0xff
    };

    if ((r = tco->write_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_WRITE_CFG,
                              sizeof(b), b)) < 0)
        return r;

    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_CFG,
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if (recv_pkt.op_code != (OPHIR_SEQ_SINGLE | OPHIR_OP_READ_CFG)) return -EIO;
    if (count != 5) return -EIO;

    *data = be32_to_cpu(*(uint32_t*)recv_pkt.data);
    return SUCCESS;
}

int ophir_read_recv_enable(tco_t *tco,
                         uint8_t *ctrl,
                         uint8_t *mac_addr,
                         uint8_t *ip_addr,
                         uint8_t *bmc_smb_addr,
                         uint8_t *if_data,
                         uint8_t *alert_data)
{
    int r;
    tco_recv_pkt_t recv_pkt;
    uint8_t count;
    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_RECV_EN,
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if (recv_pkt.op_code != (OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_RECV_EN))
        return -EIO;
    if (count != 15) return -EIO;

    *ctrl =             recv_pkt.data[0];
    memcpy(mac_addr,    recv_pkt.data + 1, 6);
    memcpy(ip_addr,     recv_pkt.data + 7, 4);
    *bmc_smb_addr =     recv_pkt.data[11];
    *if_data =          recv_pkt.data[12];
    *alert_data =       recv_pkt.data[13];
    return SUCCESS;
}

int ophir_read_status(tco_t *tco, uint16_t *status)
{
    int r;
    tco_recv_pkt_t recv_pkt;

    uint8_t count;
    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_READ_STAT,
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if (recv_pkt.op_code != (OPHIR_SEQ_SINGLE | OPHIR_OP_READ_STAT))
        return -EIO;
    if (count != 3) return -EIO;

    *status = be16_to_cpu(*(uint16_t*)recv_pkt.data);
    return SUCCESS;
}

int ophir_get_sys_mac_addr(tco_t *tco, uint8_t *addr)
{
    int r;
    tco_recv_pkt_t recv_pkt;
    uint8_t count;

    if ((r = tco->read_block(tco, OPHIR_SEQ_SINGLE | OPHIR_CMD_GET_MAC_ADDR,
                             &count, (uint8_t*)&recv_pkt)) < 0)
        return r;

    if (recv_pkt.op_code != (OPHIR_SEQ_SINGLE | OPHIR_OP_GET_MAC_ADDR))
        return -EIO;
    if (count != 7) return -EIO;

    memcpy(addr, recv_pkt.data, 6);
    return SUCCESS;
}
