#include <linux/threads.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/netdevice.h>
#include <asm/io.h>

#include "fmlcore.h"
#include "fml.h"
#include "fml_test.h"

#define NAME    "fml_test"
#define LOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

/* ------------------------------------------------------------------------- *
 * constants
 * ------------------------------------------------------------------------- */

#define FML_TEST_MAJOR          254

#define DEV_NAME                "fml_test"
#define FML_SLAVE_ADDR          ((uint8_t)0x55)

#define DELAY() udelay(50)
#define RETRIES 3

#define SUCCESS 0

/* KIRA100 IO addresses/bits */

// PCI dev and INTMUX not used by KIRA
#ifndef PP_BOARD_KIRA
#  define FML_USE_PCI
#  define FML_USE_INTMUX
#endif

#ifdef FML_USE_PCI
#  define KIRA100_PCI_VENDOR_ID 0x1743
#  define KIRA100_PCI_DEVICE_ID 0x1003
#  define KIRA100_PCI_BAR_NO    1
#  define FML_IO_BASE_OFFS      0x9b0000
#  define DEF_IRQ               27
#else
#  define FML_IO_BASE_OFFS      CPE_FML_BASE
#  define DEF_IRQ               IRQ_FML
#endif

#ifdef FML_USE_INTMUX
#  define INTMUX_IO_BASE_OFFS   0x9a0000
#  define INTMUX_IO_SIZE        (1 * sizeof(uint32_t)) /* 1 dword */
#  define FML_INT               ((uint16_t)(1 << 14))
#endif

uint32_t kira100_io_base = 0;
#define FML_IO_SIZE             0x2000 // == 8kbyte

/* ------------------------------------------------------------------------- *
 * structure definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    struct fmlcore_private fml;
    
    int initialized;
    int isr_registered;
    
    wait_queue_head_t data_wait;

    void *fml_addr;
#ifdef FML_USE_INTMUX
    void *intmux_addr;
#endif
} fml_test_priv_t;

/* ------------------------------------------------------------------------- *
 * device driver prototypes
 * ------------------------------------------------------------------------- */

static int  fml_test_mod_init(void);
static void fml_test_mod_cleanup(void);

static int fml_test_open(struct inode * inode, struct file * file);
static int fml_test_release(struct inode * inode, struct file * file);
static int fml_test_ioctl(struct inode * inode, struct file * file, uint cmd, ulong arg);

static void fml_test_data_available(fml_test_priv_t *fml_test);
static int fml_test_perform_test(fml_test_priv_t *fml_test);

static void fml_enable_client_int(fml_test_priv_t *fml_test, int en);
static int fml_write_block(fml_test_priv_t *fml_test, u8 command,
		           u8 byte_count, const u8 * data);
static int fml_read_block(fml_test_priv_t *fml_test, u8 *byte_count, u8 *data);

static irqreturn_t fml_isr(int irq, void *dev_id, struct pt_regs *regs);
static int recv_cb(struct fmlcore_private *fml);

/* ------------------------------------------------------------------------- *
 * local data
 * ------------------------------------------------------------------------- */

fml_test_priv_t fml_test;

static struct file_operations fml_test_mod_ops = {
    open:       fml_test_open,
    release:    fml_test_release,
    ioctl:      fml_test_ioctl,
};

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Thomas Rendelmann <Thomas.Rendelmann@raritan.com>");
MODULE_DESCRIPTION("KIRA100 FML (slave) test module");
MODULE_LICENSE("GPL");

module_init(fml_test_mod_init);
module_exit(fml_test_mod_cleanup);

/* ------------------------------------------------------------------------- *
 * initialization and cleanup
 * ------------------------------------------------------------------------- */

int fml_test_mod_init(void) {
    memset(&fml_test, 0, sizeof(fml_test));

    SET_MODULE_OWNER(&fml_test_mod_ops);
    
    int ret = SUCCESS, r;
    char dev_name[] = DEV_NAME;
    uint32_t fml_io_base;
#ifdef FML_USE_INTMUX
    uint32_t intmux_io_base;
#endif

    init_waitqueue_head(&fml_test.data_wait);

    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(FML_TEST_MAJOR, DEV_NAME, &fml_test_mod_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }

#ifdef FML_USE_PCI
    {
        struct pci_dev *pci;
        pci = pci_find_device(KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID, NULL);
        if (pci == NULL) {
            ERR("failed to find PCI device (%04x:%04x) containing FML core",
                KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID);
            ret = -ENODEV;
            goto fail;
        }
        kira100_io_base = pci_resource_start(pci, KIRA100_PCI_BAR_NO);
    }
#endif

    fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;
    if ((fml_test.fml_addr = ioremap_nocache(fml_io_base, FML_IO_SIZE)) == (unsigned long)NULL) {
        ERR("failed to map FML core IO space at %08x-%08x to memory",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }

#ifdef FML_USE_INTMUX
    intmux_io_base = kira100_io_base + INTMUX_IO_BASE_OFFS;
    if ((dev->intmux_addr = ioremap_nocache(intmux_io_base, INTMUX_IO_SIZE)) == NULL) {
        ERR("failed to map INTMUX IO space %08x-%08x to memory",
            intmux_io_base, intmux_io_base + INTMUX_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }
#endif

    /* fill fml struct */
    fml_test.fml.name      = dev_name;
    fml_test.fml.io_base   = (u32)fml_test.fml_addr;
    fml_test.fml.phys_base = fml_io_base;
    //fml_test.fml.fml_clk   = 8333; /* 8.333Mhz */
    fml_test.fml.fml_clk   = 6944; /* we need Tlow to be at least 72ns */
#ifdef PP_BOARD_KIRA
    fml_test.fml.sys_clk   = 100000; /* 100Mhz/AHB */
#else
    fml_test.fml.sys_clk   = 33000; /* 33Mhz/PCI */
#endif
    fml_test.fml.irq       = DEF_IRQ;
    fml_test.fml.slave     = 0;
    fml_test.fml.slave_req_callback  = recv_cb;

    // reserve memory
    if ((r = check_mem_region(fml_io_base, FML_IO_SIZE))) {
        ERR("IO space %08x-%08x already in use (err %d)",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1, r);
        ret = -EBUSY;
        goto fail;
    }
    request_mem_region(fml_io_base, FML_IO_SIZE, "fml");

    // init hardware
    if (fmlcore_init(&fml_test.fml) < 0) {
        ret = -ENODEV;
        goto fail;
    }
    fml_test.initialized = 1;

#ifdef FML_USE_INTMUX
    INFO("successfully loaded. (io: %08x, intmux: %08x, irq: %d)",
        fml_io_base, intmux_io_base, DEF_IRQ);
#else // !FML_USE_INTMUX
    INFO("successfully loaded. (io: %08x, irq: %d)",
        fml_io_base, DEF_IRQ);
#endif

    INFO("Successfully loaded FML test module.");
    return SUCCESS;

fail:
    fml_test_mod_cleanup();
    return ret;
}

void fml_test_mod_cleanup(void) {
    int r;

    // release hardware
    if (fml_test.initialized) {
        fmlcore_release(&fml_test.fml);
        fml_test.initialized = 0;
    }
    release_mem_region(kira100_io_base + FML_IO_BASE_OFFS, FML_IO_SIZE);

#ifdef FML_USE_INTMUX
    if (fml_test.intmux_addr) {
        iounmap(fml_test.intmux_addr);
        fml_test.intmux_addr = NULL;
    }
#endif
    if (fml_test.fml_addr) {
        iounmap(fml_test.fml_addr);
        fml_test.fml_addr = NULL;
    }

    if ((r = unregister_chrdev(FML_TEST_MAJOR, DEV_NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }

    INFO("Unloaded FML test module.");
}


/* ------------------------------------------------------------------------- *
 * file operations
 * ------------------------------------------------------------------------- */

/* initialize core on open call */
static int fml_test_open(struct inode * inode, struct file * file) {
    int ret = SUCCESS, r;

    DBG("Opening device.");

    // disable IRQ source
    fml_enable_client_int(&fml_test, 0);

    // request IRQ
    if ((r = request_irq(DEF_IRQ, fml_isr, SA_SHIRQ, DEV_NAME, &fml_test)) < 0) {
        ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
        ret = -EBUSY;
        goto bail;
    }
    fml_test.isr_registered = 1;
    
    DBG("Device successfully opened.");

 bail:
    return ret;
}

/* cleanup core on close call */
static int fml_test_release(struct inode * inode, struct file * file) {
    DBG("Closing device.");
    
    // release IRQ
    if (fml_test.isr_registered) {
        free_irq(DEF_IRQ, &fml_test);
        fml_test.isr_registered = 0;
    }
    
    DBG("Device successfully closed.");
    return SUCCESS;
}

static int fml_test_ioctl(struct inode * inode, struct file * file, uint cmd, ulong arg) {
    switch (cmd) {
        case FML_IOCTL_PERFORM_TEST:
            return fml_test_perform_test(&fml_test);
        default:
            ERR("Wrong command in ioctl: %x\n", cmd);
            return -EINVAL;
    }
}

/* ------------------------------------------------------------------------- *
 * this is the actual implementation of this test module
 * ------------------------------------------------------------------------- */

volatile static int data_available = 0;

static void fml_test_data_available(fml_test_priv_t *fml_test) {
    if (!data_available) {
        //DBG("reply data available.");
        data_available = 1;
        wake_up_interruptible(&fml_test->data_wait);
        fml_enable_client_int(fml_test, 0);
    }
}

static int fml_test_perform_test(fml_test_priv_t *fml_test) {
    int ret = SUCCESS;
    data_available = 0;

    // we need higher priority here...
    int old_policy = current->policy;
    int old_prio = current->rt_priority;
    struct sched_param sched_params;

    INFO("higher priorities....");
    memset(&sched_params, 0, sizeof(sched_params));
    sched_params.sched_priority = 50;
    sched_setscheduler(current, SCHED_RR, &sched_params);
    
    // turn off slave interrupts before sending
    fml_enable_client_int(fml_test, 0);
    
    char req[256] = FML_TEST_REQUEST_STRING;
    int len = strlen(req);
    DBG("Sending string %s", req);
    if ((ret = fml_write_block(fml_test, 0, len, req)) != 0) {
        ret = -EIO;
        goto bail;
    }
    
    // turn them on again
    fml_enable_client_int(fml_test, 1);
        
    // wait for data to arrive
    ret = wait_event_interruptible_timeout(fml_test->data_wait, data_available, HZ);
    if (ret < 0) {
        ret = -EINTR;
        goto bail;
    }
    if (!data_available) {
        WARN("Timeout waiting for slave reply.");
        ret = -ETIMEDOUT;
        goto bail;
    }
    
    // read the data and compare it
    u8 reply[256];
    u8 read_len;
    len = strlen(FML_TEST_REPLY_STRING);
    if ((ret = fml_read_block(fml_test, &read_len, reply)) != 0) {
        ret = -EIO;
        goto bail;
    }

    if (read_len != len) {
        WARN("Size of test reply doesn't match expected size!");
        DBG("%d vs %d", read_len, len);
        ret = -ENOMSG;
        goto bail;
    }
    
    reply[len] = '\0';
    
    if (strcmp(reply, FML_TEST_REPLY_STRING) != 0) {
        WARN("Test reply doesn't match expected reply!");
        DBG("%s", reply);
        DBG("%s", FML_TEST_REPLY_STRING);
        ret = -ENOMSG;
        goto bail;
    }
    
bail:
    sched_params.sched_priority = old_prio;
    sched_setscheduler(current, old_policy, &sched_params);

    return ret;
}

/* ------------------------------------------------------------------------- *
 * interrupt handling
 * ------------------------------------------------------------------------- */

static int recv_cb(struct fmlcore_private *fml) {
    fml_test_data_available(&fml_test);
    return 0;
}

static irqreturn_t fml_isr(int irq, void *dev_id, struct pt_regs *regs) {
//WARN("enter irq");
#ifdef FML_USE_INTMUX
    {
        uint32_t intmux = readl(fml_test.intmux_addr);
        if (!(intmux & FML_INT)) return IRQ_HANDLED;
    }
#endif

    fmlcore_irq_handler(irq, &fml_test.fml, regs);
//WARN("leave irq");
    return IRQ_HANDLED;
}

/* ------------------------------------------------------------------------- *
 * helper functions
 * ------------------------------------------------------------------------- */

/* enable/disable client interrupt */
static void fml_enable_client_int(fml_test_priv_t *fml_test, int en) {
    // workaround: FML core generates IRQ on low SINTEX during master read
    if (en) fml_test->fml.write_reg(MFMLIACK, &fml_test->fml, MIACK_ADDR);

    fml_test->fml.write_reg(
        (fml_test->fml.read_reg(&fml_test->fml, MGR_ADDR) & ~MFMLIEN) | (en ? MFMLIEN : 0),
        &fml_test->fml, MGR_ADDR);
}

/* block read/write */
static int fml_write_block(fml_test_priv_t *fml_test,
		           u8 command,
		           u8 byte_count,
		           const u8 * data) {
    int ret = 0, retry = RETRIES;
    do {
        ret = fmlcore_write_block(&fml_test->fml, command, FML_SLAVE_ADDR, 
                                  byte_count, data);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during FML master write transfer: %d", ret);
        //else WARN("Error during FML master write transfer: %d", ret); // TODO: happens often, investigate!
    } while (retry-- > 0);
    return ret;
}

static int fml_read_block(fml_test_priv_t *fml_test,
		          u8 *byte_count,
		          u8 *data) {
    int ret = 0, retry = RETRIES;
    do {
        ret = fmlcore_read_block_simple(&fml_test->fml, FML_SLAVE_ADDR, byte_count, data);
        DELAY();
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during FML master read transfer: %d", ret);
        else WARN("Error during FML master read transfer: %d", ret);
    } while (retry-- > 0);
    return ret;
}
