#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/sched.h>

#include "fmlcore.h"
#include "fml_test.h"

#define NAME                    "fml_test"
#define noLOG_DBG
#define LOG_COLORED
#include "log.h"

static volatile int fml_test_received = 0;
static wait_queue_head_t fml_test_wait;
static volatile int fml_test_thread_should_run;
static volatile int fml_test_thread_running;
static struct fmlcore_private *fml = NULL;
struct semaphore * fml_test_thread_notify;

static int fml_test_thread(void * arg);

int fml_test_slave_init(struct fmlcore_private *_fml) {
    int r;
    
    DBG("Initializing FML test slave part");
    
    fml = _fml;
    
    init_waitqueue_head(&fml_test_wait);
    fml_test_received = 0;
    
    DECLARE_MUTEX_LOCKED(mtx);
    fml_test_thread_notify = &mtx;
    fml_test_thread_should_run = 1;
    if ((r = kernel_thread(fml_test_thread, NULL, 0)) < 0) {
        ERR("failed to initialize FML test slave thread (err %d)", r);
        return -EBUSY;
    }
    down(&mtx);
    fml_test_thread_notify = NULL;

    return 0;
}

void fml_test_slave_cleanup(void) {
    DBG("Cleaning up FML test slave part");
    
    if (fml_test_thread_running) {
        DECLARE_MUTEX_LOCKED(mtx);
        fml_test_thread_notify = &mtx;
        
        fml_test_thread_should_run = 0;
        wake_up_interruptible(&fml_test_wait);
        down(&mtx);
        fml_test_thread_notify = NULL;
    }
}

static int send_test_reply(struct fmlcore_private *fml) {
    char ret_data[256];

    DBG("Sending reply, len is %d", strlen(FML_TEST_REPLY_STRING));
    snprintf(&ret_data[1], sizeof(ret_data) - 1, "%s", FML_TEST_REPLY_STRING);
    ret_data[0] = strlen(FML_TEST_REPLY_STRING);

    fmlcore_slave_write(fml, ret_data, strlen(FML_TEST_REPLY_STRING) + 1);
    fml_test_received = 0;
    return 0;
}

static int fml_test_thread(void * arg) {
    DBG("fml_test_thread started");
    fml_test_thread_running = 1;

    daemonize(NAME);

    if (fml_test_thread_notify != NULL) {
        up(fml_test_thread_notify);
    }
    
    while (fml_test_thread_should_run) {
        int ret = wait_event_interruptible(fml_test_wait, !fml_test_thread_should_run || fml_test_received);
        DBG("fml_test_thread - woken up");
        if (ret < 0) {
            if (signal_pending(current)) {
                flush_signals(current);
                continue;
            }
            WARN("Waiting for event failed!");
            break;
        }
        
        if (!fml_test_thread_should_run) {
            DBG("Have to break");
            break;
        }
        if (fml_test_received) {
            DBG("Have to send reply");
            send_test_reply(fml);
        }
        cond_resched();
    }
    DBG("fml_test_thread finished");

    fml_test_thread_running = 0;
    if (fml_test_thread_notify != NULL) {
        up(fml_test_thread_notify);
    }
    
    return 0;
}

int fml_test_slave(struct fmlcore_private *fml, char *data, size_t size) {
    DBG("received data: %d bytes\n", size);
    if (data != NULL &&
        (size - 2) == strlen(FML_TEST_REQUEST_STRING) &&
        strcmp(&data[2], FML_TEST_REQUEST_STRING) == 0) {
        
        DBG("is test data.\n");
        fml_test_received = 1;
        wake_up_interruptible(&fml_test_wait);
        return 0;
    } else {
        // no test stuff
        DBG("is NOT test data.\n");
        return 1;
    }
}
