/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Application to test the events of the FPGA Protocol Driver.
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <poll.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include "fpd_ioctl.h"

#include "cim.h"

static FPD_msg_t gmsg;


static int fpd_get_invalid_cmd( int fd, int link_if )
{
    FPD_rxinvcmd_t invalid_cmd;
    int rc;

    invalid_cmd.link_if = link_if;
    rc = ioctl(fd, FPD_GET_RX_INVALID_CMD, &invalid_cmd);
    if (rc == 0) {
        if( invalid_cmd.n_entries > 0 ) {
            int i;

            for( i = 0; i < invalid_cmd.n_entries; i++ ) {
                printf("\t\t\tCMD 0x%02x\n", invalid_cmd.inv_cmd[i]);
            }
        }
        else {
            printf("\t\t\tWEIRD!! No entries found.\n");
        }
    }
    else {
        if(errno) {
            printf("FPD_GET_RX_INVALID_CMD ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static void display_error( int fd, FPD_error_t *pinfo )
{
    switch( pinfo->type ) {
        case FPD_ERR_TYPE_PCI:
            printf("\t\tPCI Error: %08x\n", pinfo->error);
            if( pinfo->error ) {
                if( pinfo->error & PCI_ERR_PARITY ) {
                    printf("       Data Parity Detected\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_TARGET_ABORT ) {
                    printf("       Signaled Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_TARGET_ABORT ) {
                    printf("       Received Target Abort\n");
                }
                else if( pinfo->error & PCI_ERR_RCV_MASTER_ABORT ) {
                    printf("       Received Master Abort\n");
                }
                else if( pinfo->error & PCI_ERR_SIG_SYSTEM_ERROR ) {
                    printf("       Signaled System Error\n");
                }
                else if( pinfo->error & PCI_ERR_DETECTED_PARITY ) {
                    printf("       Detected Parity Error\n");
                }
            }
            else {
                printf("       NONE\n");
            }
            break;

        case FPD_ERR_TYPE_LINK_IF:
            if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
                printf("\t\tBackground Link IF Error: %08x\n", pinfo->error);
            }
            else {
                printf("\t\tLink IF %d Error: %08x\n",
                       pinfo->link_if, pinfo->error);
            }

            if( pinfo->error ) {
                if( pinfo->error & FPD_ERR_RX_INV_CMD ) {
                    printf("\t\tFPD_ERR_RX_INV_CMD\n");

                    /* retrieve the invalid command */
                    fpd_get_invalid_cmd(fd, pinfo->link_if);
                }
                if( pinfo->error & FPD_ERR_UPDATE_INV_PKT ) {
                    printf("\t\tFPD_ERR_UPDATE_INV_PKT\n");
                }
                if( pinfo->error & FPD_ERR_RX_PRI_RD_EMPTY_BUF ) {
                    printf("\t\tFPD_ERR_RX_PRI_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_SYNC ) {
                    printf("\t\tFPD_ERR_TX_PRI_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_NO_EOP ) {
                    printf("\t\tFPD_ERR_TX_PRI_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_WR_FULL_BUF ) {
                    printf("\t\tFPD_ERR_TX_PRI_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_RX_CIM_RD_EMPTY_BUF ) {
                    printf("\t\tFPD_ERR_RX_CIM_RD_EMPTY_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_SYNC ) {
                    printf("\t\tFPD_ERR_TX_CIM_SYNC\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_NO_EOP ) {
                    printf("\t\tFPD_ERR_TX_CIM_NO_EOP\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_WR_FULL_BUF ) {
                    printf("\t\tFPD_ERR_TX_CIM_WR_FULL_BUF\n");
                }
                if( pinfo->error & FPD_ERR_TX_PRI_TIMEOUT ) {
                    printf("\t\tFPD_ERR_TX_PRI_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_CIM_TIMEOUT ) {
                    printf("\t\tFPD_ERR_TX_CIM_TIMEOUT\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_0 ) {
                    printf("\t\tFPD_ERR_TX_HOST_TIMEOUT_0\n");
                }
                if( pinfo->error & FPD_ERR_TX_HOST_TIMEOUT_1 ) {
                    printf("\t\tFPD_ERR_TX_HOST_TIMEOUT_1\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_0 ) {
                    printf("\t\tFPD_ERR_RX_DMA_NO_BUF_0\n");
                }
                if( pinfo->error & FPD_ERR_RX_DMA_NO_BUF_1 ) {
                    printf("\t\tFPD_ERR_RX_DMA_NO_BUF_1\n");
                }
            }
            else {
                printf("\t\tNONE\n");
            }
            break;

        default:
            break;
    }

    return;
}

static int fpd_get_link_errors( int fd, int link_if, unsigned int *perror )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_LINK_IF;
    gmsg.error.link_if = link_if;
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
        if( perror ) {
            *perror = gmsg.error.error;
        }
    }
    else {
        if(errno) {
            printf("FPD_GET_ERROR ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static int fpd_get_pci_errors( int fd )
{
    int rc;

    gmsg.error.type = FPD_ERR_TYPE_PCI;
    gmsg.error.link_if = 0; /* don't care */
    rc = ioctl(fd, FPD_GET_ERROR, &gmsg);
    if (rc == 0) {
        /* display info */
        display_error(fd, &gmsg.error);
    }
    else {
        if(errno) {
            printf("FPD_GET_ERROR ioctl failed : %s\n", strerror(errno));
        }
    }

    return 0;
}

static void display_line_status( FPD_linestat_t *pinfo )
{
    static FPD_linestat_t current;
    static int first_time = 1;
    int i, id, line;
    unsigned int status;
    unsigned char linestat;

    if( first_time ) {
        first_time = 0;
        i = 0;
        line = 0;
        do {
            status = current.line_status[i] = pinfo->line_status[i];
            for( id = 0; id < 16; id++ ) {
                if( (line + id) >= max_line_cnt ) {
                    return;
                }

                printf("\t\tLine %d: ", line + id);
                linestat = status & 0x3;
                switch( linestat ) {
                    case 0:
                        printf("-\n");
                        break;
                    case 1:
                        printf("Paragon CIM\n");
                        break;
                    case 2:
                        printf("KX2.0 CIM\n");
                        break;
                    default:
                        printf("Unknown!\n");
                        break;
                }
                status >>= 2;
            }
            i++;
            line += 16;
        } while( line < max_line_cnt );
    }
    else {
        i = 0;
        line = 0;
        do {
            if( current.line_status[i] != pinfo->line_status[i] ) {
                status = current.line_status[i] = pinfo->line_status[i];
                for( id = 0; id < 16; id++ ) {
                    if( (line + id) >= max_line_cnt ) {
                        return;
                    }

                    printf("\t\tLine %d: ", line + id);
                    linestat = status & 0x3;
                    switch( linestat ) {
                        case 0:
                            printf("-");
                            break;
                        case 1:
                            printf("Paragon CIM");
                            break;
                        case 2:
                            printf("KX2.0 CIM");
                            break;
                        default:
                            printf("Unknown!");
                            break;
                    }

                    if( line < 32 ) {
                        if( pinfo->status_change[0] & (1 << (line+id)) ) {
                            printf(" (Status changed)\n");
                        }
                        else {
                            printf("\n");
                        }
                    }
                    else {
                        if( pinfo->status_change[1] & (1 << ((line - 32) + id)) ) {
                            printf(" (Status changed)\n");
                        }
                        else {
                            printf("\n");
                        }
                    }

                    status >>= 2;
                }
            }
            else {
                printf("\t\tLine Status %d - %d did not change\n", line, line+15);
            }
            i++;
            line += 16;
        } while( line < max_line_cnt );

    }

    return;
}

static int fpd_get_line_status( int fd )
{
    int rc;

    rc = ioctl(fd, FPD_GET_LINE_STATUS, &gmsg);
    if (rc == 0) {
        /* display info */
        display_line_status(&gmsg.linestat);
    }
    else {
        if(errno) {
            printf("FPD_GET_LINE_STATUS ioctl failed : %s\n", strerror(errno));
        }
    }


    return 0;
}

void handle_events( int fd, FPD_event_t *pevt, rx_notify_t *pnotify )
{
    int link;
    int id;

    pnotify->rx_cim = 0;
    pnotify->rx_priority = 0;
    pnotify->rx_host[0] = 0;
    pnotify->rx_host[1] = 0;

    /* Link Interfaces */
    for( link = 0; link < max_link_if; link++ ) {
        if( !(pevt->event_mask & (FPD_EVENT_MASK_LINK0 << link)) )
            continue;

        /* Host buffer */
        for( id = 0; id < FPD_HOST_CNT; id++ ) {
            if( pevt->link_if[link].host[id].has_data ) {
                if( pnotify->link_if == link ) {
                    pnotify->rx_host[id] = 1;
                }
                else if( debug_on ) {
                    printf("\tLink IF %d require Host Buffer %d processing\n",
                           link, id);
                }
            }
        }

        /* Priority buffer */
        if( pevt->link_if[link].rx_priority ) {
            if( pnotify->link_if == link ) {
                pnotify->rx_priority = 1;
            }
            else if( debug_on ) {
                printf("\tLink IF %d receive Priority Buffer data\n", link);
            }
        }

        /* CIM buffer */
        if( pevt->link_if[link].rx_cim ) {
            if( pnotify->link_if == link ) {
                pnotify->rx_cim = 1;
            }
            else if( debug_on ) {
                printf("\tLink IF %d receive CIM Buffer data\n", link);
            }
        }

        /* error detected */
        if( pevt->link_if[link].error ) {
            printf("\tLink IF %d receive errors\n", link);
            if( pnotify->link_if == link ) {
                fpd_get_link_errors( fd, link, &pnotify->error );
            }
            else {
                fpd_get_link_errors( fd, link, NULL );
            }
        }

        /* echo response detected */
        if( pevt->link_if[link].echo_rsp ) {
            if( pnotify->link_if == link ) {
                pnotify->echo_rsp = 1;
            }
            else if( debug_on ) {
                printf("\tLink IF %d receive echo response\n", link);
            }
        }
    }

    /* Background Link Interface events */
    if( pevt->event_mask & (FPD_EVENT_MASK_BGND_LINK) ) {
        if( pevt->bgnd_link_if.rx_cim ) {
            if( pnotify->link_if == FPD_LINK_IF_ID_BGND ) {
                pnotify->rx_cim = 1;
            }
            else if( debug_on ) {
                printf("\tBacground Link IF receive CIM Buffer data\n");
            }
        }

        if( pevt->bgnd_link_if.error ) {
            printf("\tBackground Link IF receive errors\n");
            if( pnotify->link_if == FPD_LINK_IF_ID_BGND ) {
                fpd_get_link_errors( fd, FPD_LINK_IF_ID_BGND, &pnotify->error );
            }
            else {
                fpd_get_link_errors( fd, FPD_LINK_IF_ID_BGND, NULL );
            }
        }

        if( pevt->bgnd_link_if.echo_rsp ) {
            if( pnotify->link_if == FPD_LINK_IF_ID_BGND ) {
                pnotify->echo_rsp = 1;
            }
            else if( debug_on ) {
                printf("\tBackground Link IF receive echo response\n");
            }
        }
    }

    if( pevt->pci_error ) {
        printf("\tDetected PCI error\n");
        fpd_get_pci_errors( fd );
    }

    if( pevt->cim_detect ) {
        printf("\tDetected changes to CIM topology\n");
        fpd_get_line_status( fd );
    }

    return;
}
