/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Diagnostic Application for the Paragon CIM support of the FPGA Protocol
 *  Device.
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "cim.h"
#include "cim_menu.h"
#include "cim_test.h"
#include "vm_lpbk.h"

#define FPD_CIM_REV        "$Revision: 1.2 $"

static int fpd_verbose = 0;


static void log( char *fmt, ... )
{
    va_list ap;
    if( fpd_verbose ) {
        printf( "[fpd-cim] " );
        va_start(ap,fmt);
        vprintf( fmt, ap );
        va_end(ap);
    }	
}

void msg( char *fmt, ... )
{
   va_list ap;
   printf( "[fpd-cim] " );
   va_start(ap,fmt);
   vprintf( fmt, ap );
   va_end(ap);
}

void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-cim] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}

static void usage( void )
{
    printf( "Use (" FPD_CIM_REV "):\n"
            "fpd-cim [options] [devname]\n\n"
            "    -l id      selected Link IF\n"
            "    -p port    selected target port\n"
            "    -t         test Paragon ports\n"
            "    -r         execute VM loopback test (receive)\n"
            "    -x         execute VM loopback test (transmit)\n"
            "    -h         help\n"
            "    -v         verbose\n"
            " [devname] is /dev/fpd\n\n");
}

int main( int argc, char * const argv[] )
{
    int ch;
    int fd;
    int link = 0;
    int port = -1;
    char paragon_test = 0;
    char vm_lpbk_rx_test = 0;
    char vm_lpbk_tx_test = 0;
    char dev[128] = "/dev/fpd";
    int option = 0;
    int rc = 0;

    while( (ch = getopt(argc, argv, "tl:p:rxhv")) != EOF ) {
        switch( ch ) {
            case 't':
                paragon_test = 1;
                break;

            case 'r':
                vm_lpbk_rx_test = 1;
                break;

            case 'x':
                vm_lpbk_tx_test = 1;
                break;

            case 'l':
                link = atoi(optarg);
                break;

            case 'p':
                port = atoi(optarg);
                break;

            case 'v':
                fpd_verbose = 1;
                break;

            case 'h':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
        log( "using %s as device node ...\n", dev );
    }

    if(( paragon_test && (vm_lpbk_rx_test || vm_lpbk_tx_test) ) ||
       ( vm_lpbk_rx_test && vm_lpbk_tx_test )) {
        err("Do not use -t, -x, and -r together\n");
        return -1;
    }

    if( (fd = open(dev, O_RDWR)) < 0 ) {
        err( "Cannot open %s : %s\n", dev, strerror(errno) );
        return 1;
    }

    /* initialization */
    if( cim_init(fd) < 0 ) {
        printf("Failed to initialize Diagnostics!\n");
        return -1;
    }

    if( paragon_test ) {
        cim_test_paragon_port(fd);
    }
    else if( vm_lpbk_tx_test ) {
        if( port < 0 ) {
            printf("Please specify a target port (-p)\n");
            return -2;
        }
#if 0
        fprintf(stderr, "\nChoose One pass (1) or forever loop (0): ");
        scanf("%d", &option);
#endif
        cim_test_vm_port_loopback(fd, link, port, FPD_CIM_BUFFER, SEND, option);
    }
    else if( vm_lpbk_rx_test ) {
        if( port < 0 ) {
            printf("Please specify a target port (-p)\n");
            return -2;
        }
#if 0
        fprintf(stderr, "\nChoose One pass (1) or forever loop (0): ");
        scanf("%d", &option);
#endif
        cim_test_vm_port_loopback(fd, link, port, FPD_CIM_BUFFER, RECEIVE, option);
    }
    else {
        cim_menu(fd, link);
    }

    close(fd);
	
    return rc;
}
